package ru.yandex.direct.grid.processing.service.smartfilter;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;

import com.google.common.collect.Lists;
import one.util.streamex.EntryStream;

import ru.yandex.direct.core.entity.feed.model.BusinessType;
import ru.yandex.direct.core.entity.feed.model.FeedType;
import ru.yandex.direct.core.entity.feed.model.Source;
import ru.yandex.direct.core.entity.performancefilter.model.Operator;
import ru.yandex.direct.core.entity.performancefilter.schema.FilterSchema;
import ru.yandex.direct.core.entity.performancefilter.schema.FilterSchemaFieldValidator;
import ru.yandex.direct.core.entity.performancefilter.schema.model.EnumFieldParams;
import ru.yandex.direct.core.entity.performancefilter.schema.model.NumberFieldParams;
import ru.yandex.direct.core.entity.performancefilter.schema.model.StringFieldParams;
import ru.yandex.direct.core.entity.performancefilter.schema.parser.AbstractPerformanceConditionValueParser;
import ru.yandex.direct.core.entity.performancefilter.schema.parser.ObjectListParser;
import ru.yandex.direct.grid.model.feed.GdBusinessType;
import ru.yandex.direct.grid.model.feed.GdFeedType;
import ru.yandex.direct.grid.model.feed.GdSource;
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFilterEnumField;
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFilterField;
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFilterNumberField;
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFilterOperator;
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFilterSchema;
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFilterStringField;

import static ru.yandex.direct.grid.processing.service.smartfilter.SmartFilterConverter.toGdConditionOperator;

public class SmartFilterSchemaConverter {

    static GdSmartFilterSchema getFilterSchema(FilterSchema filterSchema) {
        return new GdSmartFilterSchema()
                .withBusinessType(toGdBusinessType(filterSchema.businessType()))
                .withFeedType(toGdFeedType(filterSchema.feedType()))
                .withIsSiteSource(filterSchema.isSiteSource())
                .withFields(convertFields(filterSchema));
    }

    private static List<GdSmartFilterField> convertFields(FilterSchema filterSchema) {
        List<GdSmartFilterField> gdSmartFilterFieldList = new ArrayList<>();
        Map<String, FilterSchemaFieldValidator> filterFieldValidatorsMap =
                filterSchema.getFilterFieldValidatorsMap();

        EntryStream.of(filterFieldValidatorsMap)
                .forKeyValue((name, validator) ->
                        gdSmartFilterFieldList.add(convertToGdSmartFilterField(filterSchema.translateFieldNameForGd(name), validator)));

        return gdSmartFilterFieldList;
    }

    private static GdSmartFilterField convertToGdSmartFilterField(String name, FilterSchemaFieldValidator validator) {
        if (validator.getFieldParams() instanceof NumberFieldParams) {
            return getNumberField(name, validator);
        } else if (validator.getFieldParams() instanceof StringFieldParams) {
            return getStringField(name, validator);
        } else if (validator.getFieldParams() instanceof EnumFieldParams) {
            return getEnumField(name, validator);
        } else {
            throw new IllegalArgumentException("Unknown fieldParams: " + validator.getFieldParams());
        }
    }

    private static GdSmartFilterNumberField getNumberField(String name, FilterSchemaFieldValidator validator) {
        return new GdSmartFilterNumberField()
                .withName(name)
                .withOperators(convertToGdSmartFilterOperatorList(validator))
                .withMin(((NumberFieldParams) validator.getFieldParams()).getMinValue())
                .withMax(((NumberFieldParams) validator.getFieldParams()).getMaxValue())
                .withPrecision(((NumberFieldParams) validator.getFieldParams()).getPrecision());
    }

    private static GdSmartFilterStringField getStringField(String name, FilterSchemaFieldValidator validator) {
        return new GdSmartFilterStringField()
                .withName(name)
                .withOperators(convertToGdSmartFilterOperatorList(validator))
                .withMinLength(((StringFieldParams) validator.getFieldParams()).getMinStringLength())
                .withMaxLength(((StringFieldParams) validator.getFieldParams()).getMaxStringLength());
    }

    private static GdSmartFilterEnumField getEnumField(String name, FilterSchemaFieldValidator validator) {
        return new GdSmartFilterEnumField()
                .withName(name)
                .withOperators(convertToGdSmartFilterOperatorList(validator))
                .withValues(Lists.newArrayList(((EnumFieldParams) validator.getFieldParams()).getEnumValues()));
    }

    private static List<GdSmartFilterOperator> convertToGdSmartFilterOperatorList(
            FilterSchemaFieldValidator validator) {
        List<GdSmartFilterOperator> gdOperatorList = new ArrayList<>();
        for (Operator operator : validator.getOperators()) {
            GdSmartFilterOperator gdOperator = new GdSmartFilterOperator()
                    .withType(toGdConditionOperator(operator))
                    .withMaxItems(getOperatorMaxItem(validator.getParser(operator)));
            gdOperatorList.add(gdOperator);
        }
        return gdOperatorList;
    }

    private static Integer getOperatorMaxItem(AbstractPerformanceConditionValueParser<?> parser) {
        if (parser instanceof ObjectListParser) {
            return ((ObjectListParser) parser).getMaxItemCount();
        } else {
            return 1;
        }
    }

    private static GdFeedType toGdFeedType(@Nullable FeedType feedType) {
        if (feedType == null) {
            return null;
        }
        return GdFeedType.fromTypedValue(feedType.getTypedValue());
    }

    private static GdBusinessType toGdBusinessType(BusinessType businessType) {
        return GdBusinessType.valueOf(businessType.name());
    }

    private static GdSource toGdSource(@Nullable Source source) {
        if (source == null) {
            return null;
        }
        return GdSource.valueOf(source.name());
    }

    static FeedType toFeedType(@Nullable GdFeedType gdFeedType) {
        if (gdFeedType == null) {
            return null;
        }
        return FeedType.fromTypedValue(gdFeedType.getTypedValue());
    }

    static BusinessType toBusinessType(GdBusinessType gdBusinessType) {
        return BusinessType.valueOf(gdBusinessType.name());
    }

    static Source toSource(@Nullable GdSource gdSource) {
        if (gdSource == null) {
            return null;
        }
        return Source.valueOf(gdSource.name());
    }
}
