package ru.yandex.direct.grid.processing.service.smartfilter;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import org.apache.commons.collections4.CollectionUtils;

import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.grid.model.GdEntityStats;
import ru.yandex.direct.grid.model.GdGoalStats;
import ru.yandex.direct.grid.model.Order;
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFilter;
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFilterOrderBy;
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFilterOrderByField;

import static java.math.BigDecimal.ZERO;
import static org.hibernate.validator.internal.util.CollectionHelper.asSet;

// Сделано аналогично CampaignServiceUtils и OrderingProcessor. Рефакторинг может быть сделан в рамках DIRECT-92976
class SmartFilterUtils {
    private static final Map<GdSmartFilterOrderByField, Function<GdSmartFilter, ? extends Comparable>>
            FIELD_COMPARE_FUNC = ImmutableMap.<GdSmartFilterOrderByField, Function<GdSmartFilter, ? extends Comparable>>builder()
            .put(GdSmartFilterOrderByField.ID, GdSmartFilter::getId)
            .put(GdSmartFilterOrderByField.GROUP_ID, GdSmartFilter::getAdGroupId)
            .put(GdSmartFilterOrderByField.CAMPAIGN_ID, GdSmartFilter::getCampaignId)
            .put(GdSmartFilterOrderByField.FILTER_NAME, GdSmartFilter::getName)
            .put(GdSmartFilterOrderByField.PRICE_CPC, GdSmartFilter::getPriceCpc)
            .put(GdSmartFilterOrderByField.PRICE_CPA, GdSmartFilter::getPriceCpa)
            .put(GdSmartFilterOrderByField.AUTOBUDGET_PRIORITY, GdSmartFilter::getAutobudgetPriority)
            .put(GdSmartFilterOrderByField.IS_SUSPENDED, GdSmartFilter::getIsSuspended)
            .put(GdSmartFilterOrderByField.TARGET_FUNNEL, GdSmartFilter::getTargetFunnel)
            .put(GdSmartFilterOrderByField.STAT_COST, stat(GdEntityStats::getCost))
            .put(GdSmartFilterOrderByField.STAT_COST_WITH_TAX, stat(GdEntityStats::getCostWithTax))
            .put(GdSmartFilterOrderByField.STAT_SHOWS, stat(GdEntityStats::getShows))
            .put(GdSmartFilterOrderByField.STAT_CLICKS, stat(GdEntityStats::getClicks))
            .put(GdSmartFilterOrderByField.STAT_CTR, stat(GdEntityStats::getCtr))
            .put(GdSmartFilterOrderByField.STAT_REVENUE, stat(GdEntityStats::getRevenue))
            .put(GdSmartFilterOrderByField.STAT_GOALS, stat(GdEntityStats::getGoals))
            .put(GdSmartFilterOrderByField.STAT_CPM_PRICE, stat(GdEntityStats::getCpmPrice))
            .put(GdSmartFilterOrderByField.STAT_PROFITABILITY, stat(GdEntityStats::getProfitability))
            .put(GdSmartFilterOrderByField.STAT_AVG_DEPTH, stat(GdEntityStats::getAvgDepth))
            .put(GdSmartFilterOrderByField.STAT_AVG_GOAL_COST, stat(GdEntityStats::getAvgGoalCost))
            .put(GdSmartFilterOrderByField.STAT_AVG_CLICK_COST, stat(GdEntityStats::getAvgClickCost))
            .put(GdSmartFilterOrderByField.STAT_AVG_SHOW_POSITION, stat(GdEntityStats::getAvgShowPosition))
            .put(GdSmartFilterOrderByField.STAT_AVG_CLICK_POSITION, stat(GdEntityStats::getAvgClickPosition))
            .put(GdSmartFilterOrderByField.STAT_BOUNCE_RATE, stat(GdEntityStats::getBounceRate))
            .put(GdSmartFilterOrderByField.STAT_CONVERSION_RATE, stat(GdEntityStats::getConversionRate))
            .build();

    private static final Map<GdSmartFilterOrderByField, Function<GdGoalStats, ? extends Comparable>> GOAL_COMPARE_FUNC =
            ImmutableMap.<GdSmartFilterOrderByField, Function<GdGoalStats, ? extends Comparable>>builder()
                    .put(GdSmartFilterOrderByField.STAT_GOALS, GdGoalStats::getGoals)
                    .put(GdSmartFilterOrderByField.STAT_CONVERSION_RATE, GdGoalStats::getConversionRate)
                    .put(GdSmartFilterOrderByField.STAT_COST_PER_ACTION, GdGoalStats::getCostPerAction)
                    .build();

    private static final Map<GdSmartFilterOrderByField, Set<StrategyName>> STRATEGY_SET_BY_FIELD = ImmutableMap
            .<GdSmartFilterOrderByField, Set<StrategyName>>builder()
            .put(GdSmartFilterOrderByField.PRICE_CPC, ImmutableSet.of(
                    StrategyName.AUTOBUDGET_AVG_CPC_PER_CAMP,
                    StrategyName.AUTOBUDGET_AVG_CPC_PER_FILTER))
            .put(GdSmartFilterOrderByField.PRICE_CPA, ImmutableSet.of(
                    StrategyName.AUTOBUDGET_AVG_CPA_PER_CAMP,
                    StrategyName.AUTOBUDGET_AVG_CPA_PER_FILTER))
            .put(GdSmartFilterOrderByField.AUTOBUDGET_PRIORITY, ImmutableSet.of(StrategyName.AUTOBUDGET_ROI))
            .build();

    private static final GdGoalStats EMPTY_GOAL_STAT =
            new GdGoalStats().withGoals(0L).withConversionRate(ZERO).withCostPerAction(ZERO);
    private static final Comparator<Comparable> NATURAL_COMPARATOR = Comparator.nullsLast(Comparator.<Comparable>naturalOrder());

    static Comparator<GdSmartFilter> getComparator(List<GdSmartFilterOrderBy> orderBy, Map<Long, StrategyName> strategyBySmartFilterIds) {

        if (CollectionUtils.isEmpty(orderBy)) {
            return (a, b) -> 0;
        }

        Comparator<GdSmartFilter> comparator = null;
        for (GdSmartFilterOrderBy item : orderBy) {

            Function<GdSmartFilter, ? extends Comparable> func;
            if (item.getParams() == null) {
                func = FIELD_COMPARE_FUNC.get(item.getField());

                if (func == null) {
                    throw new IllegalArgumentException(String.format("No comparator for %s", item.getField()));
                }
            } else {
                Function<GdGoalStats, ? extends Comparable> goalFunc = GOAL_COMPARE_FUNC.get(item.getField());
                if (goalFunc == null) {
                    throw new IllegalArgumentException(String.format("No goal comparator for %s", item.getField()));
                }

                func = (Function<GdSmartFilter, Comparable>) smartFilter -> {

                    for (GdGoalStats goalStat : smartFilter.getGoalStats()) {
                        if (Objects.equals(goalStat.getGoalId(), item.getParams().getGoalId())) {
                            return goalFunc.apply(goalStat);
                        }
                    }

                    //Если цель не найдена, то все значения для сортировки считаем нулем, а не null
                    return goalFunc.apply(EMPTY_GOAL_STAT);
                };
            }

            Comparator<GdSmartFilter> currentComparator
                    = (a, b) -> NATURAL_COMPARATOR.compare(func.apply(a), func.apply(b));

            if (STRATEGY_SET_BY_FIELD.containsKey(item.getField())) {
                currentComparator = new StrategyComparator(strategyBySmartFilterIds, item.getField())
                        .thenComparing(currentComparator);
            }

            currentComparator = item.getOrder() == Order.ASC ? currentComparator : currentComparator.reversed();

            if (comparator == null) {
                comparator = currentComparator;
            } else {
                comparator = comparator.thenComparing(currentComparator);
            }
        }
        if (comparator == null) {
            throw new IllegalArgumentException("Empty list of comparators");
        }

        return comparator;
    }

    private static <T> Function<GdSmartFilter, T> stat(Function<GdEntityStats, T> extractor) {
        return c -> c.getStats() == null ? null : extractor.apply(c.getStats());
    }

    private static class StrategyComparator implements Comparator<GdSmartFilter> {

        private final Map<Long, StrategyName> strategyBySmartFilterIds;
        private GdSmartFilterOrderByField orderByField;

        StrategyComparator(Map<Long, StrategyName> strategyBySmartFilterIds, GdSmartFilterOrderByField orderByField) {
            this.strategyBySmartFilterIds = strategyBySmartFilterIds;
            this.orderByField = orderByField;
        }

        @Override
        public int compare(GdSmartFilter o1, GdSmartFilter o2) {
            Set<StrategyName> strategySet = STRATEGY_SET_BY_FIELD.get(orderByField);

            // Для смарт-баннеров стратегия не может быть null
            StrategyName o1Strategy = strategyBySmartFilterIds.get(o1.getId());
            StrategyName o2Strategy = strategyBySmartFilterIds.get(o2.getId());

            if ((!strategySet.contains(o1Strategy) && !strategySet.contains(o2Strategy)) ||
                    (strategySet.containsAll(asSet(o1Strategy, o2Strategy)))) {
                return 0;
            } else if (!strategySet.contains(o1Strategy)) {
                return 1;
            } else {
                return -1;
            }
        }
    }
}
