package ru.yandex.direct.grid.processing.service.smartfilter.converter;

import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;

import ru.yandex.direct.core.entity.feed.model.BusinessType;
import ru.yandex.direct.core.entity.feed.model.FeedType;
import ru.yandex.direct.core.entity.feed.model.Source;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilter;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterCondition;
import ru.yandex.direct.core.entity.performancefilter.schema.FilterSchema;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterConditionDBFormatParser;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterStorage;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.smartfilter.mutation.GdUpdateSmartFilterCondition;
import ru.yandex.direct.grid.processing.model.smartfilter.mutation.GdUpdateSmartFilters;
import ru.yandex.direct.grid.processing.model.smartfilter.mutation.GdUpdateSmartFiltersItem;
import ru.yandex.direct.grid.processing.service.smartfilter.SmartFilterConverter;
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService;
import ru.yandex.direct.grid.processing.service.validation.presentation.SkipByDefaultMappingPathNodeConverter;
import ru.yandex.direct.validation.result.DefaultPathNodeConverterProvider;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.PathNodeConverter;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.validation.ValidationUtils.hasAnyErrorsOrWarnings;
import static ru.yandex.direct.grid.processing.service.smartfilter.SmartFilterConverter.fromGdAutobudgetPriority;
import static ru.yandex.direct.grid.processing.service.smartfilter.SmartFilterConverter.fromGdSmartFilterTab;
import static ru.yandex.direct.grid.processing.service.smartfilter.SmartFilterConverter.fromGdTargetFunnel;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.emptyPath;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

public class GdUpdateSmartFiltersConverter {

    private static final PathNodeConverter VALIDATE_GD_UPDATE_SMART_FILTERS_ITEM_CONVERTER =
            SkipByDefaultMappingPathNodeConverter.builder()
                    .replace(PerformanceFilter.ID.name(), GdUpdateSmartFiltersItem.ID.name())
                    .replace(PerformanceFilter.NAME.name(), GdUpdateSmartFiltersItem.NAME.name())
                    .replace(PerformanceFilter.CONDITIONS.name(), GdUpdateSmartFiltersItem.CONDITIONS.name())
                    .replace(PerformanceFilter.PRICE_CPC.name(), GdUpdateSmartFiltersItem.PRICE_CPC.name())
                    .replace(PerformanceFilter.PRICE_CPA.name(), GdUpdateSmartFiltersItem.PRICE_CPA.name())
                    .replace(PerformanceFilter.AUTOBUDGET_PRIORITY.name(),
                            GdUpdateSmartFiltersItem.AUTOBUDGET_PRIORITY.name())
                    .replace(PerformanceFilter.TARGET_FUNNEL.name(), GdUpdateSmartFiltersItem.TARGET_FUNNEL.name())
                    .replace(PerformanceFilter.IS_SUSPENDED.name(), GdUpdateSmartFiltersItem.IS_SUSPENDED.name())
                    .build();

    private static final PathNodeConverter VALIDATE_GD_UPDATE_SMART_FILTERS_CONDITIONS_CONVERTER =
            new PathNodeConverter() {
                private PathNodeConverter innerConverter = SkipByDefaultMappingPathNodeConverter.builder()
                        .replace(PerformanceFilterCondition.FIELD_NAME.name(),
                                GdUpdateSmartFilterCondition.FIELD.name())
                        .replace(PerformanceFilterCondition.OPERATOR.name(),
                                GdUpdateSmartFilterCondition.OPERATOR.name())
                        .replace(PerformanceFilterCondition.STRING_VALUE.name(),
                                GdUpdateSmartFilterCondition.STRING_VALUE.name())
                        .replace(PerformanceFilterCondition.PARSED_VALUE,
                                GdUpdateSmartFilterCondition.STRING_VALUE.name())
                        .build();

                @Override
                public Path convert(PathNode.Field field) {
                    return innerConverter.convert(field);
                }

                /**
                 * Убирает индекс из пути для списочных объектов, т.к. списочные объекты в ядровой модели появляются
                 * только при парсинге нз несписочного STRING_VALUE в гридовой модели.
                 */
                @Override
                public Path convert(PathNode.Field field, PathNode.Index index) {
                    return emptyPath();
                }
            };

    private static final PathNodeConverterProvider PATH_NODE_CONVERTER_PROVIDER =
            DefaultPathNodeConverterProvider.builder()
                    .register(PerformanceFilter.class, VALIDATE_GD_UPDATE_SMART_FILTERS_ITEM_CONVERTER)
                    .register(PerformanceFilterCondition.class, VALIDATE_GD_UPDATE_SMART_FILTERS_CONDITIONS_CONVERTER)
                    .build();

    public static PerformanceFilter fromGd(GdUpdateSmartFiltersItem item,
                                           Map<Long, PerformanceFilter> oldFilterById,
                                           PerformanceFilterStorage performanceFilterStorage) {
        Long filterId = item.getId();
        PerformanceFilter oldFilter = oldFilterById.get(filterId);
        checkNotNull(oldFilter, "Old filter cannot be found.");
        BusinessType businessType = oldFilter.getBusinessType();
        FeedType feedType = oldFilter.getFeedType();
        Source source = oldFilter.getSource();

        List<PerformanceFilterCondition> conditions =
                fromGd(item.getConditions(), businessType, feedType, source, performanceFilterStorage);

        return new PerformanceFilter()
                .withId(filterId)
                .withName(item.getName())
                .withTargetFunnel(fromGdTargetFunnel(item.getTargetFunnel()))
                .withPriceCpc(item.getPriceCpc())
                .withPriceCpa(item.getPriceCpa())
                .withAutobudgetPriority(fromGdAutobudgetPriority(item.getAutobudgetPriority()))
                .withIsSuspended(item.getIsSuspended())
                .withTab(fromGdSmartFilterTab(item.getTab()))
                .withConditions(conditions);
    }

    @Nullable
    private static List<PerformanceFilterCondition> fromGd(List<GdUpdateSmartFilterCondition> gdConditions,
                                                           BusinessType businessType, FeedType feedType, Source source,
                                                           PerformanceFilterStorage performanceFilterStorage) {
        if (gdConditions == null) {
            return null;
        }
        if (gdConditions.isEmpty()) {
            return emptyList();
        }
        FilterSchema filterSchema = performanceFilterStorage.getFilterSchema(businessType, feedType, source);
        List<PerformanceFilterCondition> conditions = mapList(gdConditions,
                gdCondition -> new PerformanceFilterCondition(
                        filterSchema.translateFieldNameFromGd(gdCondition.getField()),
                        SmartFilterConverter.fromGdConditionOperator(gdCondition.getOperator()),
                        gdCondition.getStringValue()));
        PerformanceFilterConditionDBFormatParser.setParsedValue(filterSchema, conditions);
        return conditions;
    }

    @Nullable
    public static GdValidationResult toGdValidationResult(ValidationResult<?, Defect> vr) {
        if (hasAnyErrorsOrWarnings(vr)) {
            return GridValidationResultConversionService
                    .buildGridValidationResult(vr, path(field(GdUpdateSmartFilters.UPDATE_ITEMS)),
                            PATH_NODE_CONVERTER_PROVIDER);
        }
        return null;
    }


}
