package ru.yandex.direct.grid.processing.service.statistics;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.http.client.utils.URIBuilder;
import org.apache.poi.ss.usermodel.Workbook;
import org.jetbrains.annotations.Nullable;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.brandsafety.translation.BrandSafetyTranslations;
import ru.yandex.direct.core.entity.campaign.model.CampaignSimple;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.mdsfile.model.MdsFileSaveRequest;
import ru.yandex.direct.core.entity.mdsfile.model.MdsStorageType;
import ru.yandex.direct.core.entity.mdsfile.service.MdsFileService;
import ru.yandex.direct.core.util.ReflectionTranslator;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.excel.processing.model.brandsafety.BrandSafetyStatsExcelExportData;
import ru.yandex.direct.excel.processing.service.brandsafety.BrandSafetyStatsExcelExportService;
import ru.yandex.direct.grid.core.entity.statistics.brandsafety.model.GdiBrandSafetyStatsRow;
import ru.yandex.direct.grid.core.entity.statistics.repository.BrandSafetyStatsYtRepository;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.model.statistics.brandsafety.GdBrandSafetyStatsExcelExportLanguage;
import ru.yandex.direct.grid.processing.model.statistics.brandsafety.GdBrandSafetyStatsExcelExportRequest;
import ru.yandex.direct.grid.processing.model.statistics.brandsafety.GdBrandSafetyStatsRequest;
import ru.yandex.direct.grid.processing.model.statistics.brandsafety.GdBrandSafetyStatsRow;
import ru.yandex.direct.grid.processing.model.statistics.brandsafety.GdBrandSafetyStatsTotalRequest;
import ru.yandex.direct.grid.processing.service.statistics.validation.BrandSafetyStatisticsValidationService;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.utils.HashingUtils;
import ru.yandex.direct.web.core.entity.inventori.service.CryptaService;
import ru.yandex.direct.web.core.model.retargeting.CryptaGoalWeb;

import static java.util.Comparator.comparing;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toUnmodifiableList;
import static ru.yandex.direct.grid.processing.model.statistics.brandsafety.GdBrandSafetyStatsExcelExportLanguage.EN;
import static ru.yandex.direct.grid.processing.service.statistics.converter.BrandSafetyStatsConverter.toBrandSafetyStatsExcelExportData;
import static ru.yandex.direct.grid.processing.service.statistics.converter.BrandSafetyStatsConverter.toGdBrandSafetyStatsRow;
import static ru.yandex.direct.grid.processing.service.statistics.converter.BrandSafetyStatsConverter.toGdiBrandSafetyStatsRequest;
import static ru.yandex.direct.grid.processing.service.statistics.converter.BrandSafetyStatsConverter.toGdiBrandSafetyStatsTotalRequest;
import static ru.yandex.direct.grid.processing.service.statistics.converter.BrandSafetyStatsConverter.toGdiBrandSafetyStatsTotalRequestFromExcel;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

@Service
@ParametersAreNonnullByDefault
public class BrandSafetyStatisticsDataService {

    private static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd");
    private static final Locale RUSSIAN_LOCALE = new Locale("ru");

    private final BrandSafetyStatisticsValidationService brandSafetyStatisticsValidationService;
    private final BrandSafetyStatsYtRepository brandSafetyStatsYtRepository;
    private final BrandSafetyStatsExcelExportService brandSafetyStatsExcelExportService;
    private final CampaignRepository campaignRepository;
    private final CryptaService cryptaService;
    private final ReflectionTranslator reflectionTranslator;
    private final MdsFileService mdsFileService;
    private final ShardHelper shardHelper;

    public BrandSafetyStatisticsDataService(
            BrandSafetyStatsYtRepository brandSafetyStatsYtRepository,
            CampaignRepository campaignRepository,
            CryptaService cryptaService,
            ShardHelper shardHelper,
            BrandSafetyStatisticsValidationService brandSafetyStatisticsValidationService,
            BrandSafetyStatsExcelExportService brandSafetyStatsExcelExportService,
            ReflectionTranslator reflectionTranslator,
            MdsFileService mdsFileService
    ) {
        this.brandSafetyStatsYtRepository = brandSafetyStatsYtRepository;
        this.campaignRepository = campaignRepository;
        this.cryptaService = cryptaService;
        this.shardHelper = shardHelper;
        this.brandSafetyStatisticsValidationService = brandSafetyStatisticsValidationService;
        this.brandSafetyStatsExcelExportService = brandSafetyStatsExcelExportService;
        this.reflectionTranslator = reflectionTranslator;
        this.mdsFileService = mdsFileService;
    }

    public List<GdBrandSafetyStatsRow> getBrandSafetyStats(GdClientInfo clientInfo, GdBrandSafetyStatsRequest request) {
        brandSafetyStatisticsValidationService.validateBrandSafetyStatsRequest(request);

        var clientId = ClientId.fromLong(clientInfo.getId());
        var shardId = shardHelper.getShardByClientIdStrictly(clientId);

        var orderIdToCampaignMap = getOrderIdToCampaignMap(clientId, shardId, request.getCids());

        var gdiRequest = toGdiBrandSafetyStatsRequest(request, new ArrayList<>(orderIdToCampaignMap.keySet()));
        List<GdiBrandSafetyStatsRow> rows;
        try (TraceProfile profile = Trace.current().profile("brand_safety:stats_get_data_from_yt")) {
            rows = brandSafetyStatsYtRepository.getStats(gdiRequest);
        }

        var brandSafety = cryptaService.getBrandSafety();
        var keywordToGoalIdMap = listToMap(
                brandSafety, bs -> Long.parseLong(bs.getKeywordValue()), CryptaGoalWeb::getId);

        return rows.stream()
                .map(r -> toGdBrandSafetyStatsRow(r, orderIdToCampaignMap, keywordToGoalIdMap, gdiRequest))
                .collect(toList());
    }

    public Double getBrandSafetyStatsTotal(GdClientInfo clientInfo, GdBrandSafetyStatsTotalRequest request) {
        var clientId = ClientId.fromLong(clientInfo.getId());
        var shardId = shardHelper.getShardByClientIdStrictly(clientId);

        var orderIds = new ArrayList<>(getOrderIdToCampaignMap(clientId, shardId, request.getCids()).keySet());

        var gdiRequest = toGdiBrandSafetyStatsTotalRequest(request, orderIds);
        try (TraceProfile profile = Trace.current().profile("brand_safety:stats_get_total_from_yt")) {
            return brandSafetyStatsYtRepository.getStatsTotal(gdiRequest);
        }
    }

    public String exportBrandSafetyStatsToExcel(
            GdClientInfo clientInfo, GdBrandSafetyStatsExcelExportRequest request) {
        var clientId = ClientId.fromLong(clientInfo.getId());
        var shardId = shardHelper.getShardByClientIdStrictly(clientId);

        var orderIdToCampaignMap = getOrderIdToCampaignMap(clientId, shardId, request.getCids());

        var gdiRequest = toGdiBrandSafetyStatsRequest(request, new ArrayList<>(orderIdToCampaignMap.keySet()));

        List<GdiBrandSafetyStatsRow> rows;
        try (TraceProfile profile = Trace.current().profile("brand_safety:excel_get_data_from_yt")) {
            rows = brandSafetyStatsYtRepository.getStats(gdiRequest);
        }

        // YT cannot sort without limit, let's help YT
        rows = rows.stream()
                .sorted(comparing(GdiBrandSafetyStatsRow::getDate))
                .collect(toUnmodifiableList());

        var keywordToCategoryNameMap = getKeywordToCategoryMap(request.getLanguage());

        var gdiTotalRequest = toGdiBrandSafetyStatsTotalRequestFromExcel(
                request, new ArrayList<>(orderIdToCampaignMap.keySet()));
        var total = brandSafetyStatsYtRepository.getStatsTotal(gdiTotalRequest);

        var exportData = toBrandSafetyStatsExcelExportData(
                request, rows, orderIdToCampaignMap, keywordToCategoryNameMap, clientInfo.getChiefUser(), total);

        Workbook statsWorkbook;
        try (TraceProfile profile = Trace.current().profile("brand_safety:excel_create_workbook")) {
            statsWorkbook = brandSafetyStatsExcelExportService.createWorkbook(exportData);
        }

        try (TraceProfile profile = Trace.current().profile("brand_safety:excel_upload_to_mds")) {
            var login = clientInfo.getChiefUser().getLogin();
            var fileName = getFileName(exportData, login);

            var mdsData = saveToMds(statsWorkbook, fileName, clientId);

            var fileImprint = mdsData.getMdsMetadata().getFileImprint();
            return getDownloadUrl(login, fileImprint);
        }
    }

    private Map<Long, CampaignSimple> getOrderIdToCampaignMap(ClientId clientId, int shardId, @Nullable List<Long> cids) {
        if (cids == null || cids.isEmpty()) {
            cids = campaignRepository.getCampaignIdsWithBrandSafety(shardId, clientId.asLong());
        }

        var campaigns = campaignRepository.getCampaignsForClient(shardId, clientId, cids);
        return campaigns.stream()
                .filter(c -> c.getOrderId() != 0)
                .collect(toMap(CampaignSimple::getOrderId, c -> c));
    }

    private String getDownloadUrl(String login, String fileImprint) {
        var uriBuilder = new URIBuilder()
                .setPath("/web-api/stat/brand_safety_excel_report")
                .addParameter("file_key", fileImprint)
                .addParameter("ulogin", login);
        return uriBuilder.toString();
    }

    @Nullable
    private Map<Long, String> getKeywordToCategoryMap(GdBrandSafetyStatsExcelExportLanguage language) {
        var brandSafety = cryptaService.getBrandSafety();
        return listToMap(
                brandSafety,
                goal -> Long.parseLong(goal.getKeywordValue()),
                goal -> reflectionTranslator.translate(
                        goal.getTankerNameKey(),
                        BrandSafetyTranslations.INSTANCE,
                        EN.equals(language) ? Locale.ENGLISH : RUSSIAN_LOCALE));
    }

    private String getFileName(BrandSafetyStatsExcelExportData exportData, String login) {
        return String.format(
                "brandsafety_%s_%s_%s.xlsx",
                DATE_TIME_FORMATTER.format(exportData.getStartDate()),
                DATE_TIME_FORMATTER.format(exportData.getEndDate()),
                login);
    }

    MdsFileSaveRequest saveToMds(Workbook statsWorkbook, String name, ClientId clientId) {
        byte[] data = toBytes(statsWorkbook);
        var hash = HashingUtils.getMd5HashAsBase64YaStringWithoutPadding(data);
        var path = MdsStorageType.BRAND_SAFETY_STATS_REPORT.name().toLowerCase() + "/" + clientId;
        var mdsRequest =
                new MdsFileSaveRequest(MdsStorageType.BRAND_SAFETY_STATS_REPORT, data, hash, path).withCustomName(name);
        return mdsFileService.saveMdsFiles(List.of(mdsRequest), clientId.asLong()).get(0);
    }

    static byte[] toBytes(Workbook workbook) {
        try (var os = new ByteArrayOutputStream()) {
            workbook.write(os);
            return os.toByteArray();
        } catch (IOException e) {
            throw new RuntimeException("Couldn't get bytes from workbook");
        }
    }
}
