package ru.yandex.direct.grid.processing.service.statistics;

import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLContext;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.client.GdClient;
import ru.yandex.direct.grid.processing.model.masterreport.GdMasterReportStatisticsContainer;
import ru.yandex.direct.grid.processing.model.masterreport.GdMasterReportStatisticsPayload;
import ru.yandex.direct.grid.processing.model.statistics.GdCampaignStatisticsContainer;
import ru.yandex.direct.grid.processing.model.statistics.GdCampaignStatisticsPayload;
import ru.yandex.direct.grid.processing.model.statistics.brandsafety.GdBrandSafetyStatsExcelExportRequest;
import ru.yandex.direct.grid.processing.model.statistics.brandsafety.GdBrandSafetyStatsExcelExportResponse;
import ru.yandex.direct.grid.processing.model.statistics.brandsafety.GdBrandSafetyStatsRequest;
import ru.yandex.direct.grid.processing.model.statistics.brandsafety.GdBrandSafetyStatsResponse;
import ru.yandex.direct.grid.processing.model.statistics.brandsafety.GdBrandSafetyStatsTotalRequest;
import ru.yandex.direct.grid.processing.model.statistics.brandsafety.GdBrandSafetyStatsTotalResponse;
import ru.yandex.direct.grid.processing.model.statistics.metrika.GdMetrikaStatisticsContainer;
import ru.yandex.direct.grid.processing.model.statistics.metrika.GdMetrikaStatisticsPayload;
import ru.yandex.direct.grid.processing.service.statistics.service.CampaignStatisticsService;
import ru.yandex.direct.grid.processing.service.statistics.service.EndToEndAnalyticsService;
import ru.yandex.direct.grid.processing.service.statistics.service.MasterReportStatisticsService;

@GridGraphQLService
@ParametersAreNonnullByDefault
public class StatisticsGraphQlService {

    private final CampaignStatisticsService campaignStatisticsService;
    private final BrandSafetyStatisticsDataService brandSafetyStatisticsDataService;
    private final MasterReportStatisticsService masterReportStatisticsService;
    private final EndToEndAnalyticsService endToEndAnalyticsService;

    @Autowired
    public StatisticsGraphQlService(
            CampaignStatisticsService campaignStatisticsService,
            EndToEndAnalyticsService endToEndAnalyticsService,
            BrandSafetyStatisticsDataService brandSafetyStatisticsDataService,
            MasterReportStatisticsService masterReportStatisticsService
    ) {
        this.campaignStatisticsService = campaignStatisticsService;
        this.brandSafetyStatisticsDataService = brandSafetyStatisticsDataService;
        this.endToEndAnalyticsService = endToEndAnalyticsService;
        this.masterReportStatisticsService = masterReportStatisticsService;
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "campaignStatistics")
    public GdCampaignStatisticsPayload getCampaignStatistics(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdCampaignStatisticsContainer input) {
        return campaignStatisticsService.getCampaignStatistics(input, client, context);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "masterReportStatistics")
    public GdMasterReportStatisticsPayload getMasterReportStatistics(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdMasterReportStatisticsContainer input) {
        return masterReportStatisticsService.getStatistics(input, context);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "brandSafetyStats")
    public GdBrandSafetyStatsResponse getBrandSafetyStats(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdBrandSafetyStatsRequest input) {
        return new GdBrandSafetyStatsResponse()
                .withRows(brandSafetyStatisticsDataService.getBrandSafetyStats(client.getInfo(), input));
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "brandSafetyStatsTotal")
    public GdBrandSafetyStatsTotalResponse getBrandSafetyStatsTotal(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdBrandSafetyStatsTotalRequest input) {
        Double totalPercentage = brandSafetyStatisticsDataService.getBrandSafetyStatsTotal(client.getInfo(), input);
        return new GdBrandSafetyStatsTotalResponse()
                .withCount(totalPercentage.longValue())
                .withPercentage(totalPercentage);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "brandSafetyStatsExcelExport")
    public GdBrandSafetyStatsExcelExportResponse getBrandSafetyStatsExcelExport(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdBrandSafetyStatsExcelExportRequest input) {
        return new GdBrandSafetyStatsExcelExportResponse()
                .withFileUrl(brandSafetyStatisticsDataService.exportBrandSafetyStatsToExcel(client.getInfo(), input));
    }

    /**
     * Получить данные по сквозной аналитике из Метрики
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "endToEndAnalyticsStat")
    public GdMetrikaStatisticsPayload getEndToEndAnalyticsStats(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdMetrikaStatisticsContainer input
    ) {
        return endToEndAnalyticsService.getEndToEndStatistics(input, context, client);
    }

    /**
     * Получить данные по рекламным системам и источникам трафика из Метрики
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "metrikaMarketingStat")
    public GdMetrikaStatisticsPayload getMetrikaMarketingStats(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdMetrikaStatisticsContainer input
    ) {
        return endToEndAnalyticsService.getMetrikaMarketingStatistics(input, context, client);
    }
}
