package ru.yandex.direct.grid.processing.service.statistics.converter;

import java.time.Instant;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.campaign.model.CampaignSimple;
import ru.yandex.direct.excel.processing.model.brandsafety.BrandSafetyStatsExcelExportData;
import ru.yandex.direct.excel.processing.model.brandsafety.BrandSafetyStatsExcelExportLanguage;
import ru.yandex.direct.excel.processing.model.brandsafety.BrandSafetyStatsExcelExportRow;
import ru.yandex.direct.grid.core.entity.model.GdiLimitOffset;
import ru.yandex.direct.grid.core.entity.statistics.brandsafety.model.GdiBrandSafetyStatsGroupBy;
import ru.yandex.direct.grid.core.entity.statistics.brandsafety.model.GdiBrandSafetyStatsLanguage;
import ru.yandex.direct.grid.core.entity.statistics.brandsafety.model.GdiBrandSafetyStatsOrderBy;
import ru.yandex.direct.grid.core.entity.statistics.brandsafety.model.GdiBrandSafetyStatsOrderByField;
import ru.yandex.direct.grid.core.entity.statistics.brandsafety.model.GdiBrandSafetyStatsRequest;
import ru.yandex.direct.grid.core.entity.statistics.brandsafety.model.GdiBrandSafetyStatsRow;
import ru.yandex.direct.grid.core.entity.statistics.brandsafety.model.GdiBrandSafetyStatsTotalRequest;
import ru.yandex.direct.grid.processing.model.client.GdUserInfo;
import ru.yandex.direct.grid.processing.model.statistics.brandsafety.GdBrandSafetyStatsExcelExportLanguage;
import ru.yandex.direct.grid.processing.model.statistics.brandsafety.GdBrandSafetyStatsExcelExportRequest;
import ru.yandex.direct.grid.processing.model.statistics.brandsafety.GdBrandSafetyStatsGroupBy;
import ru.yandex.direct.grid.processing.model.statistics.brandsafety.GdBrandSafetyStatsRequest;
import ru.yandex.direct.grid.processing.model.statistics.brandsafety.GdBrandSafetyStatsRow;
import ru.yandex.direct.grid.processing.model.statistics.brandsafety.GdBrandSafetyStatsTotalRequest;

import static ru.yandex.direct.utils.DateTimeUtils.instantToMoscowDate;
import static ru.yandex.direct.utils.DateTimeUtils.startOfDayInMoscowTime;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class BrandSafetyStatsConverter {
    public static final Locale RUSSIAN_LOCALE = new Locale("ru");

    private static final DateTimeFormatter DAY_FORMATTER = DateTimeFormatter.ofPattern("dd.MM.yyyy");
    private static final DateTimeFormatter MONTH_FORMATTER = DateTimeFormatter.ofPattern("LLLL yyyy");
    private static final DateTimeFormatter QUARTER_FORMATTER = DateTimeFormatter.ofPattern("qqqq yyyy");

    public static GdiBrandSafetyStatsRequest toGdiBrandSafetyStatsRequest(
            GdBrandSafetyStatsRequest gdRequest,
            List<Long> orderIds
    ) {
        return new GdiBrandSafetyStatsRequest()
                .withStartDate(startOfDayInMoscowTime(gdRequest.getStartDate()))
                .withEndDate(startOfDayInMoscowTime(gdRequest.getEndDate()))
                .withOrderIds(orderIds)
                .withShowCampaigns(gdRequest.getShowCampaigns() != null && gdRequest.getShowCampaigns())
                .withShowCategories(gdRequest.getShowCategories() != null && gdRequest.getShowCategories())
                .withLimitOffset(new GdiLimitOffset()
                        .withLimit(gdRequest.getLimitOffset().getLimit())
                        .withOffset(gdRequest.getLimitOffset().getOffset()))
                .withOrderBy(new GdiBrandSafetyStatsOrderBy()
                        .withOrder(gdRequest.getOrderBy().getOrder())
                        .withField(GdiBrandSafetyStatsOrderByField.valueOf(gdRequest.getOrderBy().getField().name())))
                .withGroupBy(GdBrandSafetyStatsGroupBy.toSource(
                        Optional.ofNullable(gdRequest.getGroupBy()).orElse(GdBrandSafetyStatsGroupBy.DAY)))
                .withLanguage(GdBrandSafetyStatsExcelExportLanguage.toSource(
                        Optional.ofNullable(gdRequest.getLanguage()).orElse(GdBrandSafetyStatsExcelExportLanguage.RU)));
    }

    public static GdiBrandSafetyStatsRequest toGdiBrandSafetyStatsRequest(
            GdBrandSafetyStatsExcelExportRequest gdRequest,
            List<Long> orderIds
    ) {
        return new GdiBrandSafetyStatsRequest()
                .withStartDate(startOfDayInMoscowTime(gdRequest.getStartDate()))
                .withEndDate(startOfDayInMoscowTime(gdRequest.getEndDate()))
                .withOrderIds(orderIds)
                .withShowCampaigns(gdRequest.getShowCampaigns() != null && gdRequest.getShowCampaigns())
                .withShowCategories(gdRequest.getShowCategories() != null && gdRequest.getShowCategories())
                .withLimitOffset(null)
                .withOrderBy(null)
                .withGroupBy(GdBrandSafetyStatsGroupBy.toSource(
                        Optional.ofNullable(gdRequest.getGroupBy()).orElse(GdBrandSafetyStatsGroupBy.DAY)))
                .withLanguage(GdBrandSafetyStatsExcelExportLanguage.toSource(
                        Optional.ofNullable(gdRequest.getLanguage()).orElse(GdBrandSafetyStatsExcelExportLanguage.RU)));
    }

    public static GdiBrandSafetyStatsTotalRequest toGdiBrandSafetyStatsTotalRequestFromExcel(
            GdBrandSafetyStatsExcelExportRequest gdRequest,
            List<Long> orderIds
    ) {
        return new GdiBrandSafetyStatsTotalRequest()
                .withStartDate(startOfDayInMoscowTime(gdRequest.getStartDate()))
                .withEndDate(startOfDayInMoscowTime(gdRequest.getEndDate()))
                .withOrderIds(orderIds);
    }

    public static GdBrandSafetyStatsRow toGdBrandSafetyStatsRow(
            GdiBrandSafetyStatsRow gdiRow,
            Map<Long, CampaignSimple> orderIdToCampaignMap,
            Map<Long, Long> keyWordToGoalIdMap,
            GdiBrandSafetyStatsRequest request
    ) {
        var campaign = gdiRow.getOrderId() == null ? null : orderIdToCampaignMap.get(gdiRow.getOrderId());
        var goalId = gdiRow.getCategory() == null ? null : keyWordToGoalIdMap.get(gdiRow.getCategory());
        var date = instantToMoscowDate(Instant.ofEpochSecond(gdiRow.getDate()));

        return new GdBrandSafetyStatsRow()
                .withDate(date)
                .withDateString(getDateString(date, request))
                .withCid(campaign == null ? null : campaign.getId())
                .withCampaignName(campaign == null ? null : campaign.getName())
                .withGoalId(goalId)
                .withCount(gdiRow.getBannedCount())
                .withPercentage(gdiRow.getBannedPercentage());
    }

    private static String getDateString(LocalDate date, GdBrandSafetyStatsExcelExportRequest request) {
        return getDateString(date, GdBrandSafetyStatsGroupBy.toSource(request.getGroupBy()),
                GdBrandSafetyStatsExcelExportLanguage.toSource(request.getLanguage()),
                startOfDayInMoscowTime(request.getStartDate()),
                startOfDayInMoscowTime(request.getEndDate()));
    }

    private static String getDateString(LocalDate date, GdiBrandSafetyStatsRequest request) {
        return getDateString(date, request.getGroupBy(), request.getLanguage(), request.getStartDate(), request.getEndDate());
    }

    private static String getDateString(LocalDate date, GdiBrandSafetyStatsGroupBy groupBy,
                                        GdiBrandSafetyStatsLanguage language, Long startDate, Long endDate) {
        switch (groupBy) {
            case DAY:
                return DAY_FORMATTER.format(date);

            case WEEK:
                return DAY_FORMATTER.format(date) + " - " + DAY_FORMATTER.format(date.plusDays(6));

            case MONTH:
                return MONTH_FORMATTER.withLocale(getLocale(language)).format(date)
                        .replace(".", "");

            case QUARTER:
                return QUARTER_FORMATTER.withLocale(getLocale(language)).format(date)
                        .replace("st", "")
                        .replace("nd", "")
                        .replace("th", "")
                        .replace("-й", "");

            case YEAR:
                return date.getYear() + (GdiBrandSafetyStatsLanguage.RU.equals(language) ? " г." : "");

            case SELECTED_PERIOD:
                return DAY_FORMATTER.format(instantToMoscowDate(Instant.ofEpochSecond(startDate)))
                        + " - " + DAY_FORMATTER.format(instantToMoscowDate(Instant.ofEpochSecond(endDate)));
            default:
                throw new IllegalArgumentException("Unknown type of grouping " + groupBy);
        }
    }

    private static Locale getLocale(GdiBrandSafetyStatsLanguage language) {
        switch (language) {
            case EN:
                return Locale.ENGLISH;
            case RU:
                return RUSSIAN_LOCALE;
            default:
                throw new IllegalArgumentException("Unknown language " + language);
        }
    }

    public static GdiBrandSafetyStatsTotalRequest toGdiBrandSafetyStatsTotalRequest(
            GdBrandSafetyStatsTotalRequest gdRequest,
            List<Long> orderIds
    ) {
        return new GdiBrandSafetyStatsTotalRequest()
                .withStartDate(startOfDayInMoscowTime(gdRequest.getStartDate()))
                .withEndDate(startOfDayInMoscowTime(gdRequest.getEndDate()))
                .withOrderIds(orderIds);
    }

    public static BrandSafetyStatsExcelExportData toBrandSafetyStatsExcelExportData(
            GdBrandSafetyStatsExcelExportRequest gdRequest,
            List<GdiBrandSafetyStatsRow> gdiRows,
            Map<Long, CampaignSimple> orderIdToCampaignMap,
            Map<Long, String> keyWordToCategoryNameMap,
            GdUserInfo chiefUser,
            Double total
    ) {
        return new BrandSafetyStatsExcelExportData()
                .withStartDate(gdRequest.getStartDate())
                .withEndDate(gdRequest.getEndDate())
                .withShowCampaigns(gdRequest.getShowCampaigns())
                .withShowCategories(gdRequest.getShowCategories())
                .withLanguage(toExcelExportLanguage(gdRequest.getLanguage()))
                .withUserName(chiefUser.getName())
                .withUserLogin(chiefUser.getLogin())
                .withTotal(total)
                .withRows(mapList(gdiRows, row ->
                        toExcelExportRow(row, gdRequest, orderIdToCampaignMap, keyWordToCategoryNameMap)));
    }

    private static BrandSafetyStatsExcelExportLanguage toExcelExportLanguage(
            GdBrandSafetyStatsExcelExportLanguage language) {
        switch (language) {
            case EN:
                return BrandSafetyStatsExcelExportLanguage.EN;
            default:
                return BrandSafetyStatsExcelExportLanguage.RU;
        }
    }

    private static BrandSafetyStatsExcelExportRow toExcelExportRow(
            GdiBrandSafetyStatsRow gdiRow,
            GdBrandSafetyStatsExcelExportRequest gdRequest,
            Map<Long, CampaignSimple> orderIdToCampaignMap,
            Map<Long, String> keyWordToCategoryNameMap) {

        var campaign = gdiRow.getOrderId() == null ? null : orderIdToCampaignMap.get(gdiRow.getOrderId());
        var categoryName = gdiRow.getCategory() == null ?
                "" :
                keyWordToCategoryNameMap.get(gdiRow.getCategory());

        return new BrandSafetyStatsExcelExportRow()
                .withDateString(getDateString(instantToMoscowDate(Instant.ofEpochSecond(gdiRow.getDate())), gdRequest))
                .withCid(campaign == null ? 0L : campaign.getId())
                .withCampaignName(campaign == null ? "" : campaign.getName())
                .withCategoryName(categoryName)
                .withPercentage(gdiRow.getBannedPercentage());
    }
}
