package ru.yandex.direct.grid.processing.service.statistics.converter

import java.math.BigDecimal
import java.math.RoundingMode
import java.time.Instant
import java.time.temporal.ChronoUnit
import ru.yandex.direct.bannersystem.container.masterreport.MasterReportGroupByDate
import ru.yandex.direct.bannersystem.container.masterreport.MasterReportRow
import ru.yandex.direct.bannersystem.container.masterreport.absDelta
import ru.yandex.direct.bannersystem.container.masterreport.dict.MasterReportCampaignPlatform
import ru.yandex.direct.core.entity.campaign.model.Campaign
import ru.yandex.direct.core.entity.campaign.model.CampaignAttributionModel
import ru.yandex.direct.core.entity.campaign.model.CampaignCalcType
import ru.yandex.direct.core.entity.masterreport.model.MasterReportFilters
import ru.yandex.direct.core.entity.masterreport.model.MasterReportPeriod
import ru.yandex.direct.core.entity.retargeting.model.Goal
import ru.yandex.direct.grid.model.campaign.GdCampaignAttributionModel
import ru.yandex.direct.grid.model.campaign.GdCampaignPlatform
import ru.yandex.direct.grid.model.campaign.strategy.GdStrategyType
import ru.yandex.direct.grid.processing.model.campaign.GdCalcType
import ru.yandex.direct.grid.processing.model.masterreport.GdMasterReportCampaignStatus
import ru.yandex.direct.grid.processing.model.masterreport.GdMasterReportGoalStatisticsColumnValues
import ru.yandex.direct.grid.processing.model.masterreport.GdMasterReportStatisticsCampaign
import ru.yandex.direct.grid.processing.model.masterreport.GdMasterReportStatisticsColumnValues
import ru.yandex.direct.grid.processing.model.masterreport.GdMasterReportStatisticsContainer
import ru.yandex.direct.grid.processing.model.masterreport.GdMasterReportStatisticsFilter
import ru.yandex.direct.grid.processing.model.masterreport.GdMasterReportStatisticsGroupByDate
import ru.yandex.direct.grid.processing.model.masterreport.GdMasterReportStatisticsItem
import ru.yandex.direct.grid.processing.model.statistics.GdCampaignStatisticsValueHolder
import ru.yandex.direct.grid.processing.util.StatHelper

private val MILLION = BigDecimal.valueOf(1_000_000)

object MasterReportStatisticsConverter {

    @JvmStatic
    fun buildPeriod(input: GdMasterReportStatisticsContainer, instant: Instant): MasterReportPeriod {
        val inputPeriod = input.filter.period
        if (inputPeriod.preset == null) {
            return MasterReportPeriod(inputPeriod.from, inputPeriod.to)
        }
        val period = StatHelper.getDayPeriod(inputPeriod.preset, instant, null)
        return MasterReportPeriod(period.left, period.right)
    }

    @JvmStatic
    fun buildComparePeriod(input: GdMasterReportStatisticsContainer, period: MasterReportPeriod): MasterReportPeriod? {
        return if (input.filter == null || true != input.needComparePeriods) {
            null
        } else {
            val betweenFromTo = ChronoUnit.DAYS.between(period.from, period.to)
            val to = period.from.minusDays(1)
            val from = to.minusDays(betweenFromTo)
            MasterReportPeriod(from, to)
        }
    }

    @JvmStatic
    fun convertAttributionModel(attributionModel: GdCampaignAttributionModel) =
            GdCampaignAttributionModel.toSource(attributionModel)
                    ?: CampaignAttributionModel.LAST_YANDEX_DIRECT_CLICK_CROSS_DEVICE

    @JvmStatic
    fun convertGroupByDate(value: GdMasterReportStatisticsGroupByDate?): MasterReportGroupByDate {
        return if (value == null) {
            MasterReportGroupByDate.NONE
        } else when (value) {
            // Группировка по часам работает не через groupByDate, а через groupBy
            GdMasterReportStatisticsGroupByDate.HOUR -> MasterReportGroupByDate.NONE
            GdMasterReportStatisticsGroupByDate.DAY -> MasterReportGroupByDate.DAY
            GdMasterReportStatisticsGroupByDate.WEEK -> MasterReportGroupByDate.WEEK
            GdMasterReportStatisticsGroupByDate.MONTH -> MasterReportGroupByDate.MONTH
            GdMasterReportStatisticsGroupByDate.QUARTER -> MasterReportGroupByDate.QUARTER
            GdMasterReportStatisticsGroupByDate.YEAR -> MasterReportGroupByDate.YEAR
        }
    }

    @JvmStatic
    fun convertFilters(filter: GdMasterReportStatisticsFilter, filterGoalIds: Set<Long>): MasterReportFilters {
        val statusIn = filter.statusIn?.mapNotNull { GdMasterReportCampaignStatus.toSource(it) }
        val platformIn = filter.platformIn?.mapNotNull { GdCampaignPlatform.toSource(it) }
        val strategyTypeIn = filter.strategyTypeIn?.mapNotNull { GdStrategyType.toSource(it) }
        return MasterReportFilters(
                campaignIds = filter.campaignIds ?: emptySet(),
                goalIds = filterGoalIds,
                statusIn = statusIn?.toSet() ?: emptySet(),
                platformIn = platformIn?.toSet() ?: emptySet(),
                strategyTypeIn = strategyTypeIn?.toSet() ?: emptySet(),
                regionIdsIn = filter.regionIdsIn ?: emptySet(),
                calcType = GdCalcType.toSource(filter.calcType) ?: CampaignCalcType.CPC
        )
    }

    @JvmStatic
    fun toGdItem(
            row: MasterReportRow,
            campaign: Campaign?,
            goalById: Map<Long, Goal>
    ): GdMasterReportStatisticsItem {
        val gdCampaign = if (campaign == null) {
            null
        } else {
            GdMasterReportStatisticsCampaign().withId(campaign.id).withName(campaign.name)
        }
        return GdMasterReportStatisticsItem().apply {
            this.campaign = gdCampaign
            period = row.eventTime ?: (row.period)
            platform = toGdCampaignPlatform(row.platform)
            columnValues = toGdItemValues(row, goalById = goalById)
        }
    }

    @JvmStatic
    fun toGdItemValues(
            first: MasterReportRow,
            second: MasterReportRow? = null,
            goalById: Map<Long, Goal> = emptyMap()
    ): GdMasterReportStatisticsColumnValues {
        return GdMasterReportStatisticsColumnValues()
                .withShows(toGdValue(first.shows, first.showsAbsDelta(second)))
                .withClicks(toGdValue(first.clicks, first.clicksAbsDelta(second)))
                .withConversions(toGdValue(first.conversions, first.conversionsAbsDelta(second)))
                .withCost(toGdValue(first.cost, first.costAbsDelta(second)))
                .withIncome(toGdValue(first.income, first.incomeAbsDelta(second)))
                .withProfit(toGdValue(first.profit, first.profitAbsDelta(second)))
                .withUniqViewers(toGdValue(first.uniqViewers, first.uniqViewersAbsDelta(second)))
                .withDepth(toGdValue(first.depth, first.depthAbsDelta(second)))
                .withCostPerConversion(toGdValue(first.costPerConversion, first.costPerConversionAbsDelta(second)))
                .withAvgCpc(toGdValue(first.avgCpc, first.avgCpcAbsDelta(second)))
                .withCtr(toGdValue(first.ctr, first.ctrAbsDelta(second)))
                .withConversionRate(toGdValue(first.conversionRate, first.conversionRateAbsDelta(second)))
                .withRoi(toGdValue(first.roi, first.roiAbsDelta(second)))
                .withCrr(toGdValue(first.crr, first.crrAbsDelta(second)))
                .withBounceRatio(toGdValue(first.bounceRatio, first.bounceRatioAbsDelta(second)))
                .withGoalStatistics(toGdGoalStatistics(first, second, goalById))
    }

    private fun toGdCampaignPlatform(platform: MasterReportCampaignPlatform?) =
            when (platform) {
                null -> null
                MasterReportCampaignPlatform.CONTEXT -> GdCampaignPlatform.CONTEXT
                else -> GdCampaignPlatform.SEARCH
            }

    private fun toGdGoalStatistics(
            first: MasterReportRow,
            second: MasterReportRow?,
            goalById: Map<Long, Goal>
    ): List<GdMasterReportGoalStatisticsColumnValues>? {
        return if (goalById.isEmpty()) {
            null
        } else {
            val secondGoalsData = second?.multiGoalsData?.associateBy { it.id }
            first.multiGoalsData?.map {
                val goalData = secondGoalsData?.get(it.id)
                val firstProfit = absDelta(it.income, first.cost)
                val secondProfit = absDelta(goalData?.income, second?.cost)
                GdMasterReportGoalStatisticsColumnValues()
                        .withGoalId(it.id)
                        .withName(goalById[it.id]?.name)
                        .withConversions(toGdValue(it.conversions, it.conversionsAbsDelta(goalData)))
                        .withCostPerConversion(toGdValue(it.costPerConversion, it.costPerConversionAbsDelta(goalData)))
                        .withConversionRate(toGdValue(it.conversionRate, it.conversionRateAbsDelta(goalData)))
                        .withIncome(toGdValue(it.income, it.incomeAbsDelta(goalData)))
                        .withRoi(toGdValue(it.roi, it.roiAbsDelta(goalData)))
                        .withCrr(toGdValue(it.crr, it.crrAbsDelta(goalData)))
                        .withProfit(toGdValue(firstProfit, absDelta(firstProfit, secondProfit)))
            }
        }
    }

    private fun toGdValue(value: Number? = null, valueAbsDelta: Number? = null) =
            if (value is Long && valueAbsDelta is Long) {
                GdCampaignStatisticsValueHolder()
                        .withValue(value)
                        .withValueAbsDelta(valueAbsDelta)
            } else {
                GdCampaignStatisticsValueHolder()
                        .withValue(round(value))
                        .withValueAbsDelta(round(valueAbsDelta))
            }

    private fun toGdValue(value: BigDecimal?, valueAbsDelta: BigDecimal? = null) =
            GdCampaignStatisticsValueHolder()
                    .withValue(divideMoney(value))
                    .withValueAbsDelta(divideMoney(valueAbsDelta))

    private fun round(v: Number?): Double? {
        if (v == null) {
            return null
        }
        return BigDecimal.valueOf(v.toDouble()).setScale(2, RoundingMode.HALF_UP).toDouble()
    }

    private fun divideMoney(v: BigDecimal?): BigDecimal? {
        return v?.divide(MILLION)?.setScale(2, RoundingMode.HALF_UP)
    }
}
