package ru.yandex.direct.grid.processing.service.statistics.service;

import java.time.Duration;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.net.NetAcl;
import ru.yandex.direct.core.entity.campaign.model.CampaignSimple;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.CampMetrikaCountersService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.metrika.service.MetrikaGoalsService;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.uac.grut.GrutTransactionProvider;
import ru.yandex.direct.core.entity.uac.service.UacCampaignServiceHolder;
import ru.yandex.direct.core.entity.uac.service.UacDbDefineService;
import ru.yandex.direct.core.util.CoreHttpUtil;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.core.entity.campaign.service.GridCampaignService;
import ru.yandex.direct.grid.core.entity.model.GdiOfferStats;
import ru.yandex.direct.grid.core.entity.model.campaign.GdiCampaignStats;
import ru.yandex.direct.grid.core.entity.offer.service.GridOfferService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.client.GdClient;
import ru.yandex.direct.grid.processing.model.statistics.GdCampaignStatisticsContainer;
import ru.yandex.direct.grid.processing.model.statistics.GdCampaignStatisticsGroupByDate;
import ru.yandex.direct.grid.processing.model.statistics.GdCampaignStatisticsItem;
import ru.yandex.direct.grid.processing.model.statistics.GdCampaignStatisticsPayload;
import ru.yandex.direct.grid.processing.model.statistics.GdCampaignStatisticsRegionInfo;
import ru.yandex.direct.grid.processing.service.statistics.CampaignStatisticsConverter;
import ru.yandex.direct.grid.processing.service.statistics.utils.CampaignContext;
import ru.yandex.direct.grid.processing.service.statistics.utils.StatsHolder;
import ru.yandex.direct.grid.processing.service.statistics.utils.UacCampaignContext;
import ru.yandex.direct.intapi.client.IntApiClient;
import ru.yandex.direct.intapi.client.model.request.statistics.CampaignStatisticsRequest;
import ru.yandex.direct.intapi.client.model.request.statistics.ReportOptions;
import ru.yandex.direct.intapi.client.model.request.statistics.option.ReportOptionGroupByDate;
import ru.yandex.direct.intapi.client.model.response.statistics.CampaignStatisticsResponse;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static ru.yandex.direct.common.db.PpcPropertyNames.USE_YT_STATISTICS_FOR_UC;
import static ru.yandex.direct.feature.FeatureName.UNIVERSAL_CAMPAIGNS_BETA_DISABLED;
import static ru.yandex.direct.grid.processing.service.statistics.utils.StatisticsUtils.convertRequest;
import static ru.yandex.direct.grid.processing.service.statistics.utils.StatisticsUtils.convertYtStats;
import static ru.yandex.direct.grid.processing.service.statistics.utils.StatisticsUtils.isUac;
import static ru.yandex.direct.grid.processing.service.statistics.utils.StatisticsUtils.isUcCpm;
import static ru.yandex.direct.intapi.client.model.request.statistics.option.ReportOptionGroupBy.REGION;
import static ru.yandex.direct.regions.Region.GLOBAL_REGION_ID;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@Service
public class CampaignStatisticsService {

    private static final Logger logger = LoggerFactory.getLogger(CampaignStatisticsService.class);

    private final IntApiClient intApiClient;
    private final CampaignStatisticsConverter converter;
    private final GridCampaignService gridCampaignService;
    private final GridOfferService gridOfferService;
    private final GeoTreeFactory geoTreeFactory;
    private final FeatureService featureService;
    private final MetrikaGoalsService metrikaGoalsService;
    private final CampMetrikaCountersService campMetrikaCountersService;
    private final CampaignRepository campaignRepository;
    private final ShardHelper shardHelper;
    private final NetAcl netAcl;
    private final UacDbDefineService uacDbDefineService;
    private final GrutTransactionProvider grutTransactionProvider;
    private final UacCampaignServiceHolder uacCampaignServiceHolder;

    private final PpcProperty<Boolean> useYtStatisticsForUcProperty;

    @Autowired
    public CampaignStatisticsService(IntApiClient intApiClient,
                                     CampaignStatisticsConverter converter,
                                     GridCampaignService gridCampaignService,
                                     GridOfferService gridOfferService,
                                     GeoTreeFactory geoTreeFactory,
                                     MetrikaGoalsService metrikaGoalsService,
                                     PpcPropertiesSupport ppcPropertiesSupport,
                                     FeatureService featureService,
                                     CampMetrikaCountersService campMetrikaCountersService,
                                     CampaignRepository campaignRepository,
                                     ShardHelper shardHelper,
                                     NetAcl netAcl,
                                     UacDbDefineService uacDbDefineService,
                                     GrutTransactionProvider grutTransactionProvider,
                                     UacCampaignServiceHolder uacCampaignServiceHolder) {
        this.intApiClient = intApiClient;
        this.converter = converter;
        this.gridCampaignService = gridCampaignService;
        this.gridOfferService = gridOfferService;
        this.geoTreeFactory = geoTreeFactory;
        this.metrikaGoalsService = metrikaGoalsService;
        this.featureService = featureService;
        this.campMetrikaCountersService = campMetrikaCountersService;
        this.campaignRepository = campaignRepository;
        this.shardHelper = shardHelper;
        this.netAcl = netAcl;
        this.uacDbDefineService = uacDbDefineService;
        this.grutTransactionProvider = grutTransactionProvider;
        this.uacCampaignServiceHolder = uacCampaignServiceHolder;

        this.useYtStatisticsForUcProperty = ppcPropertiesSupport.get(USE_YT_STATISTICS_FOR_UC, Duration.ofMinutes(5));
    }

    public GdCampaignStatisticsPayload getCampaignStatistics(GdCampaignStatisticsContainer input,
                                                             GdClient client,
                                                             GridGraphQLContext context) {
        CampaignSimple campaign = getCampaign(input.getFilter().getCampaignId());
        boolean expandToGeoTree = nvl(input.getExpandToGeoTree(), false);
        boolean useYtStatistics = useYtStatisticsForUcProperty.getOrDefault(false);
        boolean getFromYt = isRequestForYt(input) && useYtStatistics && !isUcCpm(campaign);

        filterGoalIdsIfNeeded(context.getOperator().getUid(), client, input, campaign, getFromYt);
        CampaignStatisticsRequest request = convertRequest(input, context);

        GeoTree clientGeoTree = geoTreeFactory.getTranslocalGeoTree(client.getInfo().getCountryRegionId());

        return getFromYt ? getCampaignStatisticsFromYt(request, campaign)
                : getCampaignStatisticsFromMoc(request, clientGeoTree, expandToGeoTree);
    }

    private boolean isRequestForYt(GdCampaignStatisticsContainer input) {
        // Пока начинаем с малого - если статистика группирована по дням и нет никаких срезов - можно взять из YT
        // Условие будет расширяться по мере выгрузки срезов в YT
        return input.getGroupByDate() != GdCampaignStatisticsGroupByDate.NONE && isEmpty(input.getGroupBy());
    }

    private CampaignSimple getCampaign(Long campaignId) {
        int shard = shardHelper.getShardByCampaignId(campaignId);
        return campaignRepository.getCampaignsSimple(shard, List.of(campaignId)).get(campaignId);
    }

    private void filterGoalIdsIfNeeded(Long operatorUid,
                                       GdClient client,
                                       GdCampaignStatisticsContainer input,
                                       CampaignSimple campaign,
                                       boolean getFromYt) {
        if (getFromYt || CollectionUtils.isEmpty(input.getFilter().getGoalIds())) {
            return;
        }

        ClientId clientId = ClientId.fromLong(client.getInfo().getId());
        boolean requestFromInternalNetwork = Optional.ofNullable(CoreHttpUtil.getRemoteAddressFromAuthOrDefault())
                .map(netAcl::isInternalIp)
                .orElse(false);
        boolean allGoalsAreAvailable = requestFromInternalNetwork
                && !featureService.isEnabledForClientId(clientId, UNIVERSAL_CAMPAIGNS_BETA_DISABLED);
        if (allGoalsAreAvailable) {
            return;
        }
        List<Long> counterIds = campMetrikaCountersService.getCounterByCampaignIds(
                clientId, Set.of(campaign.getId())).get(campaign.getId());
        if (CollectionUtils.isEmpty(counterIds)) {
            return;
        }
        Set<Long> availableGoalIds = mapSet(metrikaGoalsService.getMetrikaGoalsByCounter(
                operatorUid, clientId, counterIds, null, campaign.getType()),
                Goal::getId);
        input.getFilter().withGoalIds(filterList(input.getFilter().getGoalIds(), availableGoalIds::contains));
    }

    private GdCampaignStatisticsPayload getCampaignStatisticsFromYt(CampaignStatisticsRequest request,
                                                                    CampaignSimple campaign) {
        if (isUac(campaign)) {
            return getCampaignStatisticsFromYtForUac(request);
        }

        ClientId clientId = ClientId.fromLong(campaign.getClientId());
        ReportOptions reportOptions = request.getReportOptions();
        Long campaignId = reportOptions.getCampaignId();
        LocalDate dateFrom = reportOptions.getDateFrom();
        LocalDate dateTo = reportOptions.getDateTo();
        ReportOptionGroupByDate groupByDate = reportOptions.getGroupByDate();

        boolean getRevenueOnlyByAvailableGoals =
                featureService.isEnabled(request.getUid(), FeatureName.GET_REVENUE_ONLY_BY_AVAILABLE_GOALS);
        boolean getCampaignStatsOnlyByAvailableGoals =
                featureService.isEnabled(request.getUid(), FeatureName.GRID_CAMPAIGN_GOALS_FILTRATION_FOR_STAT);
        Set<Long> availableGoalIds = null;
        if (getRevenueOnlyByAvailableGoals || getCampaignStatsOnlyByAvailableGoals) {
            availableGoalIds = mapSet(metrikaGoalsService.getAvailableMetrikaGoalsForClient(
                    request.getOperatorUid(), clientId),
                    Goal::getId);
        }
        Map<LocalDate, GdiCampaignStats> stats = gridCampaignService.getCampaignStatsForCampaignGoalsByDate(
                List.of(campaignId), dateFrom, dateTo, availableGoalIds, groupByDate,
                getRevenueOnlyByAvailableGoals, getCampaignStatsOnlyByAvailableGoals).get(campaignId);
        Map<LocalDate, GdiOfferStats> offerStats = gridOfferService.getOfferStatsByDateByCampaignId(
                clientId, List.of(campaignId), dateFrom, dateTo, groupByDate
        ).getOrDefault(campaignId, Map.of());

        return convertYtStats(stats, offerStats, new CampaignContext(true), groupByDate);
    }

    private GdCampaignStatisticsPayload getCampaignStatisticsFromYtForUac(CampaignStatisticsRequest request) {
        ClientId clientId = ClientId.fromLong(shardHelper.getClientIdByUid(request.getUid()));
        ReportOptions reportOptions = request.getReportOptions();
        Long campaignId = reportOptions.getCampaignId();
        LocalDate dateFrom = reportOptions.getDateFrom();
        LocalDate dateTo = reportOptions.getDateTo();
        ReportOptionGroupByDate groupByDate = reportOptions.getGroupByDate();

        var useGrut = uacDbDefineService.useGrutForDirectCampaignId(campaignId);
        var uacCampaign = grutTransactionProvider.runInTransactionIfNeeded(useGrut, () -> {
            var uacCampaignService = uacCampaignServiceHolder.getUacCampaignService(useGrut);
            var uacCampaignId = uacCampaignService.getCampaignIdByDirectCampaignId(campaignId);
            checkState(uacCampaignId != null);
            return uacCampaignService.getCampaignById(uacCampaignId);
        });
        UacCampaignContext rmpCampaignContext = new UacCampaignContext(uacCampaign);
        boolean getRevenueOnlyByAvailableGoals =
                featureService.isEnabled(request.getUid(), FeatureName.GET_REVENUE_ONLY_BY_AVAILABLE_GOALS);
        boolean getCampaignStatsOnlyByAvailableGoals =
                featureService.isEnabled(request.getUid(), FeatureName.GRID_CAMPAIGN_GOALS_FILTRATION_FOR_STAT);
        boolean isUacWithOnlyInstallEnabled =
                featureService.isEnabled(request.getUid(), FeatureName.ENABLE_UAC_WITH_ONLY_INSTALL);

        Set<Long> goalIds = rmpCampaignContext.getGoalIds();
        Map<LocalDate, GdiCampaignStats> stats = gridCampaignService.getCampaignStatsGroupByDate(
                // goalIdsForRevenue - null, т.к. в goalIds только мобильные цели и они все разрешены
                List.of(campaignId), dateFrom, dateTo, goalIds, null, groupByDate,
                getRevenueOnlyByAvailableGoals, getCampaignStatsOnlyByAvailableGoals, isUacWithOnlyInstallEnabled
        ).get(campaignId);
        Map<LocalDate, GdiOfferStats> offerStats = gridOfferService.getOfferStatsByDateByCampaignId(
                clientId, List.of(campaignId), dateFrom, dateTo, groupByDate
        ).getOrDefault(campaignId, Map.of());

        return convertYtStats(stats, offerStats, new CampaignContext(rmpCampaignContext), groupByDate);
    }

    private GdCampaignStatisticsPayload getCampaignStatisticsFromMoc(CampaignStatisticsRequest request,
                                                                     GeoTree clientGeoTree,
                                                                     boolean expandToGeoTree) {
        checkState(!expandToGeoTree || Objects.equals(request.getReportOptions().getGroupBy(), Set.of(REGION)));

        CampaignStatisticsResponse response = intApiClient.getCampaignStatistics(request);
        ReportOptionGroupByDate period = request.getReportOptions().getGroupByDate();

        if (expandToGeoTree) {
            return getGeoTreeStatistics(response, clientGeoTree, period);
        }

        return new GdCampaignStatisticsPayload()
                .withPeriod(GdCampaignStatisticsGroupByDate.fromSource(period))
                .withRowset(converter.convert(response.getData()))
                .withTotals(converter.convertColumnValues(response.getTotals()));
    }

    private GdCampaignStatisticsPayload getGeoTreeStatistics(CampaignStatisticsResponse response,
                                                             GeoTree clientGeoTree,
                                                             ReportOptionGroupByDate period) {
        List<GdCampaignStatisticsItem> rowset = converter.convert(response.getData());

        Set<Long> regionIdsWithStat = StreamEx.of(rowset)
                .map(row -> row.getRegion().longValue())
                .toSet();

        // Геодерево Директа и БК расходятся
        // https://st.yandex-team.ru/DIRECT-89425#5c86623a61038d00204333c6
        Set<Long> regionIdsAbsentInDirect = StreamEx.of(regionIdsWithStat)
                .remove(clientGeoTree::hasRegion)
                .toSet();
        if (!regionIdsAbsentInDirect.isEmpty()) {
            logger.warn("Found regions that are absent in Direct: {}", regionIdsAbsentInDirect);
            regionIdsWithStat = StreamEx.of(regionIdsWithStat)
                    .remove(regionIdsAbsentInDirect::contains)
                    .toSet();
        }

        Map<Long, StatsHolder> statisticsItemByRegionId = listToMap(rowset, item -> item.getRegion().longValue(),
                StatsHolder::new);

        Map<Long, List<Long>> geoSubTreeGraph = buildGeoSubTreeGraph(regionIdsWithStat, clientGeoTree);
        List<Long> rootRegionIds = getRootRegionIds(regionIdsWithStat, geoSubTreeGraph);

        fillGeoSubTreeWithStatistics(GLOBAL_REGION_ID, statisticsItemByRegionId, geoSubTreeGraph);

        Map<Long, GdCampaignStatisticsRegionInfo> geoTreeStatistics = StreamEx.of(rootRegionIds)
                .mapToEntry(regionId -> getRegionInfo(regionId, statisticsItemByRegionId, geoSubTreeGraph))
                .toMap();

        return new GdCampaignStatisticsPayload()
                .withPeriod(GdCampaignStatisticsGroupByDate.fromSource(period))
                .withRowset(emptyList())
                .withTotals(converter.convertColumnValues(response.getTotals()))
                .withGeoTreeStatistics(geoTreeStatistics);
    }

    private Map<Long, List<Long>> buildGeoSubTreeGraph(Set<Long> childRegionIds, GeoTree clientGeoTree) {
        Map<Long, List<Long>> graph = new HashMap<>();
        Set<Long> visited = new HashSet<>();

        childRegionIds.forEach(childRegionId -> {
            Long currentRegionId = childRegionId;
            visited.add(GLOBAL_REGION_ID);

            while (!visited.contains(currentRegionId)) {
                Long parentRegionId = getParentRegionId(currentRegionId, clientGeoTree);

                visited.add(currentRegionId);
                List<Long> children = graph.computeIfAbsent(parentRegionId, regionId -> new ArrayList<>());
                children.add(currentRegionId);

                currentRegionId = parentRegionId;
            }
        });

        return graph;
    }

    private Long getParentRegionId(Long regionId, GeoTree clientGeoTree) {
        return clientGeoTree.getRegion(regionId).getParent().getId();
    }

    private List<Long> getRootRegionIds(Set<Long> regionIdsWithStat, Map<Long, List<Long>> geoSubTreeGraph) {
        if (regionIdsWithStat.isEmpty()) {
            return emptyList();
        }

        Long currentRegionId = GLOBAL_REGION_ID;

        List<Long> currentChildRegionIds;
        while ((currentChildRegionIds = geoSubTreeGraph.getOrDefault(currentRegionId, List.of())).size() == 1) {
            currentRegionId = currentChildRegionIds.get(0);
        }

        return regionIdsWithStat.contains(currentRegionId) ? List.of(currentRegionId) :
                geoSubTreeGraph.get(currentRegionId);
    }

    private void fillGeoSubTreeWithStatistics(long currentRegionId,
                                              Map<Long, StatsHolder> statisticsItemByRegionId,
                                              Map<Long, List<Long>> geoSubTreeGraph) {
        List<Long> childRegionIds = geoSubTreeGraph.get(currentRegionId);
        if (isEmpty(childRegionIds)) {
            return;
        }

        StatsHolder currentRegionStatsHolder = statisticsItemByRegionId.computeIfAbsent(currentRegionId,
                i -> new StatsHolder());

        childRegionIds.forEach(childRegionId -> fillGeoSubTreeWithStatistics(childRegionId, statisticsItemByRegionId,
                geoSubTreeGraph));
        childRegionIds.forEach(childRegionId -> currentRegionStatsHolder.addStats(statisticsItemByRegionId.get(childRegionId)));
    }

    private GdCampaignStatisticsRegionInfo getRegionInfo(Long regionId,
                                                         Map<Long, StatsHolder> statisticsItemByRegionId,
                                                         Map<Long, List<Long>> geoSubTreeGraph) {
        List<Long> childRegionIds = geoSubTreeGraph.get(regionId);
        Map<Long, GdCampaignStatisticsRegionInfo> childRegionInfos = isEmpty(childRegionIds) ? emptyMap() :
                StreamEx.of(childRegionIds)
                        .mapToEntry(childRegionId -> getRegionInfo(childRegionId, statisticsItemByRegionId,
                                geoSubTreeGraph))
                        .toMap();

        return new GdCampaignStatisticsRegionInfo()
                .withStatistics(statisticsItemByRegionId.get(regionId).buildColumnValues())
                .withChildRegions(childRegionInfos);
    }
}
