package ru.yandex.direct.grid.processing.service.statistics.service

import org.springframework.stereotype.Service
import ru.yandex.direct.bannersystem.container.masterreport.MasterReportDimension
import ru.yandex.direct.bannersystem.container.masterreport.MasterReportGroupByDate
import ru.yandex.direct.bannersystem.container.masterreport.MasterReportMetric
import ru.yandex.direct.bannersystem.container.masterreport.MasterReportRow
import ru.yandex.direct.core.entity.campaign.model.Campaign
import ru.yandex.direct.core.entity.campaign.model.CampaignAttributionModel
import ru.yandex.direct.core.entity.campaign.service.CampaignService
import ru.yandex.direct.core.entity.masterreport.MasterReportService
import ru.yandex.direct.core.entity.masterreport.model.MasterReportFilters
import ru.yandex.direct.core.entity.masterreport.model.MasterReportPeriod
import ru.yandex.direct.core.entity.metrika.service.MetrikaGoalsService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext
import ru.yandex.direct.grid.processing.model.masterreport.GdMasterReportStatisticsColumnValues
import ru.yandex.direct.grid.processing.model.masterreport.GdMasterReportStatisticsContainer
import ru.yandex.direct.grid.processing.model.masterreport.GdMasterReportStatisticsGroupBy
import ru.yandex.direct.grid.processing.model.masterreport.GdMasterReportStatisticsGroupByDate
import ru.yandex.direct.grid.processing.model.masterreport.GdMasterReportStatisticsPayload
import ru.yandex.direct.grid.processing.service.cache.util.CacheUtils.normalizeLimitOffset
import ru.yandex.direct.grid.processing.service.statistics.converter.MasterReportStatisticsConverter.buildComparePeriod
import ru.yandex.direct.grid.processing.service.statistics.converter.MasterReportStatisticsConverter.buildPeriod
import ru.yandex.direct.grid.processing.service.statistics.converter.MasterReportStatisticsConverter.convertAttributionModel
import ru.yandex.direct.grid.processing.service.statistics.converter.MasterReportStatisticsConverter.convertFilters
import ru.yandex.direct.grid.processing.service.statistics.converter.MasterReportStatisticsConverter.convertGroupByDate
import ru.yandex.direct.grid.processing.service.statistics.converter.MasterReportStatisticsConverter.toGdItem
import ru.yandex.direct.grid.processing.service.statistics.converter.MasterReportStatisticsConverter.toGdItemValues
import ru.yandex.direct.grid.processing.service.statistics.validation.MasterReportStatisticsValidationService
import ru.yandex.direct.multitype.entity.LimitOffset

@JvmField
val MULTI_GOALS_METRICS = setOf(
        MasterReportMetric.MULTI_GOALS,
        MasterReportMetric.COST,
        MasterReportMetric.INCOME,
        MasterReportMetric.PROFIT,
        MasterReportMetric.CRR,
        MasterReportMetric.ROI,
        MasterReportMetric.CONVERSIONS,
        MasterReportMetric.COST_PER_CONVERSION
)

@JvmField
val SIMPLE_METRICS = setOf(*MasterReportMetric.values()) - setOf(MasterReportMetric.MULTI_GOALS)

@JvmField
val ALL_METRICS = setOf(*MasterReportMetric.values())

@Service
class MasterReportStatisticsService(
        private val campaignService: CampaignService,
        private val masterReportService: MasterReportService,
        private val metrikaGoalsService: MetrikaGoalsService,
        private val masterReportStatisticsValidationService: MasterReportStatisticsValidationService
) {
    fun getStatistics(
            input: GdMasterReportStatisticsContainer,
            context: GridGraphQLContext
    ): GdMasterReportStatisticsPayload {
        masterReportStatisticsValidationService.validateGdStatisticsContainer(input)

        val subjectUser = context.subjectUser!!
        val clientId = subjectUser.clientId
        val operatorUid = context.operator.uid

        val goalById =
            metrikaGoalsService.getAvailableMetrikaGoalsForClient(operatorUid, clientId)
                ?.associate { it.id to it }
                ?: emptyMap()
        if (input.groupBy == GdMasterReportStatisticsGroupBy.GOAL && goalById.isEmpty()) {
            return GdMasterReportStatisticsPayload()
                    .withTotals(GdMasterReportStatisticsColumnValues())
                    .withRowset(emptyList())
        }
        val filterGoalIds = selectGoalIdsOrNull(goalById.keys, input.filter.goalIds)
                ?: return GdMasterReportStatisticsPayload()
                        .withTotals(GdMasterReportStatisticsColumnValues())
                        .withRowset(emptyList())
        val filters = convertFilters(input.filter, filterGoalIds)
        val metrics = when (input.groupBy) {
            null, GdMasterReportStatisticsGroupBy.CAMPAIGN, GdMasterReportStatisticsGroupBy.PLATFORM -> ALL_METRICS
            GdMasterReportStatisticsGroupBy.GOAL -> MULTI_GOALS_METRICS
            else -> SIMPLE_METRICS
        }

        val period = buildPeriod(input, context.instant)

        val groupByDate = convertGroupByDate(input.groupByDate)
        val attrModel = convertAttributionModel(input.attributionModel)
        val limitOffset = normalizeLimitOffset(input.limitOffset)

        val dimensions = when (input.groupBy) {
            null -> getTimeDimensionOrNull(input.groupByDate)
            GdMasterReportStatisticsGroupBy.CAMPAIGN -> setOf(MasterReportDimension.CAMPAIGN)
            GdMasterReportStatisticsGroupBy.PLATFORM -> setOf(MasterReportDimension.PLATFORM)
            GdMasterReportStatisticsGroupBy.GOAL -> emptySet()
            else -> throw UnsupportedOperationException("Unsupported group by")
        }
        val chiefUid = subjectUser.chiefUid
        val stat = masterReportService.getStatistics(
                clientId,
                chiefUid,
                period,
                groupByDate,
                attrModel,
                filters,
                dimensions,
                metrics,
                limitOffset
        )
        // Для некоторых срезов не все нужные данные можно получить с помощью одного запроса в БК
        val extRows = when (input.groupBy) {
            GdMasterReportStatisticsGroupBy.GOAL -> {
                masterReportService.getStatistics(
                        clientId,
                        chiefUid,
                        period,
                        groupByDate,
                        attrModel,
                        filters,
                        setOf(MasterReportDimension.CAMPAIGN),
                        metrics,
                        limitOffset
                ).data
            }
            GdMasterReportStatisticsGroupBy.PLATFORM -> {
                // Для группировки по площадкам также нужен список кампаний для этих площадок
                // Дополнительный запрос нужен, т.к. при одном запросе по срезу CAMPAIGN+PLATFORM
                // мы получаем плоский список от БК -- в нем нет данных по конкретной площадке без указания кампаний
                masterReportService.getStatistics(
                        clientId,
                        chiefUid,
                        period,
                        groupByDate,
                        attrModel,
                        filters,
                        setOf(MasterReportDimension.CAMPAIGN, MasterReportDimension.PLATFORM),
                        metrics,
                        limitOffset
                ).data
            }
            else -> emptyList()
        }

        val comparePeriod = buildComparePeriod(input, period)
        val extTotals = getCompareTotals(
                clientId,
                chiefUid,
                comparePeriod,
                groupByDate,
                attrModel,
                filters,
                dimensions,
                metrics,
                limitOffset
        )

        val rows = stat.data + extRows
        val campaignById = getCampaignById(clientId, rows)

        val gdRows = rows.map {
            val campaign = it.campaignId?.let { cid -> campaignById[cid] }
            toGdItem(it, campaign, goalById)
        }
        return GdMasterReportStatisticsPayload().apply {
            totals = toGdItemValues(stat.totals, extTotals, goalById)
            rowset = gdRows
        }
    }

    private fun getCampaignById(clientId: ClientId, data: List<MasterReportRow>): Map<Long, Campaign> {
        val campaignIds = data.asSequence().mapNotNull { it.campaignId }.toSet()
        return campaignService.getCampaignsMap(clientId, campaignIds)
    }

    /**
     * Если запрашивается группировка по часам, то в БК это нужно передавать как срез, а не groupByDate
     * Других срезов не будет передано -- это проверяется в валидации
     */
    private fun getTimeDimensionOrNull(groupByDate: GdMasterReportStatisticsGroupByDate?): Set<MasterReportDimension> {
        return if (groupByDate != GdMasterReportStatisticsGroupByDate.HOUR) {
            emptySet()
        } else {
            setOf(MasterReportDimension.EVENT_TIME)
        }
    }

    /**
     * Возвращает список целей, которые нужно использовать для фильтрации при отправке запроса в БК.
     *
     * Если `null`, значит нет подходящих целей для запроса (например, все цели принадлежат недоступным счетчикам)
     */
    private fun selectGoalIdsOrNull(
            goalIds: Set<Long>,
            filterGoalIds: Set<Long>?
    ): Set<Long>? {
        if (filterGoalIds.isNullOrEmpty()) {
            return goalIds
        }
        val intersect = filterGoalIds.intersect(goalIds)
        if (intersect.isEmpty()) {
            return null
        }
        return intersect
    }

    /**
     * При сравнении из коробки ручка работает сильно дольше.
     * Подробности в [https://st.yandex-team.ru/DIRECT-159468#61c9bbb56888516a7ac94047]
     *
     * Поэтому, если требуется сравнение периодов, выгружаем данные (нужны только totals)
     * со второго периода дополнительным запросом.
     * При формировании ответа эти значения будут учтены при подсчете абсолютных дельт
     */
    private fun getCompareTotals(
            clientId: ClientId,
            chiefUid: Long,
            comparePeriod: MasterReportPeriod?,
            groupByDate: MasterReportGroupByDate,
            attrModel: CampaignAttributionModel,
            filters: MasterReportFilters,
            dimensions: Set<MasterReportDimension>,
            metrics: Set<MasterReportMetric>,
            limitOffset: LimitOffset
    ): MasterReportRow? {
        if (comparePeriod == null) {
            return null
        }
        val extStat = masterReportService.getStatistics(
                clientId,
                chiefUid,
                comparePeriod,
                groupByDate,
                attrModel,
                filters,
                dimensions,
                metrics,
                limitOffset
        )
        return if (extStat.data.isEmpty()) {
            // Если данных нет, то нужно вернуть totals как все null-ы, чтобы верно посчитать абсолют дельты
            // (от БК в таком случае приходит 0 и null вперемешку)
            MasterReportRow()
        } else {
            extStat.totals
        }
    }

}
