package ru.yandex.direct.grid.processing.service.statistics.utils;

import java.time.Instant;
import java.time.LocalDate;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.direct.common.util.HttpUtil;
import ru.yandex.direct.core.entity.campaign.AvailableCampaignSources;
import ru.yandex.direct.core.entity.campaign.model.CampaignSimple;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.grid.core.entity.model.GdiEntityStats;
import ru.yandex.direct.grid.core.entity.model.GdiOfferStats;
import ru.yandex.direct.grid.core.entity.model.campaign.GdiCampaignStats;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.GdLimitOffset;
import ru.yandex.direct.grid.processing.model.statistics.GdCampaignStatisticsColumn;
import ru.yandex.direct.grid.processing.model.statistics.GdCampaignStatisticsConditions;
import ru.yandex.direct.grid.processing.model.statistics.GdCampaignStatisticsContainer;
import ru.yandex.direct.grid.processing.model.statistics.GdCampaignStatisticsFilter;
import ru.yandex.direct.grid.processing.model.statistics.GdCampaignStatisticsFilterColumn;
import ru.yandex.direct.grid.processing.model.statistics.GdCampaignStatisticsGroupBy;
import ru.yandex.direct.grid.processing.model.statistics.GdCampaignStatisticsGroupByDate;
import ru.yandex.direct.grid.processing.model.statistics.GdCampaignStatisticsItem;
import ru.yandex.direct.grid.processing.model.statistics.GdCampaignStatisticsOrder;
import ru.yandex.direct.grid.processing.model.statistics.GdCampaignStatisticsOrderBy;
import ru.yandex.direct.grid.processing.model.statistics.GdCampaignStatisticsOrderByField;
import ru.yandex.direct.grid.processing.model.statistics.GdCampaignStatisticsPayload;
import ru.yandex.direct.grid.processing.model.statistics.GdCampaignStatisticsPeriod;
import ru.yandex.direct.grid.processing.model.statistics.GdCampaignStatisticsValueHolder;
import ru.yandex.direct.grid.processing.model.statistics.GdConditionOperator;
import ru.yandex.direct.grid.processing.model.statistics.GdFilterCondition;
import ru.yandex.direct.intapi.client.model.request.statistics.CampaignStatisticsRequest;
import ru.yandex.direct.intapi.client.model.request.statistics.ReportOptions;
import ru.yandex.direct.intapi.client.model.request.statistics.ReportType;
import ru.yandex.direct.intapi.client.model.request.statistics.option.ReportOptionConditionOperator;
import ru.yandex.direct.intapi.client.model.request.statistics.option.ReportOptionFilterColumn;
import ru.yandex.direct.intapi.client.model.request.statistics.option.ReportOptionGroupByDate;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.singletonMap;
import static ru.yandex.direct.grid.processing.model.statistics.GdCampaignStatisticsGroupBy.SEARCH_QUERY;
import static ru.yandex.direct.grid.processing.service.goal.GoalConstant.MAX_GOAL_COUNT_FOR_CAMPAIGN_STATISTIC_REQUEST;
import static ru.yandex.direct.grid.processing.util.StatHelper.getDayPeriod;
import static ru.yandex.direct.intapi.client.model.request.statistics.ReportType.MOC;
import static ru.yandex.direct.intapi.client.model.request.statistics.ReportType.SEARCH_QUERIES;
import static ru.yandex.direct.intapi.client.model.request.statistics.option.ReportOptionConditionOperator.EQ;
import static ru.yandex.direct.intapi.client.model.request.statistics.option.ReportOptionFilterColumn.GOAL_IDS;
import static ru.yandex.direct.intapi.client.model.request.statistics.option.ReportOptionFilterColumn.KEYWORD_IDS;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

public class StatisticsUtils {

    public static CampaignStatisticsRequest convertRequest(GdCampaignStatisticsContainer container,
                                                           GridGraphQLContext context) {
        Long clientUid = context.getSubjectUser().getUid();
        Long operatorUid = context.getOperator().getUid();

        return new CampaignStatisticsRequest()
                .withUid(clientUid)
                .withOperatorUid(operatorUid)
                .withReportOptions(convertToReportOptions(container, context))
                .withLocale(HttpUtil.getCurrentLocale().map(Locale::getLanguage).orElse(null));
    }

    private static ReportOptions convertToReportOptions(GdCampaignStatisticsContainer container,
                                                        GridGraphQLContext context) {
        Instant requestInstant = context.getInstant();
        GdCampaignStatisticsFilter filter = container.getFilter();

        normalizePeriod(filter.getPeriod(), requestInstant);
        normalizePeriod(filter.getPeriodToCompare(), requestInstant);

        var reportOptions = new ReportOptions()
                .withCampaignId(filter.getCampaignId())
                .withFilters(convertFilters(filter))
                .withColumns(mapSet(container.getColumns(), GdCampaignStatisticsColumn::toSource))
                .withGroupBy(mapSet(container.getGroupBy(), GdCampaignStatisticsGroupBy::toSource))
                .withGroupByDate(GdCampaignStatisticsGroupByDate.toSource(container.getGroupByDate()))
                .withDateFrom(filter.getPeriod().getFrom())
                .withDateTo(filter.getPeriod().getTo())
                .withReportType(getReportType(container.getGroupBy()));

        addLimitOffset(reportOptions, container.getLimitOffset());
        addSorting(reportOptions, container.getOrderBy());

        boolean comparePeriods = filter.getPeriodToCompare() != null;
        if (comparePeriods) {
            reportOptions
                    .comparePeriods()
                    .withDateFromB(filter.getPeriodToCompare().getFrom())
                    .withDateToB(filter.getPeriodToCompare().getTo());
        }

        return reportOptions;
    }

    public static void normalizePeriod(GdCampaignStatisticsPeriod period, Instant instant) {
        if (period == null || period.getPreset() == null) {
            return;
        }

        checkState(period.getFrom() == null && period.getTo() == null, "Either preset or dates must be set");

        Pair<LocalDate, LocalDate> fromAndTo = getDayPeriod(period.getPreset(), instant, null);

        period.withFrom(fromAndTo.getLeft())
                .withTo(fromAndTo.getRight())
                .withPreset(null);
    }

    private static ReportType getReportType(Set<GdCampaignStatisticsGroupBy> groupBy) {
        if (groupBy.contains(SEARCH_QUERY)) {
            return SEARCH_QUERIES;
        }

        return MOC;
    }

    private static void addLimitOffset(ReportOptions reportOptions, GdLimitOffset limitOffset) {
        if (limitOffset == null) {
            return;
        }

        reportOptions
                .withLimit(limitOffset.getLimit())
                .withOffset(limitOffset.getOffset());
    }

    private static void addSorting(ReportOptions reportOptions, GdCampaignStatisticsOrderBy orderBy) {
        if (orderBy == null) {
            return;
        }

        reportOptions
                .withOrderByField(GdCampaignStatisticsOrderByField.toSource(orderBy.getField()))
                .withOrder(GdCampaignStatisticsOrder.toSource(orderBy.getOrder()));
    }

    private static Map<ReportOptionFilterColumn, Map<ReportOptionConditionOperator, ?>> convertFilters(GdCampaignStatisticsFilter gdFilter) {
        var reportFilters = new HashMap<ReportOptionFilterColumn, Map<ReportOptionConditionOperator, ?>>();

        addGoalsToFilters(reportFilters, gdFilter);
        addKeywordsToFilters(reportFilters, gdFilter);
        addConditionsToFilters(reportFilters, gdFilter.getConditions());

        return reportFilters;
    }

    @Deprecated
    private static void addGoalsToFilters(Map<ReportOptionFilterColumn, Map<ReportOptionConditionOperator, ?>> reportFilters,
                                          GdCampaignStatisticsFilter gdFilter) {
        List<Long> goalIds = gdFilter.getGoalIds();

        if (!isEmpty(goalIds)) {
            // При запросе статистики из МОКа есть ограничение в максимум 50 целей
            List<Long> restrictedGoalIds = goalIds.size() > MAX_GOAL_COUNT_FOR_CAMPAIGN_STATISTIC_REQUEST
                    ? goalIds.subList(0, MAX_GOAL_COUNT_FOR_CAMPAIGN_STATISTIC_REQUEST) : goalIds;
            reportFilters.put(GOAL_IDS, singletonMap(EQ, restrictedGoalIds));
        }
    }

    @Deprecated
    private static void addKeywordsToFilters(Map<ReportOptionFilterColumn, Map<ReportOptionConditionOperator, ?>> reportFilters,
                                             GdCampaignStatisticsFilter gdFilter) {
        List<Long> keywordIds = gdFilter.getKeywordIds();

        if (!isEmpty(keywordIds)) {
            reportFilters.put(KEYWORD_IDS, singletonMap(EQ, keywordIds));
        }
    }

    private static void addConditionsToFilters(Map<ReportOptionFilterColumn, Map<ReportOptionConditionOperator, ?>> reportFilters,
                                               GdCampaignStatisticsConditions conditions) {
        if (conditions == null) {
            return;
        }

        if (conditions.getLongFieldConditions() != null) {
            conditions.getLongFieldConditions().forEach(condition ->
                    addConditionToFilters(reportFilters, condition, condition.getValue()));
        }
        if (conditions.getListLongFieldConditions() != null) {
            conditions.getListLongFieldConditions().forEach(condition ->
                    addConditionToFilters(reportFilters, condition, condition.getValue()));
        }
    }

    private static <T> void addConditionToFilters(Map<ReportOptionFilterColumn,
                                                  Map<ReportOptionConditionOperator, ?>> reportFilters,
                                                  GdFilterCondition filterCondition, T value) {
        reportFilters.put(GdCampaignStatisticsFilterColumn.toSource(filterCondition.getField()),
                getCondition(filterCondition.getOperator(), value));
    }

    private static <T> Map<ReportOptionConditionOperator, T> getCondition(GdConditionOperator conditionOperator,
                                                                          T value) {
        return singletonMap(GdConditionOperator.toSource(conditionOperator), value);
    }

    public static GdCampaignStatisticsValueHolder wrapToHolder(Number number) {
        return new GdCampaignStatisticsValueHolder().withValue(number);
    }

    public static GdCampaignStatisticsPayload convertYtStats(Map<LocalDate, GdiCampaignStats> stats,
                                                             Map<LocalDate, GdiOfferStats> offerStats,
                                                             CampaignContext campaignContext,
                                                             ReportOptionGroupByDate period) {
        StatsHolder totalStatsHolder = new StatsHolder();

        List<GdCampaignStatisticsItem> statRowset = toCampaignStatisticsItems(stats, offerStats, campaignContext, totalStatsHolder);

        return new GdCampaignStatisticsPayload()
                .withPeriod(GdCampaignStatisticsGroupByDate.fromSource(period))
                .withRowset(statRowset)
                .withTotals(totalStatsHolder.buildColumnValues());
    }

    public static List<GdCampaignStatisticsItem> getYtStatsForUac(Map<LocalDate, GdiCampaignStats> stats,
                                                                  Map<LocalDate, GdiOfferStats> offerStats,
                                                                  CampaignContext campaignContext) {
        return toCampaignStatisticsItems(stats, offerStats, campaignContext, null);
    }

    public static List<GdCampaignStatisticsItem> toCampaignStatisticsItems(Map<LocalDate, GdiCampaignStats> stats,
                                                                           Map<LocalDate, GdiOfferStats> offerStats,
                                                                           CampaignContext campaignContext,
                                                                           @Nullable StatsHolder totalStatsHolder) {
        return StreamEx.of(stats.keySet()).append(offerStats.keySet())
                .distinct()
                .mapToEntry(date -> convertToHolder(stats.get(date), offerStats.get(date), campaignContext))
                .peekValues(dailyStatsHolder -> {
                    if (totalStatsHolder != null) {
                        updateTotalStats(totalStatsHolder, dailyStatsHolder);
                    }
                })
                .mapKeyValue(StatisticsUtils::convertToStatisticsItem)
                .sortedBy(GdCampaignStatisticsItem::getDate)
                .toList();
    }

    private static StatsHolder convertToHolder(@Nullable GdiCampaignStats gdiCampaignStats,
                                               @Nullable GdiOfferStats gdiOfferStats,
                                               CampaignContext campaignContext) {
        return new StatsHolder(nvl(gdiCampaignStats, () -> new GdiCampaignStats()
                .withStat(new GdiEntityStats())
                .withGoalStats(List.of())), nvl(gdiOfferStats, GdiOfferStats::new), campaignContext);
    }

    private static void updateTotalStats(StatsHolder totalStatHolder,
                                         StatsHolder dailyStatHolder) {
        totalStatHolder.addStats(dailyStatHolder);
    }

    private static GdCampaignStatisticsItem convertToStatisticsItem(LocalDate date,
                                                                    StatsHolder dailyStatsHolder) {
        return new GdCampaignStatisticsItem()
                .withDate(date)
                .withColumnValues(dailyStatsHolder.buildColumnValues());
    }

    public static boolean isUcCpm(CampaignSimple campaign) {
        return campaign != null
                && campaign.getType() == CampaignType.CPM_BANNER && AvailableCampaignSources.INSTANCE.isUC(campaign.getSource());
    }

    public static boolean isUac(CampaignSimple campaign) {
        return campaign != null
                && campaign.getType() == CampaignType.MOBILE_CONTENT && AvailableCampaignSources.INSTANCE.isUC(campaign.getSource());
    }
}
