package ru.yandex.direct.grid.processing.service.statistics.utils;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;

import ru.yandex.direct.grid.core.entity.model.GdiGoalStats;
import ru.yandex.direct.grid.core.entity.model.GdiOfferStats;
import ru.yandex.direct.grid.core.entity.model.campaign.GdiCampaignStats;
import ru.yandex.direct.grid.processing.model.statistics.GdCampaignStatisticsColumnValues;
import ru.yandex.direct.grid.processing.model.statistics.GdCampaignStatisticsItem;

import static java.lang.Math.max;
import static java.util.Collections.emptyList;
import static ru.yandex.direct.grid.processing.service.statistics.utils.StatisticsUtils.wrapToHolder;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.ifNotNullOrDefault;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.ytwrapper.dynamic.dsl.YtMappingUtils.fromMicros;

/**
 * Контейнер с данными статистики для промежуточного хранения
 */
public class StatsHolder {

    private Long shows;
    private Long clicks;
    private Long costMicros;
    private Long revenue;
    private Long conversions;

    // Поля ТК
    private Long carts;
    private Long purchases;

    // РМП поля
    private Long installs;
    private Long postViewConversions;
    private Long postViewInstalls;

    public StatsHolder() {
        shows = clicks = conversions = costMicros = revenue = 0L;
    }

    StatsHolder(GdiCampaignStats campaignStats, GdiOfferStats offerStats, CampaignContext campaignContext) {
        var commonStats = campaignStats.getStat();
        var goalStats = campaignStats.getGoalStats();

        this.shows = ifNotNullOrDefault(commonStats.getShows(), BigDecimal::longValue, 0L);
        this.clicks = ifNotNullOrDefault(commonStats.getClicks(), BigDecimal::longValue, 0L);
        this.costMicros = ifNotNullOrDefault(commonStats.getCostMicros(), BigDecimal::longValue, 0L);
        this.revenue = ifNotNullOrDefault(commonStats.getRevenue(), BigDecimal::longValue, 0L);

        this.carts = ifNotNull(offerStats.getCarts(), BigDecimal::longValue);
        this.purchases = ifNotNull(offerStats.getPurchases(), BigDecimal::longValue);

        if (campaignContext.getUacCampaignContext() != null) {
            var uacContext = campaignContext.getUacCampaignContext();
            boolean withShows = uacContext.hasImpressionUrl();

            Map<Long, List<GdiGoalStats>> goalStatsByGoalId = StreamEx.of(goalStats)
                    .groupingBy(GdiGoalStats::getGoalId);
            long conversionsClick = StreamEx.of(goalStatsByGoalId.getOrDefault(uacContext.getGoalClickId(), emptyList()))
                    .map(GdiGoalStats::getGoals)
                    .reduce(Long::sum)
                    .orElse(0L);
            long conversionsView = StreamEx.of(goalStatsByGoalId.getOrDefault(uacContext.getGoalViewId(), emptyList()))
                    .map(withShows ? GdiGoalStats::getGoalsWithShows : GdiGoalStats::getGoals)
                    .reduce(Long::sum)
                    .orElse(0L);
            long installsClick = StreamEx.of(goalStatsByGoalId.getOrDefault(uacContext.getInstallsGoalClickId(), emptyList()))
                    .map(GdiGoalStats::getGoals)
                    .reduce(Long::sum)
                    .orElse(0L);
            long installsView = StreamEx.of(goalStatsByGoalId.getOrDefault(uacContext.getInstallsGoalViewId(), emptyList()))
                    .map(withShows ? GdiGoalStats::getGoalsWithShows : GdiGoalStats::getGoals)
                    .reduce(Long::sum)
                    .orElse(0L);

            if (withShows) {
                this.conversions = max(conversionsClick, conversionsView);
                this.installs = max(installsClick, installsView);
            } else {
                this.conversions = conversionsClick;
                this.installs = installsClick;
            }

            this.postViewConversions  = conversions - conversionsClick;
            this.postViewInstalls = installs - installsClick;
        } else {
            if (campaignContext.isUseCommonStatsForConversions()) {
                this.conversions = Optional.ofNullable(commonStats.getGoals())
                        .map(BigDecimal::longValue)
                        .orElse(0L);
            } else {
                this.conversions = StreamEx.of(goalStats)
                        .map(GdiGoalStats::getGoals)
                        .reduce(Long::sum)
                        .orElse(0L);
            }
        }
    }

    public StatsHolder(GdCampaignStatisticsItem campaignStats) {
        var values = campaignStats.getColumnValues();

        this.shows = Optional.ofNullable(values.getShows().getValue()).map(Number::longValue).orElse(0L);
        this.clicks = Optional.ofNullable(values.getClicks().getValue()).map(Number::longValue).orElse(0L);
        this.conversions = Optional.ofNullable(values.getConversions().getValue()).map(Number::longValue).orElse(0L);
        this.costMicros = Optional.ofNullable(values.getCost().getValue())
                .map(Number::doubleValue)
                .map(cost -> cost * 1_000_000)
                .map(Double::longValue)
                .orElse(0L);
        this.revenue = Optional.ofNullable(values.getRevenue().getValue()).map(Number::longValue).orElse(0L);
    }

    public GdCampaignStatisticsColumnValues buildColumnValues() {
        return new GdCampaignStatisticsColumnValues()
                .withClicks(wrapToHolder(clicks))
                .withCost(wrapToHolder(fromMicros(costMicros)))
                .withRevenue(wrapToHolder(revenue))
                .withShows(wrapToHolder(shows))
                .withConversions(wrapToHolder(conversions))
                .withAvgCpc(wrapToHolder(getCpc()))
                .withCostPerConversion(wrapToHolder(getCpa()))
                .withCtr(wrapToHolder(getCtr()))
                .withAvgCpm(wrapToHolder(getCpm()))
                .withConversionRate(wrapToHolder(getCr()))
                .withCarts(wrapToHolder(carts))
                .withPurchases(wrapToHolder(purchases))
                .withInstalls(wrapToHolder(installs))
                .withPostViewConversions(wrapToHolder(postViewConversions))
                .withPostViewInstalls(wrapToHolder(postViewInstalls));
    }

    @Nullable
    BigDecimal getCpc() {
        return clicks != null && clicks != 0L ? fromMicros(costMicros / clicks) : null;
    }

    @Nullable
    BigDecimal getCpa() {
        return conversions != null && conversions != 0L ? fromMicros(costMicros / conversions) : null;
    }

    @Nullable
    BigDecimal getCpm() {
        return shows != null && shows != 0L ? fromMicros(1000 * costMicros / shows) : null;
    }

    @Nullable
    BigDecimal getCtr() {
        return clicks != null && shows != null && shows != 0L ? BigDecimal.valueOf(100.0 * clicks / shows) : null;
    }

    @Nullable
    BigDecimal getCr() {
        return clicks != null && clicks != 0L ? BigDecimal.valueOf(100.0 * conversions / clicks) : null;
    }

    public void addStats(StatsHolder dailyStatHolder) {
        shows += dailyStatHolder.shows;
        clicks += dailyStatHolder.clicks;
        conversions += dailyStatHolder.conversions;
        costMicros += dailyStatHolder.costMicros;
        revenue += dailyStatHolder.revenue;

        carts = nullableSum(carts, dailyStatHolder.carts);
        purchases = nullableSum(purchases, dailyStatHolder.purchases);

        installs = nullableSum(installs, dailyStatHolder.installs);
        postViewConversions = nullableSum(postViewConversions, dailyStatHolder.postViewConversions);
        postViewInstalls = nullableSum(postViewInstalls, dailyStatHolder.postViewInstalls);
    }

    private Long nullableSum(Long a, Long b) {
        if (a == null && b == null) {
            return null;
        }

        return nvl(a, 0L) + nvl(b, 0L);
    }
}
