package ru.yandex.direct.grid.processing.service.statistics.validation;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.grid.processing.model.GdLimitOffset;
import ru.yandex.direct.grid.processing.model.statistics.brandsafety.GdBrandSafetyStatsRequest;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.NumberConstraints.inRange;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notLessThan;

@Service
@ParametersAreNonnullByDefault
public class BrandSafetyStatisticsValidationService {

    @Autowired
    private GridValidationService gridValidationService;

    private static final int MAX_LIMIT = 10_000;
    private static final int MIN_LIMIT = 1;
    private static final int MIN_OFFSET = 0;

    private static final Validator<GdLimitOffset, Defect>
            GD_LIMIT_OFFSET_VALIDATOR = limitOffset -> {
        ModelItemValidationBuilder<GdLimitOffset> vb = ModelItemValidationBuilder.of(limitOffset);

        vb.item(GdLimitOffset.LIMIT)
                .check(notNull())
                .check(inRange(MIN_LIMIT, MAX_LIMIT));

        vb.item(GdLimitOffset.OFFSET)
                .check(notNull())
                .check(notLessThan(MIN_OFFSET));

        return vb.getResult();
    };

    private static final Validator<GdBrandSafetyStatsRequest, Defect> GD_BRAND_SAFETY_STATS_REQUEST_DEFECT_VALIDATOR =
            input -> {
                ModelItemValidationBuilder<GdBrandSafetyStatsRequest> vb = ModelItemValidationBuilder.of(input);

                vb.item(GdBrandSafetyStatsRequest.LIMIT_OFFSET)
                        .check(notNull())
                        .checkBy(GD_LIMIT_OFFSET_VALIDATOR, When.isValid());

                return vb.getResult();
            };

    public void validateBrandSafetyStatsRequest(GdBrandSafetyStatsRequest request) {
        gridValidationService.applyValidator(GD_BRAND_SAFETY_STATS_REQUEST_DEFECT_VALIDATOR, request, false);
    }
}
