package ru.yandex.direct.grid.processing.service.statistics.validation

import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.metrika.service.validation.MetrikaCountersValidationService
import ru.yandex.direct.core.entity.metrika.service.validation.MetrikaGoalsValidationService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.grid.processing.model.statistics.metrika.GdMetrikaStatisticsContainer
import ru.yandex.direct.grid.processing.model.statistics.metrika.GdMetrikaStatisticsFilter
import ru.yandex.direct.grid.processing.service.validation.GridValidationService
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder

@Service
class EndToEndAnalyticsValidationService(
    private val gridValidationService: GridValidationService,
    private val metrikaGoalsValidationService: MetrikaGoalsValidationService,
) {
    fun validateInput(input: GdMetrikaStatisticsContainer) {
        val vb = ModelItemValidationBuilder.of(input)
        val filterVB = vb.modelItem(GdMetrikaStatisticsContainer.FILTER)
        filterVB.item(GdMetrikaStatisticsFilter.COUNTER_ID)
            .check(CommonConstraints.notNull())
            .check(CommonConstraints.validId())
        filterVB.item(GdMetrikaStatisticsFilter.PERIOD)
            .check(
                Constraint.fromPredicate(
                    { (it.from != null && it.to != null) || it.preset != null },
                    CommonDefects.invalidValue()
                )
            )
        gridValidationService.throwGridValidationExceptionIfHasErrors(vb.result)
    }

    /**
     * Проверяем, есть ли у клиента права на счётчик.
     *
     * Принимает [input], а не счётчик, чтобы сразу указать правильный путь в дефекте
     */
    fun validateClientRightForCounter(input: GdMetrikaStatisticsContainer, clientId: ClientId) {
        val vb = ModelItemValidationBuilder.of(input)
        val filterVB = vb.modelItem(GdMetrikaStatisticsContainer.FILTER)

        // на этом этапе счётчик валиден
        val counterId = input.filter.counterId!!
        val availableCounterIds = metrikaGoalsValidationService.getAvailableCounterIds(clientId, listOf(counterId))

        filterVB.item(GdMetrikaStatisticsFilter.COUNTER_ID)
            .checkBy { MetrikaCountersValidationService.validateCounterId(it, availableCounterIds) }
        gridValidationService.throwGridValidationExceptionIfHasErrors(vb.result)
    }
}
