package ru.yandex.direct.grid.processing.service.statistics.validation

import org.springframework.stereotype.Service
import ru.yandex.direct.grid.processing.model.masterreport.GdMasterReportStatisticsContainer
import ru.yandex.direct.grid.processing.model.masterreport.GdMasterReportStatisticsFilter
import ru.yandex.direct.grid.processing.model.masterreport.GdMasterReportStatisticsGroupBy
import ru.yandex.direct.grid.processing.model.masterreport.GdMasterReportStatisticsGroupByDate
import ru.yandex.direct.grid.processing.model.statistics.GdCampaignStatisticsPeriod
import ru.yandex.direct.grid.processing.service.validation.GridValidationService
import ru.yandex.direct.validation.builder.Constraint.fromPredicateOfNullable
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.DateConstraints
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder
import javax.annotation.ParametersAreNonnullByDefault

@Service
@ParametersAreNonnullByDefault
class MasterReportStatisticsValidationService(
        private val gridValidationService: GridValidationService
) {

    fun validateGdStatisticsContainer(request: GdMasterReportStatisticsContainer) {
        gridValidationService.applyValidator(this::checkContainer, request, false)
    }

    private fun checkContainer(
            container: GdMasterReportStatisticsContainer
    ): ValidationResult<GdMasterReportStatisticsContainer, Defect<*>> {
        val vb = ModelItemValidationBuilder.of(container)
        vb.item(GdMasterReportStatisticsContainer.GROUP_BY_DATE)
                .checkBy { this.checkGroupByDate(it, container.groupBy) }
        vb.item(GdMasterReportStatisticsContainer.FILTER)
                .checkBy(this::checkFilter)
        vb.item(GdMasterReportStatisticsContainer.GROUP_BY)
                .checkBy { this.checkGroupBy(it, container.needComparePeriods) }
        return vb.result
    }

    private fun checkGroupByDate(
            groupByDate: GdMasterReportStatisticsGroupByDate?,
            groupBy: GdMasterReportStatisticsGroupBy?
    ): ValidationResult<GdMasterReportStatisticsGroupByDate, Defect<*>> {
        val vb = ItemValidationBuilder.of<GdMasterReportStatisticsGroupByDate, Defect<*>>(groupByDate)
        vb.check(fromPredicateOfNullable({ checkGroupByDate(groupBy, it) }, CommonDefects.inconsistentState()))
        return vb.result
    }

    private fun checkGroupByDate(
            groupBy: GdMasterReportStatisticsGroupBy?,
            groupByDate: GdMasterReportStatisticsGroupByDate?
    ) = when (groupByDate) {
        null -> true
        // groupByDate может быть задан, только если нет срезов groupBy
        else -> groupBy == null
    }

    private fun checkFilter(
            filter: GdMasterReportStatisticsFilter
    ): ValidationResult<GdMasterReportStatisticsFilter, Defect<*>> {
        val vb = ModelItemValidationBuilder.of(filter)
        vb.item(GdMasterReportStatisticsFilter.PERIOD)
                .checkBy(this::checkPeriod)
        return vb.result
    }

    private fun checkPeriod(
            period: GdCampaignStatisticsPeriod
    ): ValidationResult<GdCampaignStatisticsPeriod, Defect<*>> {
        val vb = ModelItemValidationBuilder.of(period)
        vb.item(GdCampaignStatisticsPeriod.TO)
                .check(DateConstraints.isNotBeforeThan(period.from), When.isTrue(period.from != null))
        return vb.result
    }

    private fun checkGroupBy(
            groupBy: GdMasterReportStatisticsGroupBy?,
            needComparePeriods: Boolean?
    ): ValidationResult<GdMasterReportStatisticsGroupBy, Defect<*>> {
        val vb = ItemValidationBuilder.of<GdMasterReportStatisticsGroupBy, Defect<*>>(groupBy)
        vb.check(fromPredicateOfNullable({ checkGroupBy(needComparePeriods, it) }, CommonDefects.inconsistentState()))
        return vb.result
    }

    private fun checkGroupBy(
            needComparePeriods: Boolean?,
            groupBy: GdMasterReportStatisticsGroupBy?
    ) = when (groupBy) {
        null -> true
        // groupBy может быть задан, только если не нужно сравнение периодов
        else -> needComparePeriods != true
    }
}
