package ru.yandex.direct.grid.processing.service.strategy

import io.leangen.graphql.annotations.GraphQLArgument
import io.leangen.graphql.annotations.GraphQLContext
import io.leangen.graphql.annotations.GraphQLQuery
import io.leangen.graphql.annotations.GraphQLRootContext
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.grid.model.campaign.GdCampaignTruncated
import ru.yandex.direct.grid.model.strategy.GdPackageStrategy
import ru.yandex.direct.grid.model.strategy.GdStrategyWithConversion
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext
import ru.yandex.direct.grid.processing.model.campaign.GdConversionStrategyLearningStatusData
import ru.yandex.direct.grid.processing.model.client.GdClient
import ru.yandex.direct.grid.processing.model.strategy.query.GdPackageStrategiesContainer
import ru.yandex.direct.grid.processing.model.strategy.query.GdPackageStrategiesContext
import ru.yandex.direct.grid.processing.service.strategy.loader.TruncatedCampaignsLoader
import ru.yandex.direct.grid.processing.service.strategy.query.PackageStrategyService
import java.util.concurrent.CompletableFuture

/**
 * Сервис, возвращающий данные о пакетных стратегиях клиента
 */
@GridGraphQLService
class PackageStrategyGraphQlService(
    private val packageStrategyService: PackageStrategyService,
    private val truncatedCampaignsLoader: TruncatedCampaignsLoader
) {
    companion object {
        const val STRATEGIES_RESOLVER_NAME = "strategies"
        const val LINKED_CAMPAIGNS_RESOLVER_NAME = "linkedCampaigns"
        const val CONVERSION_LEARNING_DATA = "conversionLearningData"
    }

    @GraphQLQuery(name = STRATEGIES_RESOLVER_NAME)
    fun listStrategies(
        @GraphQLRootContext context: GridGraphQLContext,
        @GraphQLContext client: GdClient,
        @GraphQLArgument(name = "input") input: GdPackageStrategiesContainer
    ): CompletableFuture<GdPackageStrategiesContext> =
        CompletableFuture.completedFuture(
            packageStrategyService.listStrategies(
                context,
                ClientId.fromLong(client.info.id),
                input
            )
        )

    @GraphQLQuery(name = LINKED_CAMPAIGNS_RESOLVER_NAME)
    fun getLinkedCampaigns(
        @GraphQLContext strategy: GdPackageStrategy
    ): CompletableFuture<List<GdCampaignTruncated>> =
        truncatedCampaignsLoader.get().load(strategy)

    @GraphQLQuery(name = CONVERSION_LEARNING_DATA)
    fun getPackageStrategyConversionLearningData(
        @GraphQLContext strategy: GdStrategyWithConversion
    ): CompletableFuture<GdConversionStrategyLearningStatusData>? =
        packageStrategyService.getPackageStrategyConversionLearningData(strategy)
}
