package ru.yandex.direct.grid.processing.service.strategy.dataloader;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDate;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.dataloader.MappedBatchLoaderWithContext;
import org.springframework.context.annotation.Scope;
import org.springframework.context.annotation.ScopedProxyMode;
import org.springframework.stereotype.Component;
import org.springframework.web.context.WebApplicationContext;

import ru.yandex.direct.core.entity.container.LocalDateRange;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.core.entity.model.GdiGoalStats;
import ru.yandex.direct.grid.core.entity.model.campaign.GdiAggregatorGoal;
import ru.yandex.direct.grid.core.entity.model.campaign.GdiCampaignStats;
import ru.yandex.direct.grid.core.entity.strategy.GridStatPackageStrategyService;
import ru.yandex.direct.grid.model.campaign.GdMeaningfulGoal;
import ru.yandex.direct.grid.model.strategy.GdPackageStrategy;
import ru.yandex.direct.grid.model.strategy.GdStrategyWithConversion;
import ru.yandex.direct.grid.model.strategy.GdStrategyWithLastBidderRestartTime;
import ru.yandex.direct.grid.model.strategy.GdStrategyWithMeaningfulGoals;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.campaign.GdConversionStrategyLearningStatusData;
import ru.yandex.direct.grid.processing.service.dataloader.GridBatchingDataLoader;
import ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider;
import ru.yandex.direct.grid.processing.service.strategy.utils.ConversionPackageStrategyLearningData;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.ytwrapper.dynamic.dsl.YtQueryUtil.DECIMAL_MULT;
import static ru.yandex.misc.lang.ObjectUtils.max;

@Component
@ParametersAreNonnullByDefault
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
public class ConversionPackageStrategyLearningStatusDataLoader
        extends GridBatchingDataLoader<GdStrategyWithConversion, GdConversionStrategyLearningStatusData> {

    public static final int MAX_DAYS_COUNT_FOR_STRATEGY_LEARNING = 14;

    public ConversionPackageStrategyLearningStatusDataLoader(
            GridContextProvider gridContextProvider,
            GridStatPackageStrategyService gridStatPackageStrategyService) {

        this.dataLoader = mappedDataLoader(
                gridContextProvider,
                getBatchLoadFunction(gridStatPackageStrategyService)
        );
    }

    private MappedBatchLoaderWithContext<GdStrategyWithConversion, GdConversionStrategyLearningStatusData> getBatchLoadFunction(
            GridStatPackageStrategyService gridStatPackageStrategyService) {
        return (strategies, environment) -> {
            GridGraphQLContext context = environment.getContext();
            User subjectUser = context.getSubjectUser();
            checkNotNull(subjectUser, "subjectUser should be set in gridContext");

            ClientId clientId = subjectUser.getClientId();

            return CompletableFuture.completedFuture(getStatusDataByStrategy(clientId, strategies, gridStatPackageStrategyService));
        };
    }

    private Map<GdStrategyWithConversion, GdConversionStrategyLearningStatusData> getStatusDataByStrategy(
            ClientId clientId,
            Set<GdStrategyWithConversion> strategies,
            GridStatPackageStrategyService gridStatPackageStrategyService) {

        Set<GdPackageStrategy> packageStrategies = StreamEx.of(strategies)
                .select(GdPackageStrategy.class)
                .filter(strategy -> !strategy.getStatusArchived())
                .toSet();

        //Нет нужды ходить за статистикой, при отсутствии нужных стратегий
        if (packageStrategies.isEmpty()) {
            return Map.of();
        }

        Map<Long, Long> strategyGoalIdsByStrategyId = getStrategyGoalIdsByStrategyId(packageStrategies);
        Map<Long, List<Long>> meaningfulGoalsByStrategyId = extractMeaningfulGoalsByStrategyId(packageStrategies);

        Map<GdPackageStrategy, GdConversionStrategyLearningStatusData> statusDataByPackageStrategy =
                getStatusDataByStrategyForPackageStrategies(
                        clientId,
                        packageStrategies,
                        strategyGoalIdsByStrategyId,
                        meaningfulGoalsByStrategyId,
                        gridStatPackageStrategyService
                );

        return EntryStream.of(statusDataByPackageStrategy)
                .selectKeys(GdStrategyWithConversion.class)
                .toMap();
    }

    private Map<Long, Long> getStrategyGoalIdsByStrategyId(Set<GdPackageStrategy> strategies) {
        return StreamEx.of(strategies)
                .mapToEntry(GdPackageStrategy::getId, Function.identity())
                .selectValues(GdStrategyWithConversion.class)
                .mapValues(strategy -> {
                    if (strategy.getGoalId() == null) {
                        return ((GdStrategyWithMeaningfulGoals) strategy).getMeaningfulGoals().get(0).getGoalId();
                    } else {
                        return strategy.getGoalId();
                    }
                })
                .toMap();
    }

    private Map<Long, List<Long>> extractMeaningfulGoalsByStrategyId(Set<GdPackageStrategy> strategies) {
        return StreamEx.of(strategies)
                .mapToEntry(GdPackageStrategy::getId, Function.identity())
                .selectValues(GdStrategyWithMeaningfulGoals.class)
                .mapValues(GdStrategyWithMeaningfulGoals::getMeaningfulGoals)
                .flatMapValues(List::stream)
                .mapValues(GdMeaningfulGoal::getGoalId)
                .grouping();
    }

    private Map<GdPackageStrategy, GdConversionStrategyLearningStatusData> getStatusDataByStrategyForPackageStrategies(
            ClientId clientId,
            Set<GdPackageStrategy> strategies,
            Map<Long, Long> strategyGoalIdsByStrategyId,
            Map<Long, List<Long>> meaningfulGoalsByStrategyId,
            GridStatPackageStrategyService gridStatPackageStrategyService) {

        LocalDate now = LocalDate.now();

        Map<Long, LocalDateRange> dateRangesByStrategyId = StreamEx.of(strategies)
                .mapToEntry(GdPackageStrategy::getId, strategy -> calculateStatStartDateForStrategy(strategy, now))
                .mapValues(from -> new LocalDateRange().withFromInclusive(from).withToInclusive(now))
                .toMap();

        Map<Long, GdiAggregatorGoal> aggregatorGoalByStrategyId = getAggregatorGoalByStrategyId(
                strategyGoalIdsByStrategyId,
                meaningfulGoalsByStrategyId
        );

        Map<Long, GdiCampaignStats> strategyStatsByStrategyId = gridStatPackageStrategyService.getStatsByStrategyIdsWithDifferentDateRange(
                clientId,
                dateRangesByStrategyId,
                aggregatorGoalByStrategyId,
                strategyGoalIdsByStrategyId
        );

        Map<Long, LocalDateRange> lastDayDateRangesByStrategyId = listToMap(
                strategies,
                GdPackageStrategy::getId,
                strategy -> new LocalDateRange().withFromInclusive(now.minusDays(1)).withToInclusive(now)
        );

        Map<Long, GdiCampaignStats> strategyLastDayStatsByStrategyId = gridStatPackageStrategyService.getStatsByStrategyIdsWithDifferentDateRange(
                clientId,
                lastDayDateRangesByStrategyId,
                aggregatorGoalByStrategyId,
                strategyGoalIdsByStrategyId
        );

        Map<Long, GdPackageStrategy> strategiesById = StreamEx.of(strategies)
                .toMap(GdPackageStrategy::getId, Function.identity());

        return EntryStream.of(strategyStatsByStrategyId)
                .mapKeys(strategiesById::get)
                .mapToValue((strategy, stats) -> getStatusDataForStrategy(
                        stats,
                        strategyLastDayStatsByStrategyId.get(strategy.getId()),
                        strategy,
                        now
                ))
                .toMap();
    }

    private Map<Long, GdiAggregatorGoal> getAggregatorGoalByStrategyId(
            Map<Long, Long> strategyGoalIdsByStrategyId,
            Map<Long, List<Long>> meaningfulGoalsByStrategyId) {

        return EntryStream.of(strategyGoalIdsByStrategyId)
                .mapToValue((strategyId, goalId) -> {
                    if (goalId.equals(MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID)) {
                        return new GdiAggregatorGoal().withId(MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID)
                                        .withSubGoalIds(List.copyOf(meaningfulGoalsByStrategyId.get(strategyId)));
                    } else {
                        return new GdiAggregatorGoal().withId(goalId)
                                        .withSubGoalIds(List.of(goalId));
                    }
                })
                .toMap();
    }

    private static LocalDate calculateStatStartDateForStrategy(GdPackageStrategy strategy, LocalDate now) {
        LocalDate from = now.minusDays(MAX_DAYS_COUNT_FOR_STRATEGY_LEARNING);

        if (strategy instanceof GdStrategyWithLastBidderRestartTime && ((GdStrategyWithLastBidderRestartTime) strategy).getLastBidderRestartTime() != null) {
            from = max(from, ((GdStrategyWithLastBidderRestartTime) strategy).getLastBidderRestartTime().toLocalDate());
        }

        return from;
    }

    private GdConversionStrategyLearningStatusData getStatusDataForStrategy(
            GdiCampaignStats strategyStats,
            GdiCampaignStats lastDayStrategyStats,
            GdPackageStrategy strategy,
            LocalDate now) {
        GdiGoalStats goalStats = strategyStats.getGoalStats().stream().findAny().get();

        Long conversionCount = goalStats.getGoals();
        BigDecimal costPerAction = goalStats.getCostPerAction();
        BigDecimal conversionRate = goalStats.getConversionRate().divide(DECIMAL_MULT, 2, RoundingMode.HALF_UP);

        LocalDate restartDate = calculateStatStartDateForStrategy(strategy, now);

        var conversionPackageStrategyLearningData = new ConversionPackageStrategyLearningData(
                strategy,
                restartDate,
                strategyStats,
                lastDayStrategyStats
        );

        var status = conversionPackageStrategyLearningData.getStatus(now);

        return new GdConversionStrategyLearningStatusData()
                .withRestartDate(restartDate)
                .withStatisticCalculationStartTime(restartDate)
                .withConversionCount(conversionCount)
                .withAverageConversion(costPerAction)
                .withConversionRate(conversionRate)
                .withCostRevenueRatio(conversionPackageStrategyLearningData.goalsCrr())
                .withStatus(status);
    }
}
