package ru.yandex.direct.grid.processing.service.strategy.loader

import org.dataloader.BatchLoaderEnvironment
import org.springframework.context.annotation.Scope
import org.springframework.context.annotation.ScopedProxyMode
import org.springframework.stereotype.Component
import org.springframework.web.context.WebApplicationContext
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.grid.model.campaign.GdCampaignTruncated
import ru.yandex.direct.grid.model.strategy.GdPackageStrategy
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext
import ru.yandex.direct.grid.processing.service.campaign.CampaignInfoService
import ru.yandex.direct.grid.processing.service.dataloader.GridBatchingDataLoader
import ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider
import java.util.concurrent.CompletableFuture
import java.util.concurrent.CompletionStage

@Component
// DataLoader'ы хранят состояние, поэтому жить должны в рамках запроса
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
class TruncatedCampaignsLoader(
    private val campaignInfoService: CampaignInfoService,
    gridContextProvider: GridContextProvider
) : GridBatchingDataLoader<GdPackageStrategy, List<GdCampaignTruncated>>() {

    init {
        dataLoader = mappedDataLoader(gridContextProvider, this::load)
    }

    private fun load(
        strategies: Set<GdPackageStrategy>,
        environment: BatchLoaderEnvironment
    ): CompletionStage<Map<GdPackageStrategy, List<GdCampaignTruncated>>> {
        val context = environment.getContext<GridGraphQLContext>()
        val clientInfo = context.queriedClient
            ?: throw IllegalArgumentException("Queried client is undefined")

        val campaignByIds = campaignByIds(ClientId.fromLong(clientInfo.id), strategies)
        return CompletableFuture.completedFuture(
            strategies.associateWith {
                it.cids.mapNotNull(campaignByIds::get)
            }
        )
    }

    private fun campaignByIds(
        clientId: ClientId,
        strategies: Set<GdPackageStrategy>
    ): Map<Long, GdCampaignTruncated> {
        val campaignIds = strategies.flatMap { it.cids }.toSet()
        return campaignInfoService.getTruncatedCampaigns(
            clientId,
            campaignIds
        )
    }
}
