package ru.yandex.direct.grid.processing.service.strategy.mutation

import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPackageStrategy
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.campaign.service.CampaignOperationService
import ru.yandex.direct.core.entity.campaign.service.CampaignOptions
import ru.yandex.direct.core.entity.campaign.service.RestrictedCampaignsUpdateOperation
import ru.yandex.direct.core.entity.strategy.model.CommonStrategy
import ru.yandex.direct.core.entity.strategy.repository.StrategyTypedRepository
import ru.yandex.direct.dbutil.model.UidAndClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext
import ru.yandex.direct.grid.processing.model.strategy.mutation.GdPackageStrategyUnbindCampaigns
import ru.yandex.direct.grid.processing.model.strategy.mutation.GdUnbindedCampaignItem
import ru.yandex.direct.grid.processing.model.strategy.mutation.GdUnbindedCampaigns
import ru.yandex.direct.grid.processing.model.strategy.mutation.GdUpdatePackageStrategyiesUnbindCampaigns
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService
import ru.yandex.direct.grid.processing.util.ResponseConverter
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.validation.result.PathHelper

@Service
class PackageStrategyCampaignUnbindService(
    private val campaignOperationService: CampaignOperationService,
    private val shardHelper: ShardHelper,
    private val validationService: GridValidationResultConversionService,
    private val strategyTypedRepository: StrategyTypedRepository,
    private val campaignTypedRepository: CampaignTypedRepository,
) {

    fun unbind(
        context: GridGraphQLContext,
        input: GdUpdatePackageStrategyiesUnbindCampaigns
    ): GdUnbindedCampaigns {
        val uidAndClientId = UidAndClientId.of(context.subjectUser?.uid, context.subjectUser?.clientId)
        val shard = shardHelper.getShardByClientId(uidAndClientId.clientId)
        val unbindContainers = campaignsToUnbind(shard, input.unbindItems)
        return if (unbindContainers.isNotEmpty()) {
            unbind(
                context.operator.uid,
                uidAndClientId,
                unbindContainers
            )
        } else {
            GdUnbindedCampaigns()
                .withUnbindedCampaigns(emptyList())
        }
    }

    private fun campaignsToUnbind(
        shard: Int,
        items: List<GdPackageStrategyUnbindCampaigns>
    ): List<CampaignUnbindContainer> {
        val campaignsByStrategy = items.associateBy({ it.strategyId }) { it.campaignIds }
        val campaignsById = campaignTypedRepository
            .getTypedCampaigns(shard, items.map { it.campaignIds }.flatten())
            .mapNotNull { it as? CampaignWithPackageStrategy }
            .associateBy { it.id }
        val strategies = strategyTypedRepository.getTyped(shard, campaignsByStrategy.keys)
        return strategies
            .mapNotNull { it as? CommonStrategy }
            .filter { it.isPublic }
            .map { strategy ->
                val campaignIdsToUnbind = campaignsByStrategy[strategy.id] ?: emptyList()
                val linkedCampaigns = strategy.cids?.toSet() ?: emptySet()
                campaignIdsToUnbind.filter(linkedCampaigns::contains)
                    .mapNotNull {
                        campaignsById[it]?.let { campaign ->
                            CampaignUnbindContainer(
                                it,
                                strategy,
                                campaign.javaClass
                            )
                        }
                    }
            }.flatten()
    }

    private fun unbind(
        operatorUid: Long,
        uidAndClientId: UidAndClientId,
        unbindContainers: List<CampaignUnbindContainer>
    ): GdUnbindedCampaigns {
        val operation = updateOperation(operatorUid, uidAndClientId, unbindContainers)

        val massResult = operation.apply()

        val gdValidationResult = validationService.buildGridValidationResult(
            massResult.validationResult,
            PathHelper.path(PathHelper.field(GdUnbindedCampaigns.UNBINDED_CAMPAIGNS))
        )
        val items = ResponseConverter.getResults(massResult) {
            GdUnbindedCampaignItem().withId(it)
        }
        return GdUnbindedCampaigns()
            .withUnbindedCampaigns(items)
            .withValidationResult(gdValidationResult)
    }

    private fun updateOperation(
        operatorUid: Long,
        uidAndClientId: UidAndClientId,
        unbindContainers: List<CampaignUnbindContainer>
    ): RestrictedCampaignsUpdateOperation {
        val modelChanges = unbindContainers.map { container ->
            unbindModelChanges(container)
        }
        return campaignOperationService.createRestrictedCampaignUpdateOperation(
            modelChanges,
            operatorUid,
            uidAndClientId,
            CampaignOptions()
        )
    }

    private fun unbindModelChanges(
        container: CampaignUnbindContainer
    ): ModelChanges<CampaignWithPackageStrategy> {
        return ModelChanges(container.campaignId, CampaignWithPackageStrategy::class.java)
            .process(null, CampaignWithPackageStrategy.STRATEGY_ID)
    }

    companion object {
        internal data class CampaignUnbindContainer(
            val campaignId: Long,
            val strategy: CommonStrategy,
            val campaignClass: Class<out CampaignWithPackageStrategy>
        )
    }
}
