package ru.yandex.direct.grid.processing.service.strategy.mutation

import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.strategy.container.StrategyOperationOptions
import ru.yandex.direct.core.entity.strategy.model.CommonStrategy
import ru.yandex.direct.core.entity.strategy.service.StrategyOperationFactory
import ru.yandex.direct.core.entity.strategy.service.add.StrategyAddOperation
import ru.yandex.direct.core.entity.strategy.service.update.StrategyUpdateOperation
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext
import ru.yandex.direct.grid.processing.model.strategy.mutation.GdAddPackageStrategies
import ru.yandex.direct.grid.processing.model.strategy.mutation.GdAddPackageStrategyPayload
import ru.yandex.direct.grid.processing.model.strategy.mutation.GdAddPackageStrategyPayloadItem
import ru.yandex.direct.grid.processing.model.strategy.mutation.GdUpdatePackageStrategies
import ru.yandex.direct.grid.processing.model.strategy.mutation.GdUpdatePackageStrategiesStatusArchive
import ru.yandex.direct.grid.processing.model.strategy.mutation.GdUpdatePackageStrategiesStatusArchivePayload
import ru.yandex.direct.grid.processing.model.strategy.mutation.GdUpdatePackageStrategyPayload
import ru.yandex.direct.grid.processing.model.strategy.mutation.GdUpdatePackageStrategyPayloadItem
import ru.yandex.direct.grid.processing.service.strategy.mutation.validation.PackageStrategiesValidationService
import ru.yandex.direct.grid.processing.util.ResponseConverter.getResults
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.validation.result.PathHelper.field
import ru.yandex.direct.validation.result.PathHelper.path

@Service
class PackageStrategyMutationService(
    private val strategyOperationFactory: StrategyOperationFactory,
    private val shardHelper: ShardHelper,
    private val validationService: PackageStrategiesValidationService
) {

    fun updateStrategies(
        context: GridGraphQLContext,
        input: GdUpdatePackageStrategies
    ): GdUpdatePackageStrategyPayload {
        val updateOperation = createUpdateOperation(context, input)
        val massResult = updateOperation.prepareAndApply()
        val gdValidationResult = validationService.getValidationResult(
            massResult,
            path(field(GdUpdatePackageStrategies.STRATEGY_UPDATE_ITEMS))
        )
        val items = getResults(massResult) {
            GdUpdatePackageStrategyPayloadItem().withId(it)
        }
        return GdUpdatePackageStrategyPayload()
            .withUpdatedStrategies(items)
            .withValidationResult(gdValidationResult)
    }

    fun addPackageStrategies(
        input: GdAddPackageStrategies,
        context: GridGraphQLContext
    ): GdAddPackageStrategyPayload {
        val operation = createAddOperation(context, input)
        val massResult = operation.prepareAndApply()

        val gdValidationResult = validationService.getAddValidationResult(
            massResult,
            path(field(GdAddPackageStrategies.STRATEGY_ADD_ITEMS))
        )
        val items = getResults(massResult) {
            GdAddPackageStrategyPayloadItem().withId(it)
        }
        return GdAddPackageStrategyPayload()
            .withAddedStrategies(items)
            .withValidationResult(gdValidationResult)
    }

    private fun createAddOperation(
        context: GridGraphQLContext,
        input: GdAddPackageStrategies
    ): StrategyAddOperation {
        validationService.validateAddPackageStrategies(input)
        val clientId = context.subjectUser?.clientId
        val clientUid = context.subjectUser?.uid
        val operator = context.operator.uid
        val shard = shardHelper.getShardByClientId(clientId)
        val updates = input.strategyAddItems.map(PackageStrategyConverter::toGdAddPackageStrategy)
        val strategies = updates.map(PackageStrategyConverter::toStrategy)
        return strategyOperationFactory.createStrategyAddOperation(
            shard,
            operator,
            clientId,
            clientUid,
            strategies,
            StrategyOperationOptions()
        )
    }

    fun changeStrategiesArchiveStatus(
        context: GridGraphQLContext,
        input: GdUpdatePackageStrategiesStatusArchive
    ): GdUpdatePackageStrategiesStatusArchivePayload {
        val clientId = context.subjectUser?.clientId
        val shard = shardHelper.getShardByClientId(clientId)

        val operation = strategyOperationFactory.createChangeStatusArchiveOperation(
            shard, clientId, input.strategyIds, input.statusArchive
        )

        val massResult = operation.prepareAndApply()

        val gdValidationResult = validationService.getValidationResult(
            massResult,
            path(field(GdUpdatePackageStrategiesStatusArchive.STRATEGY_IDS))
        )

        val updatedIds = getResults(massResult) { it }

        return GdUpdatePackageStrategiesStatusArchivePayload()
            .withUpdatedStrategyIds(updatedIds)
            .withValidationResult(gdValidationResult)
    }

    private fun createUpdateOperation(
        context: GridGraphQLContext,
        input: GdUpdatePackageStrategies
    ): StrategyUpdateOperation<*> {
        validationService.validateUpdatePackageStrategies(input)
        val clientId = context.subjectUser?.clientId
        val clientUid = context.subjectUser?.uid
        val operator = context.operator.uid
        val shard = shardHelper.getShardByClientId(clientId)
        val updates = input.strategyUpdateItems.map(PackageStrategyConverter::toGdUpdatePackageStrategy)
        val modelChanges = updates.map(PackageStrategyConverter::toModelChanges)
        return strategyOperationFactory.createStrategyUpdateOperation(
            shard,
            clientId,
            clientUid,
            operator,
            StrategyOperationOptions(),
            modelChanges as List<ModelChanges<CommonStrategy>>
        )
    }
}
