package ru.yandex.direct.grid.processing.service.strategy.query

import ru.yandex.direct.grid.model.GdEntityStats
import ru.yandex.direct.grid.model.Order
import ru.yandex.direct.grid.model.strategy.GdPackageStrategy
import ru.yandex.direct.grid.model.strategy.GdStrategyName
import ru.yandex.direct.grid.processing.model.strategy.query.GdPackageStrategyOrderBy
import ru.yandex.direct.grid.processing.model.strategy.query.GdPackageStrategyOrderByField

object PackageStrategyOrderByProcessor {

    fun comparator(orderBy: List<GdPackageStrategyOrderBy>?): Comparator<GdPackageStrategy> {
        return if (orderBy.isNullOrEmpty()) {
            Comparator { _, _ -> 0 }
        } else {
            orderBy
                .map { getComparator(it.field, it.order) }
                .reduce { acc, comparator -> acc.thenComparing(comparator) }
        }
    }

    private fun <T : Comparable<T>> naturalComparator(order: Order) =
        if (order == Order.DESC) {
            Comparator.nullsLast(Comparator.reverseOrder<T>())
        } else {
            Comparator.nullsLast(Comparator.naturalOrder<T>())
        }

    private fun getComparator(
        field: GdPackageStrategyOrderByField,
        order: Order
    ): Comparator<GdPackageStrategy> = Comparator { a: GdPackageStrategy, b: GdPackageStrategy ->
        fun <T : Comparable<T>> compareStat(f: (GdEntityStats) -> T): Int =
            naturalComparator<T>(order).compare(a.stats?.let { f(it) }, b.stats?.let { f(it) })

        when (field) {
            GdPackageStrategyOrderByField.ID -> naturalComparator<Long>(order).compare(a.id, b.id)
            GdPackageStrategyOrderByField.NAME -> naturalComparator<String>(order).compare(a.name, b.name)
            GdPackageStrategyOrderByField.TYPE -> naturalComparator<GdStrategyName>(order).compare(a.type, b.type)
            GdPackageStrategyOrderByField.STAT_COST -> compareStat(GdEntityStats::getCost)
            GdPackageStrategyOrderByField.STAT_COST_WITH_TAX -> compareStat(GdEntityStats::getCostWithTax)
            GdPackageStrategyOrderByField.STAT_SHOWS -> compareStat(GdEntityStats::getShows)
            GdPackageStrategyOrderByField.STAT_CLICKS -> compareStat(GdEntityStats::getClicks)
            GdPackageStrategyOrderByField.STAT_CTR -> compareStat(GdEntityStats::getCtr)
            GdPackageStrategyOrderByField.STAT_REVENUE -> compareStat(GdEntityStats::getRevenue)
            GdPackageStrategyOrderByField.STAT_GOALS -> compareStat(GdEntityStats::getGoals)
            GdPackageStrategyOrderByField.STAT_CPM_PRICE -> compareStat(GdEntityStats::getCpmPrice)
            GdPackageStrategyOrderByField.STAT_PROFITABILITY -> compareStat(GdEntityStats::getProfitability)
            GdPackageStrategyOrderByField.STAT_CRR -> compareStat(GdEntityStats::getCrr)
            GdPackageStrategyOrderByField.STAT_AVG_DEPTH -> compareStat(GdEntityStats::getAvgDepth)
            GdPackageStrategyOrderByField.STAT_AVG_GOAL_COST -> compareStat(GdEntityStats::getAvgGoalCost)
            GdPackageStrategyOrderByField.STAT_AVG_CLICK_COST -> compareStat(GdEntityStats::getAvgClickCost)
            GdPackageStrategyOrderByField.STAT_AVG_SHOW_POSITION -> compareStat(GdEntityStats::getAvgShowPosition)
            GdPackageStrategyOrderByField.STAT_AVG_CLICK_POSITION -> compareStat(GdEntityStats::getAvgClickCost)
            GdPackageStrategyOrderByField.STAT_BOUNCE_RATE -> compareStat(GdEntityStats::getBounceRate)
            GdPackageStrategyOrderByField.STAT_CONVERSION_RATE -> compareStat(GdEntityStats::getConversionRate)
        }
    }
}

