package ru.yandex.direct.grid.processing.service.strategy.query

import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.campaign.model.MeaningfulGoal
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_CPI_GOAL_ID
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.ENGAGED_SESSION_GOAL_ID
import ru.yandex.direct.core.entity.strategy.model.AutobudgetAvgCpi
import ru.yandex.direct.core.entity.strategy.model.BaseStrategy
import ru.yandex.direct.core.entity.strategy.model.CommonStrategy
import ru.yandex.direct.core.entity.strategy.model.StrategyWithMeaningfulGoals
import ru.yandex.direct.core.entity.strategy.repository.StrategyTypedRepository
import ru.yandex.direct.core.entity.strategy.service.StrategyConstants
import ru.yandex.direct.dbschema.ppc.tables.Strategies
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.grid.core.entity.strategy.GridStatPackageStrategyService
import ru.yandex.direct.grid.model.strategy.GdPackageStrategy
import ru.yandex.direct.grid.model.strategy.GdStrategyWithConversion
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext
import ru.yandex.direct.grid.processing.model.campaign.GdConversionStrategyLearningStatusData
import ru.yandex.direct.grid.processing.model.strategy.query.GdPackageStrategiesContainer
import ru.yandex.direct.grid.processing.model.strategy.query.GdPackageStrategiesContext
import ru.yandex.direct.grid.processing.model.strategy.query.GdPackageStrategyFilter
import ru.yandex.direct.grid.processing.service.cache.GridCacheService
import ru.yandex.direct.grid.processing.service.cache.util.CacheUtils.normalizeLimitOffset
import ru.yandex.direct.grid.processing.service.shortener.GridShortenerService
import ru.yandex.direct.grid.processing.service.strategy.container.PackageStrategiesCacheFilterData
import ru.yandex.direct.grid.processing.service.strategy.container.PackageStrategiesCacheRecordInfo
import ru.yandex.direct.grid.processing.service.strategy.dataloader.ConversionPackageStrategyLearningStatusDataLoader
import ru.yandex.direct.grid.processing.service.strategy.query.PackageStrategyFilterProcessor.DEFAULT_PACKAGE_STRATEGY_FILTER_PROCESSOR
import ru.yandex.direct.grid.processing.service.strategy.query.PackageStrategyFilterProcessor.ENTITY_STATS_FILTER_PROCESSOR
import ru.yandex.direct.grid.processing.service.strategy.query.PackageStrategyFilterProcessor.applyGoalsStatFilters
import ru.yandex.direct.grid.processing.service.validation.GridValidationService
import ru.yandex.direct.grid.processing.util.GoalHelper
import ru.yandex.direct.grid.processing.util.StatHelper
import ru.yandex.direct.grid.processing.util.StatHelper.calcTotalStats
import ru.yandex.direct.grid.processing.util.StatHelper.convertInternalStatsToOuter
import ru.yandex.direct.multitype.repository.filter.ConditionFilterFactory.whereEqFilter
import java.util.concurrent.CompletableFuture

@Service
class PackageStrategyService(
    val gridCacheService: GridCacheService,
    val strategyTypedRepository: StrategyTypedRepository,
    val shardHelper: ShardHelper,
    val gridShortenerService: GridShortenerService,
    val gridValidationService: GridValidationService,
    val statService: GridStatPackageStrategyService,
    val conversionPackageStrategyLearningStatusDataLoader: ConversionPackageStrategyLearningStatusDataLoader,
) {

    fun listStrategies(
        rootContext: GridGraphQLContext,
        clientId: ClientId,
        input: GdPackageStrategiesContainer
    ): GdPackageStrategiesContext {
        gridValidationService.validateGdPackagiesStrategyContainer(input)

        return fromCacheOr(rootContext, clientId.asLong(), input) {
            getGdPackageStrategiesContext(rootContext, clientId, input)
        }
    }

    private fun getAllClientStrategies(clientId: ClientId): List<BaseStrategy> {
        val shard = shardHelper.getShardByClientId(clientId)
        return strategyTypedRepository.getSafely(
            shard,
            whereEqFilter(Strategies.STRATEGIES.CLIENT_ID, clientId.asLong()),
            StrategyConstants.STRATEGY_TYPE_TO_CLASS.values
        )
    }

    private fun getGdPackageStrategiesContext(
        rootContext: GridGraphQLContext,
        clientId: ClientId,
        input: GdPackageStrategiesContainer
    ): GdPackageStrategiesContext {
        val strategies = getAllClientStrategies(clientId).mapNotNull { it as? CommonStrategy }
        strategies.forEach(this::enrichWithEngagedSession)
        strategies.forEach(this::setGoalIdForStrategy)
        val gdStrategies = strategies.mapNotNull { GdPackageConverter.convert(it) }
        if (!input.filterKey.isNullOrEmpty()) {
            val savedFilter = gridShortenerService.getSavedFilter(
                input.filterKey,
                clientId,
                GdPackageStrategyFilter::class.java
            ) {
                GdPackageStrategyFilter().withStrategyIdIn(emptySet())
            }
            input.filter = savedFilter
        }
        val filtered = filterPackageStrategies(input.filter, gdStrategies)
        if (isStatNeed(rootContext, input)) {
            enrichWithStatistics(rootContext, input, filtered)
        }
        val filteredByStat = filterPackageStrategiesByStats(input.filter, filtered)
        val totalStats = if (isStatNeed(rootContext, input)) {
            calcTotalStats(filteredByStat.map { it.stats })
        } else {
            calcTotalStats(emptyList())
        }
        val totalGoalStats = totalStats?.let {
            StatHelper.calcTotalGoalStats(it, filteredByStat.map { it.goalStats })
        }
        val sorted = filteredByStat.sortedWith(PackageStrategyOrderByProcessor.comparator(input.orderBy))
        return GdPackageStrategiesContext()
            .withTotalCount(sorted.size)
            .withRowset(sorted)
            .withFilter(input.filter)
            .withTotalStats(totalStats)
            .withTotalGoalStats(totalGoalStats)
    }

    fun enrichWithEngagedSession(strategy: CommonStrategy) {
        if (strategy is StrategyWithMeaningfulGoals && strategy.meaningfulGoals.isNullOrEmpty()) {
            strategy.meaningfulGoals = listOf(MeaningfulGoal().withGoalId(ENGAGED_SESSION_GOAL_ID))
        }
    }

    private fun enrichWithStatistics(
        rootContext: GridGraphQLContext,
        input: GdPackageStrategiesContainer,
        strategies: List<GdPackageStrategy>
    ) {
        val strategyStats = if (input.statRequirements.useCampaignGoalIds == true) {
            statService.getStatsForAssignedGoals(
                rootContext.subjectUser?.clientId!!,
                rootContext.operator.uid,
                strategies.map { it.id }.toSet(),
                input.statRequirements.from,
                input.statRequirements.to,
                true
            )
        } else {
            val goalIds =
                GoalHelper.combineGoalIds(input.statRequirements.goalIds, input.filter.goalStats)

            statService.getStatsByGoals(
                rootContext.subjectUser?.clientId!!,
                rootContext.operator.uid,
                strategies.map { it.id }.toSet(),
                input.statRequirements.from,
                input.statRequirements.to,
                goalIds,
                true
            )
        }
        strategies.forEach {
            strategyStats[it.id]?.let { statistics ->
                //TODO возможно тут стоит воспользоваться методом internalStatsToOuter
                it.stats = convertInternalStatsToOuter(statistics.stat)
                it.goalStats = statistics.goalStats?.map(StatHelper::internalGoalStatToOuter)
            }
        }
    }

    private fun isStatNeed(
        rootContext: GridGraphQLContext,
        input: GdPackageStrategiesContainer
    ) = rootContext.fetchedFieldsReslover.packageStrategy.stats &&
        input.statRequirements != null

    private fun filterPackageStrategies(
        filter: GdPackageStrategyFilter?,
        strategies: List<GdPackageStrategy>
    ) = if (filter == null) {
        strategies
    } else {
        DEFAULT_PACKAGE_STRATEGY_FILTER_PROCESSOR
            .filter(filter, strategies)
    }

    private fun filterPackageStrategiesByStats(
        filter: GdPackageStrategyFilter?,
        strategies: List<GdPackageStrategy>
    ) = if (filter == null) {
        strategies
    } else {
        ENTITY_STATS_FILTER_PROCESSOR
            .filter(filter.stat, strategies)
            .filter { applyGoalsStatFilters(filter.goalStats, it) }
    }

    private fun fromCacheOr(
        rootContext: GridGraphQLContext,
        client: Long,
        input: GdPackageStrategiesContainer,
        or: () -> GdPackageStrategiesContext
    ): GdPackageStrategiesContext {
        val recordInfo = PackageStrategiesCacheRecordInfo(
            client,
            input.cacheKey,
            PackageStrategiesCacheFilterData()
                .withFilter(input.filter)
                .withOrderBy(input.orderBy)
        )
        val range = normalizeLimitOffset(input.limitOffset)
        val fromCache = gridCacheService.getFromCache(recordInfo, range)
        return if (fromCache.isPresent) {
            fromCache.get()
        } else {
            val context = or()
            gridCacheService.getResultAndSaveToCacheIfRequested(
                recordInfo,
                context,
                context.rowset,
                range,
                rootContext.fetchedFieldsReslover.packageStrategy.cacheKey
            )
        }
    }

    fun setGoalIdForStrategy(strategy: CommonStrategy) {
        if (strategy is AutobudgetAvgCpi && strategy.goalId == null) {
            strategy.goalId = DEFAULT_CPI_GOAL_ID;
        }
    }

    fun getPackageStrategyConversionLearningData(
        strategy: GdStrategyWithConversion
    ) : CompletableFuture<GdConversionStrategyLearningStatusData>?{
        return conversionPackageStrategyLearningStatusDataLoader.get().load(strategy)
    }
}
