package ru.yandex.direct.grid.processing.service.tools;

import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.logging.log4j.util.Strings;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.geobasehelper.GeoBaseHelper;
import ru.yandex.direct.geosearch.GeosearchClient;
import ru.yandex.direct.geosearch.model.GeoObject;
import ru.yandex.direct.geosearch.model.Lang;
import ru.yandex.direct.grid.processing.model.GdCoordinates;
import ru.yandex.direct.grid.processing.model.GdGeoToolsAddress;
import ru.yandex.direct.grid.processing.model.GdGeoToolsResponse;
import ru.yandex.direct.grid.processing.model.GdPhoneCodes;
import ru.yandex.direct.grid.processing.model.GdRegion;

import static ru.yandex.direct.common.util.HttpUtil.getCurrentLocale;

@Service
@ParametersAreNonnullByDefault
public class GeoToolsService {

    private final GeoBaseHelper geoBaseHelper;
    private final GeosearchClient geosearchClient;
    private final TranslationService translationService;

    @Autowired
    public GeoToolsService(GeoBaseHelper geoBaseHelper, GeosearchClient geosearchClient,
                           TranslationService translationService) {
        this.geoBaseHelper = geoBaseHelper;
        this.geosearchClient = geosearchClient;
        this.translationService = translationService;
    }

    @Nullable
    public GdGeoToolsResponse getMetroByCoordinates(GdCoordinates coordinates) {
        List<GeoObject> geoObjects = geosearchClient.searchMetro(GeoToolsConverter.coordinatesToString(coordinates),
                getLang());
        if (geoObjects.size() > 0) {
            return GeoToolsConverter.toGdGeoToolsResponse(geoObjects.get(0));
        }
        return null;
    }

    @Nullable
    public GdGeoToolsResponse getByAddress(GdGeoToolsAddress address) {
        translationService.getLocale();
        List<GeoObject> geoObjects = geosearchClient.searchAddress(address.getAddress(), getLang());
        if (geoObjects.size() > 0) {
            return GeoToolsConverter.toGdGeoToolsResponse(geoObjects.get(0));
        }
        return null;
    }

    @Nullable
    public GdGeoToolsResponse getGeoByCoordinates(GdCoordinates coordinates) {
        return getByAddress(new GdGeoToolsAddress().withAddress(GeoToolsConverter.coordinatesToString(coordinates)));
    }

    private Lang getLang() {
        Locale locale = getCurrentLocale()
                .orElse(translationService.getLocale());
        String languageTag = locale.toLanguageTag();
        Optional<Lang> optionalLang = Stream.of(Lang.values())
                .filter(lang -> lang.name().equals(languageTag.toUpperCase()))
                .findFirst();
        return optionalLang.orElse(Lang.RU);
    }

    public GdPhoneCodes getPhoneCodeByRegionId(GdRegion region) {
        String phoneCodes = geoBaseHelper.getPhoneCodeByRegionId(Long.valueOf(region.getRegionId()));
        if (Strings.isEmpty(phoneCodes)) {
            return new GdPhoneCodes().withCodes(Collections.emptyList());
        }
        return new GdPhoneCodes().withCodes(StreamEx.split(phoneCodes, " ")
                .filter(code -> code.matches("[0-9]+"))
                .map(Integer::parseInt)
                .toList());
    }
}
