package ru.yandex.direct.grid.processing.service.trackingphone;

import java.time.Duration;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.banner.type.href.BannersUrlHelper;
import ru.yandex.direct.core.entity.calltracking.model.CalltrackingSettings;
import ru.yandex.direct.core.entity.calltracking.model.CalltrackingSettingsInput;
import ru.yandex.direct.core.entity.calltracking.model.SettingsPhone;
import ru.yandex.direct.core.entity.calltrackingphone.repository.CalltrackingPhoneRepository;
import ru.yandex.direct.core.entity.calltrackingsettings.service.CalltrackingSettingsService;
import ru.yandex.direct.core.entity.campaign.service.CampMetrikaCountersService;
import ru.yandex.direct.core.entity.clientphone.ClientPhoneService;
import ru.yandex.direct.core.entity.domain.service.DomainService;
import ru.yandex.direct.core.entity.metrika.container.CounterIdWithDomain;
import ru.yandex.direct.core.entity.metrika.repository.CalltrackingExternalRepository;
import ru.yandex.direct.core.entity.metrika.repository.CalltrackingPhonesWithoutReplacementsRepository;
import ru.yandex.direct.core.entity.metrikacounter.model.MetrikaCounterWithAdditionalInformation;
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhone;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.model.client.GdClientMetrikaCounterTruncated;
import ru.yandex.direct.grid.processing.model.trackingphone.GdCalltrackingMetrikaCountersPayload;
import ru.yandex.direct.grid.processing.model.trackingphone.GdCalltrackingOnSite;
import ru.yandex.direct.grid.processing.model.trackingphone.GdCalltrackingOnSiteByUrlPayload;
import ru.yandex.direct.grid.processing.model.trackingphone.GdCalltrackingOnSiteCounterStatus;
import ru.yandex.direct.grid.processing.model.trackingphone.GdCalltrackingOnSiteInputItem;
import ru.yandex.direct.grid.processing.model.trackingphone.GdCalltrackingOnSitePayload;
import ru.yandex.direct.grid.processing.model.trackingphone.GdCalltrackingOnSitePayloadItem;
import ru.yandex.direct.grid.processing.model.trackingphone.GdResetCalltrackingOnSitePhones;
import ru.yandex.direct.grid.processing.model.trackingphone.GdResetCalltrackingOnSitePhonesItem;
import ru.yandex.direct.grid.processing.model.trackingphone.GdSetCalltrackingOnSitePhones;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdResetCalltrackingOnSitePhone;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdResetCalltrackingOnSitePhonesPayload;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdResetCalltrackingOnSitePhonesPayloadItem;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdSetCalltrackingOnSitePhone;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdSetCalltrackingOnSitePhonesPayload;
import ru.yandex.direct.grid.processing.service.client.converter.ClientDataConverter;
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService;
import ru.yandex.direct.metrika.client.MetrikaClientException;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.utils.InterruptedRuntimeException;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static ru.yandex.direct.grid.processing.service.trackingphone.Converter.toGdSetCalltrackingOnSitePhonesPayloadItem;
import static ru.yandex.direct.grid.processing.util.ResultConverterHelper.getSuccessfullyUpdatedIds;
import static ru.yandex.direct.utils.DateTimeUtils.MSK;
import static ru.yandex.direct.utils.FunctionalUtils.filterToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;
import static ru.yandex.direct.validation.Predicates.not;

@GridGraphQLService
@ParametersAreNonnullByDefault
public class CalltrackingOnSiteDataService {
    private static final Logger logger = LoggerFactory.getLogger(CalltrackingOnSiteDataService.class);

    private static final int PHONES_LIMIT = 20;
    private static final int PHONES_WITHOUT_REPLACEMENTS_JOB_SUCCESSFUL_START_HOUR = 9;
    /**
     * Период, за который ищутся сторонние коллтрекинги на сайте рекламодателя
     */
    private static final Duration EXTERNAL_CALLTRACKING_PERIOD = Duration.ofDays(7);

    private final BannersUrlHelper bannersUrlHelper;
    private final CalltrackingOnSiteValidationService calltrackingOnSiteValidationService;
    private final CalltrackingPhoneRepository calltrackingPhoneRepository;
    private final CalltrackingPhonesWithoutReplacementsRepository calltrackingPhonesWithoutReplacementsRepository;
    private final CalltrackingExternalRepository calltrackingExternalRepository;
    private final CalltrackingSettingsService calltrackingSettingsService;
    private final CampMetrikaCountersService campMetrikaCountersService;
    private final ClientPhoneService clientPhoneService;
    private final DomainService domainService;
    private final ShardHelper shardHelper;
    private final GridValidationResultConversionService validationResultConverter;

    public CalltrackingOnSiteDataService(
            BannersUrlHelper bannersUrlHelper,
            CalltrackingOnSiteValidationService calltrackingOnSiteValidationService,
            CalltrackingPhoneRepository calltrackingPhoneRepository,
            CalltrackingPhonesWithoutReplacementsRepository calltrackingPhonesWithoutReplacementsRepository,
            CalltrackingExternalRepository calltrackingExternalRepository,
            CalltrackingSettingsService calltrackingSettingsService,
            CampMetrikaCountersService campMetrikaCountersService,
            ClientPhoneService clientPhoneService,
            DomainService domainService,
            ShardHelper shardHelper,
            GridValidationResultConversionService validationResultConverter
    ) {
        this.bannersUrlHelper = bannersUrlHelper;
        this.calltrackingOnSiteValidationService = calltrackingOnSiteValidationService;
        this.calltrackingPhoneRepository = calltrackingPhoneRepository;
        this.calltrackingPhonesWithoutReplacementsRepository = calltrackingPhonesWithoutReplacementsRepository;
        this.calltrackingExternalRepository = calltrackingExternalRepository;
        this.calltrackingSettingsService = calltrackingSettingsService;
        this.campMetrikaCountersService = campMetrikaCountersService;
        this.clientPhoneService = clientPhoneService;
        this.domainService = domainService;
        this.shardHelper = shardHelper;
        this.validationResultConverter = validationResultConverter;
    }

    GdCalltrackingMetrikaCountersPayload getCalltrackingMetrikaCounters(
            ClientId clientId, User operator, String url, Set<Long> counterIds) {
        try {
            return doGetCalltrackingMetrikaCounters(clientId, operator, url, counterIds);
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            return new GdCalltrackingMetrikaCountersPayload()
                    .withIsMetrikaAvailable(false)
                    .withCounters(Set.of());
        }
    }

    GdCalltrackingOnSitePayload getCalltrackingOnSite(
            ClientId clientId,
            User operator,
            GdCalltrackingOnSite input
    ) {
        int shard = shardHelper.getShardByClientId(clientId);
        var settings = getCalltrackingSettings(shard, clientId, input);
        return getGdCalltrackingOnSitePayload(shard, clientId, operator.getUid(), settings, true);
    }

    GdCalltrackingOnSiteByUrlPayload getCalltrackingOnSiteByUrl(
            ClientId clientId,
            User operator,
            String url
    ) {
        int shard = shardHelper.getShardByClientId(clientId);
        var settings = getCalltrackingSettings(shard, clientId, url);
        if (settings == null) {
            return new GdCalltrackingOnSiteByUrlPayload()
                    .withIsMetrikaAvailable(true)
                    .withItem(null);
        }
        var payload = getGdCalltrackingOnSitePayload(shard, clientId, operator.getUid(), List.of(settings), false);
        return new GdCalltrackingOnSiteByUrlPayload()
                .withIsMetrikaAvailable(payload.getIsMetrikaAvailable())
                .withItem(payload.getItems().get(0));
    }

    public GdCalltrackingOnSitePayload getGdCalltrackingOnSitePayload(
            int shard,
            ClientId clientId,
            Long operatorUid,
            List<CalltrackingSettings> settings,
            boolean needSuggestPhones
    ) {
        var domainIds = mapList(settings, CalltrackingSettings::getDomainId);
        var domainsById = domainService.getDomainByIdFromPpc(shard, domainIds);

        var clicksOnPhones = computeClicksOnPhones(shard, clientId, settings, domainsById, needSuggestPhones);

        var counterIds = listToSet(settings, CalltrackingSettings::getCounterId);
        var metrikaCountersContainer = computeCountersContainer(shard, clientId, operatorUid, counterIds);

        Set<String> activePhones = getCalltrackingOnSiteActivePhones(clientId, settings);
        var externalCalltrackingDomains = getExternalCalltrackingDomains(domainsById.values());
        Set<String> phonesWithoutReplacements = getPhonesWithoutReplacements(clientId, counterIds);

        List<GdCalltrackingOnSitePayloadItem> resultItems = mapList(
                settings,
                s -> toCalltrackingOnSitePayloadItem(
                        s,
                        domainsById,
                        clicksOnPhones,
                        metrikaCountersContainer.clientCounterIds,
                        metrikaCountersContainer.operatorCounterIds,
                        activePhones,
                        phonesWithoutReplacements,
                        externalCalltrackingDomains,
                        needSuggestPhones
                )
        );
        return new GdCalltrackingOnSitePayload()
                .withIsMetrikaAvailable(metrikaCountersContainer.isMetrikaAvailable)
                .withItems(resultItems);
    }

    private Set<String> getExternalCalltrackingDomains(Collection<String> domains) {
        var externalCalltrackingDomains = calltrackingExternalRepository.getExternalCalltrackingDomains(
                domains,
                EXTERNAL_CALLTRACKING_PERIOD
        );
        logger.info("Found external calltracking by domains {}", externalCalltrackingDomains);
        return externalCalltrackingDomains;
    }

    private Map<CounterIdWithDomain, Map<String, Integer>> computeClicksOnPhones(
            int shard,
            ClientId clientId,
            List<CalltrackingSettings> settings,
            Map<Long, String> domainsById,
            boolean needSuggestPhones
    ) {
        var counterIdWithDomains = listToSet(
                settings,
                s -> new CounterIdWithDomain(s.getCounterId(), domainsById.get(s.getDomainId()))
        );
        var clicksOnPhones = campMetrikaCountersService.getClicksOnPhonesByDomainWithCounterIds(counterIdWithDomains);
        var attributedClicksOnPhones = clientPhoneService.attributeClicksOnPhones(shard, clientId, clicksOnPhones);
        if (needSuggestPhones) {
            // Если к итоговому результату необходимо будет добавить номера из Метрики,
            // то предварительно нужно вычистить среди них номера Телефонии
            var telephonyPhones = filterTelephonyPhones(clicksOnPhones);
            attributedClicksOnPhones.values().forEach(e -> e.keySet().removeIf(telephonyPhones::contains));
        }
        return attributedClicksOnPhones;
    }

    private Set<String> filterTelephonyPhones(Map<CounterIdWithDomain, Map<String, Integer>> clicksOnPhones) {
        Set<String> clickedPhones = clicksOnPhones.values().stream()
                .map(Map::keySet)
                .flatMap(Collection::stream)
                .collect(Collectors.toSet());
        return calltrackingPhoneRepository.remainExistingPhones(clickedPhones);
    }

    private List<CalltrackingSettings> getCalltrackingSettings(int shard,
                                                               ClientId clientId,
                                                               GdCalltrackingOnSite input) {
        List<GdCalltrackingOnSiteInputItem> inputItems = input.getItems();

        if (inputItems.size() == 1 && inputItems.get(0).getCalltrackingSettingsId() == null) {
            calltrackingOnSiteValidationService.validateGetCalltrackingOnGetByUrl(input);

            GdCalltrackingOnSiteInputItem inputItem = inputItems.get(0);

            Long counterId = inputItem.getCounterId();
            String domain = bannersUrlHelper.extractHostFromHrefWithoutWwwOrNull(inputItem.getUrl());
            Map<String, Long> domainIdsByDomain = domainService.getOrCreateDomainIdByDomain(shard, List.of(domain));
            Long domainId = domainIdsByDomain.get(domain);

            Optional<CalltrackingSettings> calltrackingSettings =
                    calltrackingSettingsService.getByDomainId(shard, clientId, domainId);
            return calltrackingSettings
                    .map(settings -> {
                        // чтобы вернуть counterId что передал фронт
                        settings = settings.withCounterId(counterId);
                        return List.of(settings);
                    })
                    .orElse(List.of(
                            new CalltrackingSettings()
                                    .withClientId(clientId)
                                    .withCounterId(counterId)
                                    .withDomainId(domainId)
                                    .withPhonesToTrack(emptyList())
                    ));
        } else {
            calltrackingOnSiteValidationService.validateGetCalltrackingOnSiteGetById(input);
            var calltrackingSettingsIds = mapList(inputItems, GdCalltrackingOnSiteInputItem::getCalltrackingSettingsId);
            return calltrackingSettingsService.getByIds(shard, clientId, calltrackingSettingsIds);
        }
    }

    @Nullable
    private CalltrackingSettings getCalltrackingSettings(int shard, ClientId clientId, String url) {
        calltrackingOnSiteValidationService.validateGetCalltrackingOnGetByUrl(url);

        String domain = bannersUrlHelper.extractHostFromHrefWithoutWwwOrNull(url);
        Long domainId = domainService.getOrCreateDomainIdByDomain(shard, List.of(domain)).get(domain);
        return calltrackingSettingsService.getByDomainId(shard, clientId, domainId).orElse(null);
    }

    private GdCalltrackingOnSitePayloadItem toCalltrackingOnSitePayloadItem(
            CalltrackingSettings calltrackingSettings,
            Map<Long, String> domainsById,
            Map<CounterIdWithDomain, Map<String, Integer>> clicksOnPhones,
            Set<Long> clientAvailableCounterIds,
            Set<Long> operatorAvailableCounterIds,
            Set<String> activePhones,
            Set<String> phonesWithoutReplacements,
            Set<String> externalCalltrackingDomains,
            boolean needSuggestPhones
    ) {
        Map<String, LocalDateTime> phoneToLastUpdate = listToMap(calltrackingSettings.getPhonesToTrack(),
                SettingsPhone::getPhone, SettingsPhone::getCreateTime);

        String domain = domainsById.getOrDefault(calltrackingSettings.getDomainId(), "");
        var counterIdWithDomain = new CounterIdWithDomain(calltrackingSettings.getCounterId(), domain);
        Map<String, Integer> clicksByPhone = clicksOnPhones.getOrDefault(counterIdWithDomain, Map.of());

        List<String> phonesToTrack = calltrackingSettings.getPhonesToTrack()
                .stream()
                .map(SettingsPhone::getPhone)
                .collect(Collectors.toList());

        Stream<String> phones = needSuggestPhones
                ? Stream.concat(phonesToTrack.stream(), clicksByPhone.keySet().stream())
                : phonesToTrack.stream();
        List<String> sortedUniquePhones = phones
                .distinct()
                .sorted(Comparator.comparing(phone -> clicksByPhone.getOrDefault(phone, 0)).reversed())
                .limit(PHONES_LIMIT)
                .collect(Collectors.toList());

        var gdCalltrackingPhones = mapList(
                sortedUniquePhones,
                phone -> Converter.toGdCalltrackingPhoneOnSite(
                        phone,
                        phoneToLastUpdate,
                        phonesToTrack,
                        clicksByPhone,
                        activePhones,
                        phonesWithoutReplacements));

        GdCalltrackingOnSiteCounterStatus status = makeStatus(
                calltrackingSettings.getCounterId(), clientAvailableCounterIds, operatorAvailableCounterIds);

        return Converter.toGdCalltrackingOnSitePayloadItem(
                gdCalltrackingPhones,
                calltrackingSettings,
                domainsById,
                status,
                externalCalltrackingDomains.contains(domain)
        );
    }

    private static GdCalltrackingOnSiteCounterStatus makeStatus(Long counterId,
                                                                Set<Long> clientAvailableCounterIds,
                                                                Set<Long> operatorAvailableCounterIds) {
        boolean inClientCounterIds = clientAvailableCounterIds.contains(counterId);
        boolean inOperatorCounterIds = operatorAvailableCounterIds.contains(counterId);

        if (inClientCounterIds) {
            return inOperatorCounterIds ? GdCalltrackingOnSiteCounterStatus.OK :
                    GdCalltrackingOnSiteCounterStatus.NO_WRITE_PERMISSIONS;
        }

        return GdCalltrackingOnSiteCounterStatus.NOT_AVAILABLE;
    }

    GdSetCalltrackingOnSitePhonesPayload setCalltrackingOnSitePhones(
            ClientId clientId,
            User operator,
            GdSetCalltrackingOnSitePhones input
    ) {
        List<CalltrackingSettingsInput> items = mapList(
                input.getSetItems(),
                it -> new CalltrackingSettingsInput()
                        .withUrl(it.getUrl())
                        .withCounterId(it.getCounterId())
                        .withPhones(mapList(it.getCalltrackingPhones(), GdSetCalltrackingOnSitePhone::getRedirectPhone))
        );

        int shard = shardHelper.getShardByClientId(clientId);

        Map<String, String> domainByUrl = calltrackingSettingsService.getDomainByUrl(items);
        Map<String, Long> domainIdByDomain = domainService.getOrCreateDomainIdByDomain(shard, domainByUrl.values());

        MassResult<Long> massResult = calltrackingSettingsService.save(
                shard,
                clientId,
                operator.getUid(),
                items,
                domainByUrl,
                domainIdByDomain
        );

        return toGdSetCalltrackingOnSitePhonesPayload(shard, clientId, massResult, domainIdByDomain);
    }

    GdResetCalltrackingOnSitePhonesPayload resetCalltrackingOnSitePhones(
            ClientId clientId,
            GdResetCalltrackingOnSitePhones input
    ) {
        int shard = shardHelper.getShardByClientId(clientId);
        var now = LocalDateTime.now();
        List<Long> clientPhoneIds = new ArrayList<>();
        Map<Long, Set<String>> resetPhonesByCalltrackingSettingsId = new HashMap<>();
        var payloadItems = new ArrayList<GdResetCalltrackingOnSitePhonesPayloadItem>();

        var settingsIds = mapList(
                input.getResetItems(),
                GdResetCalltrackingOnSitePhonesItem::getCalltrackingSettingsId
        );

        var phonesBySettingsId = clientPhoneService.getTelephonyPhonesBySettingIds(clientId, settingsIds);

        for (var item : input.getResetItems()) {
            Long settingsId = item.getCalltrackingSettingsId();
            var resetPhones = listToSet(item.getCalltrackingPhones(), GdResetCalltrackingOnSitePhone::getRedirectPhone);
            var clientPhones = phonesBySettingsId.getOrDefault(settingsId, List.of())
                    .stream()
                    .filter(p -> resetPhones.contains(p.getPhoneNumber().getPhone()))
                    .collect(Collectors.toList());
            Set<String> dbPhones =
                    clientPhones.stream().map(p -> p.getPhoneNumber().getPhone()).collect(Collectors.toSet());

            if (resetPhones.size() != dbPhones.size()) {
                payloadItems.add(new GdResetCalltrackingOnSitePhonesPayloadItem()
                        .withCalltrackingSettingsId(settingsId)
                        .withSuccess(false));
            } else {
                clientPhoneIds.addAll(mapList(clientPhones, ClientPhone::getId));
                resetPhonesByCalltrackingSettingsId.put(settingsId, resetPhones);
            }
        }
        if (!clientPhoneIds.isEmpty()) {
            clientPhoneService.resetLastShowTime(shard, clientPhoneIds, now);
            EntryStream.of(resetPhonesByCalltrackingSettingsId)
                    .forKeyValue((settingsId, phones) ->
                            calltrackingSettingsService.resetSettings(shard, settingsId, phones, now));
            resetPhonesByCalltrackingSettingsId.keySet()
                    .forEach(settingsId -> payloadItems.add(
                            new GdResetCalltrackingOnSitePhonesPayloadItem()
                                    .withCalltrackingSettingsId(settingsId)
                                    .withSuccess(true)));
        }
        return new GdResetCalltrackingOnSitePhonesPayload()
                .withItems(payloadItems);
    }

    private GdSetCalltrackingOnSitePhonesPayload toGdSetCalltrackingOnSitePhonesPayload(
            int shard,
            ClientId clientId,
            MassResult<Long> massResult,
            Map<String, Long> domainIdByDomain
    ) {
        var settingIds = getSuccessfullyUpdatedIds(massResult, Function.identity());

        var settings = calltrackingSettingsService.getByIds(shard, clientId, settingIds);
        Map<Long, String> domainByDomainId = domainIdByDomain.entrySet()
                .stream()
                .collect(Collectors.toMap(Map.Entry::getValue, Map.Entry::getKey));

        var payloadItems = mapList(settings, s -> toGdSetCalltrackingOnSitePhonesPayloadItem(domainByDomainId, s));
        return new GdSetCalltrackingOnSitePhonesPayload()
                .withItems(payloadItems)
                .withValidationResult(validationResultConverter.buildGridValidationResult(massResult.getValidationResult()));
    }

    /**
     * Получение списка номеров, к которым привязаны подменники Телефонии для коллтрекинга на сайте
     *
     * @param clientId id клиента
     * @param settings настройки связанные с номерами
     * @return список номеров, к которым привязаны подменники Телефонии для коллтрекинга на сайте
     */
    private Set<String> getCalltrackingOnSiteActivePhones(
            ClientId clientId,
            List<CalltrackingSettings> settings) {
        List<ClientPhone> clientPhones = clientPhoneService.getTelephonyPhonesBySettings(clientId, settings);
        return clientPhones.stream().map(phone -> phone.getPhoneNumber().getPhone()).collect(Collectors.toSet());
    }

    private GdCalltrackingMetrikaCountersPayload doGetCalltrackingMetrikaCounters(
            ClientId clientId, User operator, String url, Set<Long> counterIds) {
        Long operatorUid = operator.getUid();

        CampMetrikaCountersService.CountersContainer countersContainer =
                campMetrikaCountersService.getAvailableCountersByClientAndOperatorUid(
                        clientId,
                        operatorUid,
                        counterIds);
        List<MetrikaCounterWithAdditionalInformation> filteredCounters =
                countersContainer.getClientAvailableCounters().stream()
                        .filter(counter -> countersContainer.getOperatorEditableCounterIds().contains(counter.getId()))
                        // Filter by input ids
                        .filter(counter -> counterIds.contains(counter.getId()))
                        // Initial sort order - by counter id
                        .sorted(Comparator.comparingLong(MetrikaCounterWithAdditionalInformation::getId))
                        .collect(Collectors.toList());
        List<Long> filteredCounterIds = mapList(filteredCounters, MetrikaCounterWithAdditionalInformation::getId);

        // Get domain
        String domain = bannersUrlHelper.extractHostFromHrefWithoutWwwOrNull(url);
        if (domain == null) {
            return toGdCalltrackingMetrikaCountersPayload(filteredCounters);
        }
        // Get clicks on phones and sort
        Map<Long, Map<String, Integer>> clicksByPhonesByCounterId =
                campMetrikaCountersService.getClicksOnPhonesByCounters(domain, filteredCounterIds);
        Map<Long, Integer> clicksByCounterId = clicksByPhonesByCounterId.entrySet()
                .stream()
                .collect(Collectors.toMap(
                        Map.Entry::getKey,
                        entry -> entry.getValue()
                                .values()
                                .stream()
                                .reduce(0, Integer::sum)
                ));

        List<MetrikaCounterWithAdditionalInformation> sortedCounters = StreamEx.of(filteredCounters)
                .reverseSorted(Comparator.comparingInt(counter ->
                        clicksByCounterId.getOrDefault(counter.getId(), 0)))
                .toList();
        int shard = shardHelper.getShardByClientId(clientId);
        List<CalltrackingSettings> calltrackingSettingsList =
                calltrackingSettingsService.getByDomain(shard, clientId, List.of(domain));
        if (!calltrackingSettingsList.isEmpty()) {
            Set<Long> calltrackingCounters = calltrackingSettingsList.stream()
                    .map(CalltrackingSettings::getCounterId).collect(Collectors.toSet());
            CampMetrikaCountersService.CountersContainer container =
                    campMetrikaCountersService.getAvailableCountersByClientAndOperatorUid(
                            clientId,
                            operatorUid,
                            calltrackingCounters);
            List<MetrikaCounterWithAdditionalInformation> countersFromCalltracking =
                    StreamEx.of(container.getClientAvailableCounters())
                            .filter(counter -> calltrackingCounters.contains(counter.getId()))
                            .filter(counter -> !sortedCounters.contains(counter))
                            .append(sortedCounters)
                            .toList();
            return toGdCalltrackingMetrikaCountersPayload(countersFromCalltracking);
        }
        return toGdCalltrackingMetrikaCountersPayload(sortedCounters);
    }

    private Set<String> getPhonesWithoutReplacements(ClientId clientId, Set<Long> counterIds) {
        Long timestamp = getTimestampForQuery();
        Set<String> phones = calltrackingPhonesWithoutReplacementsRepository.getPhonesWithoutReplacements(
                clientId,
                counterIds,
                timestamp
        );
        if (!phones.isEmpty()) {
            logger.info("Phones without replacements from YT table : {}", phones);
        }
        return phones;
    }

    /**
     * Вычисляет timestamp для запроса за телефонами, у которых за день не было ни одной успешной подмены номера
     * до 9:00 МСК смотрим на данные за позавчера
     * после 9:00 МСК смотрим на данные за вчера
     * <p>
     * Данные о телефонах, у которых за день не было ни одной успешной подмены номера вычисляются джобой
     * PopulatePhonesWithoutReplacementsDynTableJob
     * в 9:00 МСК джоба успешно запускается и записывает данные за вчера
     */
    private Long getTimestampForQuery() {
        var now = ZonedDateTime.now(MSK);
        if (now.getHour() < PHONES_WITHOUT_REPLACEMENTS_JOB_SUCCESSFUL_START_HOUR) {
            now = now.minusDays(2);
        } else {
            now = now.minusDays(1);
        }
        return now.toLocalDate().atStartOfDay(ZoneOffset.UTC).toEpochSecond();
    }

    private GdCalltrackingMetrikaCountersPayload toGdCalltrackingMetrikaCountersPayload(
            List<MetrikaCounterWithAdditionalInformation> countersInfo) {
        Set<GdClientMetrikaCounterTruncated> counters = countersInfo.stream()
                .map(ClientDataConverter::toGdClientMetrikaCounterTruncated)
                .collect(Collectors.toCollection(LinkedHashSet::new));
        return new GdCalltrackingMetrikaCountersPayload()
                .withIsMetrikaAvailable(true)
                .withCounters(counters);
    }

    MetrikaCountersContainer computeCountersContainer(
            int shard,
            ClientId clientId,
            long operatorUid,
            Set<Long> counterIds
    ) {
        try {
            CampMetrikaCountersService.CountersContainer countersContainer =
                    campMetrikaCountersService.getAvailableCountersByClientAndOperatorUid(
                            clientId,
                            operatorUid,
                            counterIds);
            Set<Long> clientCounterIds = mapSet(countersContainer.getClientAvailableCounters(),
                    MetrikaCounterWithAdditionalInformation::getId);
            Set<Long> operatorCounterIds = countersContainer.getOperatorEditableCounterIds();
            Set<Long> availableCounterIds = filterToSet(counterIds, clientCounterIds::contains);
            Set<Long> notAvailableCounterIds = filterToSet(counterIds, not(clientCounterIds::contains));
            // нужно для того чтобы своевременно обрабатывать случаи,
            // когда у клиента отвалились все доступы к счетчику
            calltrackingSettingsService.updateCounterAvailability(
                    shard,
                    clientId,
                    availableCounterIds,
                    notAvailableCounterIds
            );
            return new MetrikaCountersContainer(clientCounterIds, operatorCounterIds, true);
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            logger.warn(
                    "Got an exception when querying for metrika counters for clientID = {} and operator uid = {}",
                    clientId, operatorUid, e
            );
            return new MetrikaCountersContainer(emptySet(), emptySet(), false);
        }
    }

    private static class MetrikaCountersContainer {
        final Set<Long> clientCounterIds;
        final Set<Long> operatorCounterIds;
        final boolean isMetrikaAvailable;

        public MetrikaCountersContainer(
                Set<Long> clientCounterIds,
                Set<Long> operatorCounterIds,
                boolean isMetrikaAvailable
        ) {
            this.clientCounterIds = clientCounterIds;
            this.operatorCounterIds = operatorCounterIds;
            this.isMetrikaAvailable = isMetrikaAvailable;
        }
    }
}
