package ru.yandex.direct.grid.processing.service.trackingphone;

import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLContext;
import io.leangen.graphql.annotations.GraphQLMutation;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.annotations.EnableLoggingOnValidationIssues;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.client.GdClient;
import ru.yandex.direct.grid.processing.model.trackingphone.GdCalltrackingMetrikaCounters;
import ru.yandex.direct.grid.processing.model.trackingphone.GdCalltrackingMetrikaCountersPayload;
import ru.yandex.direct.grid.processing.model.trackingphone.GdCalltrackingOnSite;
import ru.yandex.direct.grid.processing.model.trackingphone.GdCalltrackingOnSiteByUrlPayload;
import ru.yandex.direct.grid.processing.model.trackingphone.GdCalltrackingOnSiteCampaigns;
import ru.yandex.direct.grid.processing.model.trackingphone.GdCalltrackingOnSiteCampaignsPayload;
import ru.yandex.direct.grid.processing.model.trackingphone.GdCalltrackingOnSitePayload;
import ru.yandex.direct.grid.processing.model.trackingphone.GdResetCalltrackingOnSitePhones;
import ru.yandex.direct.grid.processing.model.trackingphone.GdSetCalltrackingOnSitePhones;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdResetCalltrackingOnSitePhonesPayload;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdSetCalltrackingOnSitePhonesPayload;

@GridGraphQLService
@ParametersAreNonnullByDefault
public class CalltrackingOnSiteGraphQLService {
    public static final String CALLTRACKING_ON_SITE_RESOLVER_NAME = "calltrackingOnSite";
    public static final String CALLTRACKING_ON_SITE_BY_URL_RESOLVER_NAME = "calltrackingOnSiteByUrl";
    public static final String SET_CALLTRACKING_ON_SITE_PHONES = "setCalltrackingOnSitePhones";
    public static final String RESET_CALLTRACKING_ON_SITE_PHONES = "resetCalltrackingOnSitePhones";
    public static final String GET_CALLTRACKING_METRIKA_COUNTERS_NAME = "calltrackingMetrikaCounters";
    public static final String GET_CALLTRACKING_ON_SITE_CAMPAIGNS_NAME = "calltrackingOnSiteCampaigns";

    private final CalltrackingOnSiteDataService calltrackingOnSiteDataService;
    private final CalltrackingOnSiteCampaignDataService calltrackingOnSiteCampaignDataService;

    @Autowired
    public CalltrackingOnSiteGraphQLService(
            CalltrackingOnSiteDataService calltrackingOnSiteDataService,
            CalltrackingOnSiteCampaignDataService calltrackingOnSiteCampaignDataService
    ) {
        this.calltrackingOnSiteDataService = calltrackingOnSiteDataService;
        this.calltrackingOnSiteCampaignDataService = calltrackingOnSiteCampaignDataService;
    }

    /**
     * Ручка принимает на вход GdCalltrackingOnSite, который в свою очередь состоит из GdCalltrackingOnSiteInputItem.
     * В балковом режиме ручка работает только для случая, когда для всех GdCalltrackingOnSiteInputItem указан
     * calltrackingSettingsId. Для того чтобы получить настройки по url + counterId необходим передать балку
     * из одного элемента с этими данными.
     */
    @GraphQLNonNull
    @GraphQLQuery(name = CALLTRACKING_ON_SITE_RESOLVER_NAME)
    public GdCalltrackingOnSitePayload getCalltrackingOnSite(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdCalltrackingOnSite input) {
        ClientId clientId = ClientId.fromLong(client.getInfo().getId());
        User operator = context.getOperator();

        return calltrackingOnSiteDataService.getCalltrackingOnSite(clientId, operator, input);
    }

    /**
     * Возвращает сохраненные настройки коллтрекинга на сайте по URL без саджестов номеров из Метрики
     */
    @GraphQLNonNull
    @GraphQLQuery(name = CALLTRACKING_ON_SITE_BY_URL_RESOLVER_NAME)
    public GdCalltrackingOnSiteByUrlPayload getCalltrackingOnSiteByUrl(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client,
            @GraphQLNonNull @GraphQLArgument(name = "url") String url) {
        ClientId clientId = ClientId.fromLong(client.getInfo().getId());
        User operator = context.getOperator();

        return calltrackingOnSiteDataService.getCalltrackingOnSiteByUrl(clientId, operator, url);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = GET_CALLTRACKING_METRIKA_COUNTERS_NAME)
    public GdCalltrackingMetrikaCountersPayload getCalltrackingMetrikaCounters(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client,
            @GraphQLArgument(name = "input") GdCalltrackingMetrikaCounters input) {
        ClientId clientId = context.getSubjectUser().getClientId();
        User operator = context.getOperator();
        String url = input.getUrl();
        Set<Long> counterIds = input.getCounterIds();
        return calltrackingOnSiteDataService.getCalltrackingMetrikaCounters(clientId, operator, url, counterIds);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = GET_CALLTRACKING_ON_SITE_CAMPAIGNS_NAME)
    public GdCalltrackingOnSiteCampaignsPayload calltrackingOnSiteCampaigns(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client,
            @GraphQLArgument(name = "input") GdCalltrackingOnSiteCampaigns input) {
        ClientId clientId = ClientId.fromLong(client.getInfo().getId());
        return calltrackingOnSiteCampaignDataService.calltrackingOnSiteCampaigns(clientId, input);
    }

    @EnableLoggingOnValidationIssues
    @GraphQLNonNull
    @GraphQLMutation(name = SET_CALLTRACKING_ON_SITE_PHONES)
    @PreAuthorizeWrite
    public GdSetCalltrackingOnSitePhonesPayload setCalltrackingOnSitePhones(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdSetCalltrackingOnSitePhones input
    ) {
        ClientId clientId = context.getSubjectUser().getClientId();
        User operator = context.getOperator();
        return calltrackingOnSiteDataService.setCalltrackingOnSitePhones(clientId, operator, input);
    }

    @EnableLoggingOnValidationIssues
    @GraphQLNonNull
    @GraphQLMutation(name = RESET_CALLTRACKING_ON_SITE_PHONES)
    @PreAuthorizeWrite
    public GdResetCalltrackingOnSitePhonesPayload resetCalltrackingOnSitePhones(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdResetCalltrackingOnSitePhones input
    ) {
        ClientId clientId = context.getSubjectUser().getClientId();
        return calltrackingOnSiteDataService.resetCalltrackingOnSitePhones(clientId, input);
    }
}
