package ru.yandex.direct.grid.processing.service.trackingphone;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nonnull;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.calltracking.model.CalltrackingSettings;
import ru.yandex.direct.core.entity.clientphone.ClientPhoneUtils;
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhone;
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhoneType;
import ru.yandex.direct.core.entity.trackingphone.model.PhoneNumber;
import ru.yandex.direct.core.entity.vcard.model.Phone;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.GdPhone;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.GdPhoneWithId;
import ru.yandex.direct.grid.processing.model.trackingphone.GdCalltrackingOnSiteCounterStatus;
import ru.yandex.direct.grid.processing.model.trackingphone.GdCalltrackingOnSitePayloadItem;
import ru.yandex.direct.grid.processing.model.trackingphone.GdCalltrackingPhoneOnSite;
import ru.yandex.direct.grid.processing.model.trackingphone.GdGetPhonesPayload;
import ru.yandex.direct.grid.processing.model.trackingphone.GdGetPhonesPayloadItem;
import ru.yandex.direct.grid.processing.model.trackingphone.GdPhoneType;
import ru.yandex.direct.grid.processing.model.trackingphone.GdSetCalltrackingOnSitePhonesItem;
import ru.yandex.direct.grid.processing.model.trackingphone.GdTrackingPhone;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdAddClientPhone;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdSetCalltrackingOnSitePhonesPayloadItem;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdUpdateClientPhone;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdUpdateTelephonyRedirectPhone;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdUpdateTelephonyRedirectPhones;
import ru.yandex.direct.validation.result.MappingPathNodeConverter;
import ru.yandex.direct.validation.result.PathNodeConverter;

import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class Converter {

    public static final PathNodeConverter CLIENT_PHONE_PATH_CONVERTER =
            MappingPathNodeConverter.builder(ClientPhone.class.getName())
                    .replace(
                            ClientPhone.PHONE_NUMBER.name(),
                            GdUpdateTelephonyRedirectPhone.REDIRECT_PHONE.name())
                    .build();
    public static final PathNodeConverter CALLTRACKING_SETTINGS_PATH_CONVERTER =
            MappingPathNodeConverter.builder(CalltrackingSettings.class.getName())
                    .replace(
                            CalltrackingSettings.PHONES_TO_TRACK.name(),
                            GdSetCalltrackingOnSitePhonesItem.CALLTRACKING_PHONES.name())
                    .replace(
                            CalltrackingSettings.DOMAIN_ID.name(),
                            GdSetCalltrackingOnSitePhonesItem.URL.name())
                    .build();

    private static final Duration NEW_PHONE_MAX_AGE = Duration.ofHours(3);

    private Converter() {
    }

    static GdGetPhonesPayload toGdGetPhonesPayload(
            Map<Long, List<ClientPhone>> orgPhonesByPermalink,
            List<ClientPhone> telephonyPhones,
            List<ClientPhone> manualPhones
    ) {
        return new GdGetPhonesPayload()
                .withManualPhones(
                        mapList(manualPhones, Converter::toGdTrackingPhone)
                )
                .withOrganizationPhones(
                        toGdGetOrgPhonesPayloadItem(orgPhonesByPermalink)
                )
                .withTelephonyPhones(
                        toGdGetTelephonyPhonesPayloadItem(telephonyPhones)
                );
    }

    static List<GdGetPhonesPayloadItem> toGdGetTelephonyPhonesPayloadItem(List<ClientPhone> phones) {
        return StreamEx.of(phones)
                .mapToEntry(ClientPhone::getPermalinkId, Converter::toGdTrackingPhone)
                .collapseKeys()
                .mapKeyValue((id, list) -> new GdGetPhonesPayloadItem().withPermalinkId(id).withRowset(list))
                .toList();
    }

    static List<GdGetPhonesPayloadItem> toGdGetOrgPhonesPayloadItem(Map<Long, List<ClientPhone>> orgPhonesByPermalink) {
        return EntryStream.of(orgPhonesByPermalink)
                .mapKeyValue((permalink, orgPhones) -> {
                    var gdPhones = mapList(orgPhones, Converter::toGdTrackingPhone);
                    return new GdGetPhonesPayloadItem().withPermalinkId(permalink).withRowset(gdPhones);
                })
                .toList();
    }

    static GdTrackingPhone toGdTrackingPhone(ClientPhone phone) {
        PhoneNumber phoneNumber;
        String redirectPhone;
        if (phone.getPhoneType() == ClientPhoneType.TELEPHONY) {
            phoneNumber = phone.getTelephonyPhone();
            redirectPhone = phone.getPhoneNumber().getPhone();
        } else {
            phoneNumber = phone.getPhoneNumber();
            redirectPhone = null;
        }
        return new GdTrackingPhone()
                .withPhone(ifNotNull(phoneNumber, PhoneNumber::getPhone))
                .withExtension(ifNotNull(phoneNumber, PhoneNumber::getExtension))
                .withPhoneId(phone.getId())
                .withPhoneType(GdPhoneType.valueOf(phone.getPhoneType().toString()))
                .withComment(phone.getComment())
                .withRedirectPhone(redirectPhone)
                .withFormattedPhone(ifNotNull(phoneNumber, Converter::toGdPhone))
                .withFormattedRedirectPhone(ifNotNull(redirectPhone, Converter::toGdPhone))
                .withIsHidden(nvl(phone.getIsHidden(), false));
    }

    static ClientPhone toManualPhone(ClientId clientId, GdAddClientPhone input) {
        return new ClientPhone()
                .withClientId(clientId)
                .withPhoneNumber(new PhoneNumber().withPhone(input.getPhone()).withExtension(input.getExtension()))
                .withComment(input.getComment());
    }

    static ClientPhone toManualPhone(ClientId clientId, GdUpdateClientPhone input) {
        return new ClientPhone()
                .withId(input.getPhoneId())
                .withClientId(clientId)
                .withPhoneNumber(new PhoneNumber().withPhone(input.getPhone()).withExtension(input.getExtension()))
                .withComment(input.getComment());
    }

    static ClientPhone toTelephonyPhone(ClientId clientId, GdUpdateTelephonyRedirectPhone input) {
        return new ClientPhone()
                .withId(input.getPhoneId())
                .withClientId(clientId)
                .withPhoneType(ClientPhoneType.TELEPHONY)
                .withPhoneNumber(new PhoneNumber().withPhone(input.getRedirectPhone()));
    }

    static List<ClientPhone> toTelephonyPhones(ClientId clientId, GdUpdateTelephonyRedirectPhones input) {
        List<ClientPhone> clientPhones = new ArrayList<>();
        for (GdUpdateTelephonyRedirectPhone updateItem: input.getUpdateItems()) {
            clientPhones.add(toTelephonyPhone(clientId, updateItem));
        }
        return clientPhones;
    }

    static GdCalltrackingOnSitePayloadItem toGdCalltrackingOnSitePayloadItem(
            List<GdCalltrackingPhoneOnSite> gdCalltrackingPhones,
            CalltrackingSettings calltrackingSettings,
            Map<Long, String> domainsById,
            GdCalltrackingOnSiteCounterStatus status,
            boolean hasExternalCalltracking
    ) {
        return new GdCalltrackingOnSitePayloadItem()
                .withCalltrackingSettingsId(calltrackingSettings.getCalltrackingSettingsId())
                .withDomain(domainsById.get(calltrackingSettings.getDomainId()))
                .withCounterId(calltrackingSettings.getCounterId())
                .withCounterStatus(status)
                .withCalltrackingPhones(gdCalltrackingPhones)
                .withHasExternalCalltracking(hasExternalCalltracking);
    }

    static GdCalltrackingPhoneOnSite toGdCalltrackingPhoneOnSite(
            String phone,
            Map<String, LocalDateTime> phoneToLastUpdate,
            List<String> phonesToTrack,
            Map<String, Integer> clicksByPhone,
            Set<String> activePhones,
            Set<String> phonesWithoutReplacements
    ) {
        return new GdCalltrackingPhoneOnSite()
                .withRedirectPhone(phone)
                .withFormattedRedirectPhone(toGdPhone(phone))
                .withHasClicks(clicksByPhone.containsKey(phone))
                .withIsCalltrackingOnSiteEnabled(phonesToTrack.contains(phone))
                .withIsCalltrackingOnSiteActive(activePhones.contains(phone))
                .withIsNew(getIsNew(phone, phoneToLastUpdate))
                .withHasNotReplacements(phonesWithoutReplacements.contains(phone));
    }

    static GdSetCalltrackingOnSitePhonesPayloadItem toGdSetCalltrackingOnSitePhonesPayloadItem(
            Map<Long, String> domainByDomainId,
            CalltrackingSettings settings
    ) {
        return new GdSetCalltrackingOnSitePhonesPayloadItem()
                .withCalltrackingSettingsId(settings.getCalltrackingSettingsId())
                .withDomain(domainByDomainId.get(settings.getDomainId()));
    }

    public static GdPhoneWithId toGdPhoneWithId(@Nonnull PhoneNumber phoneNumber, long id) {
        return new GdPhoneWithId()
                .withId(id)
                .withPhone(toGdPhone(phoneNumber));
    }

    private static GdPhone toGdPhone(@Nonnull PhoneNumber phoneNumber) {
        Phone phone = ClientPhoneUtils.toPhone(phoneNumber);
        return toGdPhone(phone);
    }

    static GdPhone toGdPhone(String phoneNumber) {
        Phone phone = ClientPhoneUtils.toPhone(phoneNumber, null);
        return toGdPhone(phone);
    }

    private static GdPhone toGdPhone(Phone phone) {
        return new GdPhone()
                .withCountryCode(phone.getCountryCode())
                .withCityCode(phone.getCityCode())
                .withPhoneNumber(phone.getPhoneNumber())
                .withExtension(phone.getExtension());
    }

    private static boolean getIsNew(String phone, Map<String, LocalDateTime> phoneToLastUpdate) {
        if (!phoneToLastUpdate.containsKey(phone)) {
            return false;
        }
        return phoneToLastUpdate.get(phone).isAfter(LocalDateTime.now().minus(NEW_PHONE_MAX_AGE));
    }
}
