package ru.yandex.direct.grid.processing.service.trackingphone;

import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

import org.dataloader.DataLoader;
import org.dataloader.MappedBatchLoaderWithContext;
import org.springframework.context.annotation.Scope;
import org.springframework.context.annotation.ScopedProxyMode;
import org.springframework.stereotype.Component;
import org.springframework.web.context.WebApplicationContext;

import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.model.trackingphone.GdTrackingPhone;
import ru.yandex.direct.grid.processing.service.dataloader.GridBatchingDataLoader;
import ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

/**
 * {@link DataLoader} для отложенной batch'евой загрузки привязанных подменных номеров
 *
 * @see GridBatchingDataLoader
 * @see ru.yandex.direct.grid.processing.service.dataloader.GridDataLoaderRegistry
 */
@Component
// DataLoader'ы хранят состояние, поэтому жить должны в рамках запроса
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
public class TrackingPhoneDataLoader extends GridBatchingDataLoader<Long, GdTrackingPhone> {

    public TrackingPhoneDataLoader(GridContextProvider gridContextProvider, TrackingPhoneDataService dataService) {
        dataLoader = mappedDataLoader(gridContextProvider, getBatchLoadFunction(dataService));
    }

    private MappedBatchLoaderWithContext<Long, GdTrackingPhone> getBatchLoadFunction(
            TrackingPhoneDataService dataService
    ) {
        return (ids, environment) -> {
            GridGraphQLContext context = environment.getContext();
            GdClientInfo queriedClient = context.getQueriedClient();
            checkNotNull(queriedClient, "queriedClient should be set in gridContext");

            ClientId clientId = ClientId.fromLong(queriedClient.getId());
            Map<Long, GdTrackingPhone> phonesById = dataService.getGdTrackingPhonesById(clientId, ids);

            Map<Long, GdTrackingPhone> result = listToMap(ids, Function.identity(), phonesById::get);
            return CompletableFuture.completedFuture(result);
        };
    }
}
