package ru.yandex.direct.grid.processing.service.trackingphone;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Service;

import ru.yandex.altay.model.language.LanguageOuterClass;
import ru.yandex.direct.core.entity.clientphone.ClientPhoneReplaceService;
import ru.yandex.direct.core.entity.clientphone.ClientPhoneService;
import ru.yandex.direct.core.entity.organizations.service.OrganizationService;
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhone;
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhoneType;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.organizations.GdOrganization;
import ru.yandex.direct.grid.processing.model.trackingphone.GdGetPhonesPayload;
import ru.yandex.direct.grid.processing.model.trackingphone.GdTrackingPhone;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdAddClientPhone;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdAddClientPhonePayload;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdDeleteClientPhone;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdDeleteClientPhonePayload;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdUpdateAdsPhone;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdUpdateAdsPhonePayload;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdUpdateClientPhone;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdUpdateClientPhonePayload;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdUpdateTelephonyRedirectPhone;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdUpdateTelephonyRedirectPhonePayload;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdUpdateTelephonyRedirectPhones;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdUpdateTelephonyRedirectPhonesPayload;
import ru.yandex.direct.grid.processing.service.organizations.OrganizationsConverter;
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.organizations.swagger.OrganizationApiInfo;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.result.Path;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.grid.processing.service.trackingphone.Converter.toGdGetPhonesPayload;
import static ru.yandex.direct.grid.processing.service.trackingphone.Converter.toManualPhone;
import static ru.yandex.direct.grid.processing.service.trackingphone.Converter.toTelephonyPhone;
import static ru.yandex.direct.grid.processing.service.trackingphone.Converter.toTelephonyPhones;
import static ru.yandex.direct.grid.processing.util.ResponseConverter.getSuccessfullyResults;
import static ru.yandex.direct.organizations.swagger.OrganizationsClient.getLanguageByName;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.emptyPath;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

@Service
@ParametersAreNonnullByDefault
public class TrackingPhoneDataService {
    private final ClientPhoneService clientPhoneService;
    private final ClientPhoneReplaceService clientPhoneReplaceService;
    private final OrganizationService organizationService;
    private final TrackingPhoneValidationService trackingPhoneValidationService;
    private final GridValidationService gridValidationService;
    private final GridValidationResultConversionService validationResultConverter;

    public TrackingPhoneDataService(ClientPhoneService clientPhoneService,
                                    ClientPhoneReplaceService clientPhoneReplaceService,
                                    OrganizationService organizationService,
                                    TrackingPhoneValidationService trackingPhoneValidationService,
                                    GridValidationService gridValidationService,
                                    GridValidationResultConversionService validationResultConverter) {
        this.clientPhoneService = clientPhoneService;
        this.clientPhoneReplaceService = clientPhoneReplaceService;
        this.organizationService = organizationService;
        this.trackingPhoneValidationService = trackingPhoneValidationService;
        this.gridValidationService = gridValidationService;
        this.validationResultConverter = validationResultConverter;
    }

    /**
     * Получаем три типа номеров:
     * - номера организации - отправляется запрос в API Справочника,
     * оттуда получаем номер Организации (по {@code permalinkId}) и счетчики метрики,
     * сохраняем эти номера в таблицу {@code client_phones}
     * - персональные номера клиента - номера с типом {@code MANUAL} {@code client_phones}
     * - номера из Телефонии - номера с типом {@code TELEPHONY} из {@code client_phones}
     */
    GdGetPhonesPayload getClientPhones(ClientId clientId, List<Long> permalinkIds) {
        Map<Long, List<ClientPhone>> orgPhonesByPermalink =
                clientPhoneService.getAndSaveOrganizationPhones(clientId, permalinkIds);

        List<ClientPhone> clientPhones = clientPhoneService.getAllClientPhones(clientId, permalinkIds);

        List<ClientPhone> telephonyPhones = clientPhoneService.getAndSaveTelephonyPhones(clientId, permalinkIds);

        List<ClientPhone> manualPhones = filterList(clientPhones, p -> p.getPhoneType() == ClientPhoneType.MANUAL);

        return toGdGetPhonesPayload(orgPhonesByPermalink, telephonyPhones, manualPhones);
    }

    List<GdTrackingPhone> getLinkedPhones(ClientId clientId, List<Long> bannerIds) {
        Map<Long, Long> phoneIdsByBannerIds = clientPhoneService.getPhoneIdsByBannerIds(clientId, bannerIds);
        List<ClientPhone> clientPhones = clientPhoneService.getByPhoneIds(clientId, phoneIdsByBannerIds.values());
        return mapList(clientPhones, Converter::toGdTrackingPhone);
    }

    List<GdOrganization> getLinkedOrganizations(ClientId clientId, List<Long> bannerIds) {
        Map<Long, Long> permalinkIdsByBannerIds = organizationService.getPermalinkIdsByBannerIds(clientId, bannerIds);
        LanguageOuterClass.Language language = getLanguageByName(LocaleContextHolder.getLocale().getLanguage())
                .orElse(LanguageOuterClass.Language.EN);
        List<OrganizationApiInfo> organizations =
                organizationService.getClientOrganizationsByIds(permalinkIdsByBannerIds.values(), language);
        return mapList(organizations, OrganizationsConverter::toGdOrganization);
    }

    Map<Long, GdTrackingPhone> getGdTrackingPhonesById(ClientId clientId, Collection<Long> phoneIds) {
        List<ClientPhone> phones = clientPhoneService.getByPhoneIds(clientId, phoneIds);
        return StreamEx.of(phones)
                .mapToEntry(ClientPhone::getId, Function.identity())
                .mapValues(Converter::toGdTrackingPhone)
                .toMap();
    }

    GdAddClientPhonePayload addClientPhone(ClientId clientId, GdAddClientPhone input) {
        ClientPhone clientPhone = toManualPhone(clientId, input);
        Result<Long> result = clientPhoneService.addManualClientPhone(clientPhone);
        if (result.isSuccessful()) {
            return new GdAddClientPhonePayload()
                    .withPhoneId(result.getResult());
        }
        GdValidationResult validationResult = gridValidationService.getValidationResult(result, emptyPath());
        return new GdAddClientPhonePayload()
                .withValidationResult(validationResult);
    }

    GdUpdateClientPhonePayload updateClientPhone(ClientId clientId, GdUpdateClientPhone input) {
        ClientPhone clientPhone = toManualPhone(clientId, input);
        Result<Long> result = clientPhoneService.updateClientPhone(clientPhone);
        if (result.isSuccessful()) {
            return new GdUpdateClientPhonePayload()
                    .withPhoneId(result.getResult());
        }
        GdValidationResult validationResult = gridValidationService.getValidationResult(result, emptyPath());
        return new GdUpdateClientPhonePayload()
                .withValidationResult(validationResult);
    }

    GdUpdateAdsPhonePayload updateAdsPhone(ClientId clientId, Long operatorUid, GdUpdateAdsPhone input) {
        trackingPhoneValidationService.validateUpdateAdsPhone(clientId, input);
        Long phoneId = input.getPhoneId();
        List<Long> bannerIds = input.getAdIds();
        List<Long> permalinkIds = input.getPermalinkIds();
        MassResult<Long> updateResult = clientPhoneService.updateBannersPhone(clientId, operatorUid, phoneId, bannerIds,
                permalinkIds);
        GdValidationResult validationResult =
                gridValidationService.getValidationResult(updateResult, path());
        List<Long> updatedAds = getSuccessfullyResults(updateResult, Function.identity());
        return new GdUpdateAdsPhonePayload()
                .withUpdatedAdIds(updatedAds)
                .withValidationResult(validationResult);
    }

    GdDeleteClientPhonePayload deleteClientPhone(Long operatorUid, User client, GdDeleteClientPhone input) {
        UidAndClientId uidAndClientId = UidAndClientId.of(client.getUid(), client.getClientId());
        return input.getReplacePhoneId() == null
                ? deletePhone(client.getClientId(), input)
                : deleteAndReplacePhone(uidAndClientId, operatorUid, input);
    }

    /**
     * Удалить {@link GdDeleteClientPhone#getPhoneIds()}
     */
    private GdDeleteClientPhonePayload deletePhone(ClientId clientId, GdDeleteClientPhone input) {
        MassResult<Long> result = clientPhoneService.delete(clientId, input.getPhoneIds());
        List<Long> deletedPhoneIds = getSuccessfullyResults(result, Function.identity());
        Path path = path(field(GdDeleteClientPhone.PHONE_IDS));
        GdValidationResult vr = gridValidationService.getValidationResult(result, path);
        return new GdDeleteClientPhonePayload()
                .withDeletePhoneIds(deletedPhoneIds)
                .withValidationResult(vr);
    }

    /**
     * Удалить номера {@link GdDeleteClientPhone#getPhoneIds()},
     * и заменить их на {@link GdDeleteClientPhone#getReplacePhoneId()}
     */
    private GdDeleteClientPhonePayload deleteAndReplacePhone(
            UidAndClientId uidAndClientId,
            Long operatorUid,
            GdDeleteClientPhone input
    ) {
        ClientId clientId = uidAndClientId.getClientId();
        trackingPhoneValidationService.validateReplaceTrackingPhone(clientId, input);
        List<Long> phoneIds = input.getPhoneIds();
        Long replacePhoneId = input.getReplacePhoneId();
        MassResult<Long> replaceResult =
                clientPhoneReplaceService.replaceTrackingPhone(uidAndClientId, operatorUid, phoneIds, replacePhoneId);
        Path replacePath = path(field(GdDeleteClientPhone.REPLACE_PHONE_ID));
        var replaceVr = gridValidationService.getValidationResult(replaceResult, replacePath);
        if (replaceResult.getValidationResult().hasAnyErrors()) {
            return new GdDeleteClientPhonePayload()
                    .withUpdatedAdIds(emptyList())
                    .withDeletePhoneIds(emptyList())
                    .withValidationResult(replaceVr);
        }
        MassResult<Long> deleteResult = clientPhoneService.delete(clientId, phoneIds);
        Path deletePath = path(field(GdDeleteClientPhone.PHONE_IDS));
        var deleteVr = gridValidationService.getValidationResult(deleteResult, deletePath);
        List<Long> deletedPhoneIds = getSuccessfullyResults(deleteResult, Function.identity());
        return new GdDeleteClientPhonePayload()
                .withUpdatedAdIds(deletedPhoneIds)
                .withDeletePhoneIds(phoneIds)
                .withValidationResult(deleteVr);
    }

    GdUpdateTelephonyRedirectPhonePayload updateTelephonyRedirectPhone(ClientId clientId,
                                                                       GdUpdateTelephonyRedirectPhone input) {

        ClientPhone clientPhone = toTelephonyPhone(clientId, input);
        Result<Long> result = clientPhoneService.updateTelephonyRedirectPhone(clientPhone);

        if (result.isSuccessful()) {
            return new GdUpdateTelephonyRedirectPhonePayload()
                    .withPhoneId(result.getResult());
        }

        GdValidationResult validationResult = gridValidationService.getValidationResult(result, emptyPath());
        return new GdUpdateTelephonyRedirectPhonePayload().withValidationResult(validationResult);
    }

    GdUpdateTelephonyRedirectPhonesPayload updateTelephonyRedirectPhones(ClientId clientId,
                                                                         GdUpdateTelephonyRedirectPhones input) {
        List<ClientPhone> clientPhones = toTelephonyPhones(clientId, input);
        MassResult<Long> results = clientPhoneService.updateTelephonyRedirectPhones(clientId, clientPhones);
        List<GdUpdateTelephonyRedirectPhonePayload> items = new ArrayList<>();
        for (Result<Long> result : results.toResultList()) {
            if (result.isSuccessful()) {
                items.add(new GdUpdateTelephonyRedirectPhonePayload()
                        .withPhoneId(result.getResult()));
            }
        }
        return new GdUpdateTelephonyRedirectPhonesPayload()
                .withItems(items)
                .withValidationResult(validationResultConverter.buildGridValidationResult(results.getValidationResult()));
    }
}
