package ru.yandex.direct.grid.processing.service.trackingphone;

import java.util.List;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLContext;
import io.leangen.graphql.annotations.GraphQLMutation;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.annotations.EnableLoggingOnValidationIssues;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.banner.GdTextAd;
import ru.yandex.direct.grid.processing.model.client.GdClient;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.model.organizations.GdOrganization;
import ru.yandex.direct.grid.processing.model.trackingphone.GdGetLinkedOrganizations;
import ru.yandex.direct.grid.processing.model.trackingphone.GdGetLinkedPhones;
import ru.yandex.direct.grid.processing.model.trackingphone.GdGetPhonesInput;
import ru.yandex.direct.grid.processing.model.trackingphone.GdGetPhonesPayload;
import ru.yandex.direct.grid.processing.model.trackingphone.GdTrackingPhone;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdAddClientPhone;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdAddClientPhonePayload;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdDeleteClientPhone;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdDeleteClientPhonePayload;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdUpdateAdsPhone;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdUpdateAdsPhonePayload;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdUpdateClientPhone;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdUpdateClientPhonePayload;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdUpdateTelephonyRedirectPhone;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdUpdateTelephonyRedirectPhonePayload;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdUpdateTelephonyRedirectPhones;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdUpdateTelephonyRedirectPhonesPayload;

@GridGraphQLService
@ParametersAreNonnullByDefault
public class TrackingPhoneGraphQLService {

    private static final String PHONES_RESOLVER_NAME = "phones";
    private static final String LINKED_ORGANIZATIONS_RESOLVER_NAME = "linkedOrganizations";
    private static final String LINKED_PHONES_RESOLVER_NAME = "linkedPhones";

    private final TrackingPhoneDataService dataService;
    private final TrackingPhoneDataLoader phoneDataLoader;

    @Autowired
    public TrackingPhoneGraphQLService(TrackingPhoneDataService dataService, TrackingPhoneDataLoader phoneDataLoader) {
        this.dataService = dataService;
        this.phoneDataLoader = phoneDataLoader;
    }

    @GraphQLNonNull
    @GraphQLQuery(name = PHONES_RESOLVER_NAME)
    public GdGetPhonesPayload getPhones(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdGetPhonesInput input) {
        ClientId clientId = context.getSubjectUser().getClientId();
        return dataService.getClientPhones(clientId, input.getPermalinkIds());
    }

    @GraphQLQuery(name = "phone")
    public CompletableFuture<GdTrackingPhone> getPhone(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdTextAd textAd
    ) {
        Long phoneId = textAd.getPhoneId();
        if (phoneId == null) {
            return CompletableFuture.completedFuture(null);
        }
        return phoneDataLoader.get().load(phoneId);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = LINKED_PHONES_RESOLVER_NAME)
    public List<GdTrackingPhone> getLinkedPhones(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdGetLinkedPhones input) {
        GdClientInfo info = client.getInfo();
        ClientId clientId = ClientId.fromLong(info.getId());
        return dataService.getLinkedPhones(clientId, input.getAdIds());
    }

    @GraphQLNonNull
    @GraphQLQuery(name = LINKED_ORGANIZATIONS_RESOLVER_NAME)
    public List<GdOrganization> getLinkedOrganizations(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdGetLinkedOrganizations input) {
        GdClientInfo info = client.getInfo();
        ClientId clientId = ClientId.fromLong(info.getId());
        return dataService.getLinkedOrganizations(clientId, input.getAdIds());

    }

    @EnableLoggingOnValidationIssues
    @GraphQLNonNull
    @GraphQLMutation(name = "updateAdsPhone")
    @PreAuthorizeWrite
    public GdUpdateAdsPhonePayload updateAdsPhone(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateAdsPhone input
    ) {
        ClientId clientId = context.getSubjectUser().getClientId();
        return dataService.updateAdsPhone(clientId, context.getOperator().getUid(), input);
    }

    @EnableLoggingOnValidationIssues
    @GraphQLNonNull
    @GraphQLMutation(name = "addClientPhone")
    @PreAuthorizeWrite
    public GdAddClientPhonePayload addClientPhone(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddClientPhone input
    ) {
        ClientId clientId = context.getSubjectUser().getClientId();
        return dataService.addClientPhone(clientId, input);
    }

    @EnableLoggingOnValidationIssues
    @GraphQLNonNull
    @GraphQLMutation(name = "updateClientPhone")
    @PreAuthorizeWrite
    public GdUpdateClientPhonePayload updateClientPhone(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateClientPhone input
    ) {
        ClientId clientId = context.getSubjectUser().getClientId();
        return dataService.updateClientPhone(clientId, input);
    }

    @EnableLoggingOnValidationIssues
    @GraphQLNonNull
    @GraphQLMutation(name = "deleteClientPhone")
    @PreAuthorizeWrite
    public GdDeleteClientPhonePayload deleteClientPhone(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdDeleteClientPhone input
    ) {
        return dataService.deleteClientPhone(context.getOperator().getUid(), context.getSubjectUser(), input);
    }

    @EnableLoggingOnValidationIssues
    @GraphQLNonNull
    @GraphQLMutation(name = "updateTelephonyRedirectPhone")
    @PreAuthorizeWrite
    public GdUpdateTelephonyRedirectPhonePayload updateTelephonyRedirectPhone(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateTelephonyRedirectPhone input
    ) {
        ClientId clientId = context.getSubjectUser().getClientId();
        return dataService.updateTelephonyRedirectPhone(clientId, input);
    }

    @EnableLoggingOnValidationIssues
    @GraphQLNonNull
    @GraphQLMutation(name = "updateTelephonyRedirectPhones")
    @PreAuthorizeWrite
    public GdUpdateTelephonyRedirectPhonesPayload updateTelephonyRedirectPhones(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateTelephonyRedirectPhones input
    ) {
        ClientId clientId = context.getSubjectUser().getClientId();
        return dataService.updateTelephonyRedirectPhones(clientId, input);
    }
}
