package ru.yandex.direct.grid.processing.service.trackingphone;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.clientphone.ClientPhoneService;
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhone;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdDeleteClientPhone;
import ru.yandex.direct.grid.processing.model.trackingphone.mutation.GdUpdateAdsPhone;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.clientphone.validation.ClientPhoneValidationService.isPhoneManual;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;

@Service
@ParametersAreNonnullByDefault
public class TrackingPhoneValidationService {
    private final GridValidationService gridValidationService;
    private final ClientPhoneService clientPhoneService;

    @Autowired
    public TrackingPhoneValidationService(GridValidationService gridValidationService,
                                          ClientPhoneService clientPhoneService) {
        this.gridValidationService = gridValidationService;
        this.clientPhoneService = clientPhoneService;
    }


    private ValidationResult<GdUpdateAdsPhone, Defect> validateUpdateAdsRequest(GdUpdateAdsPhone req,
                                                                                ClientId clientId) {
        ModelItemValidationBuilder<GdUpdateAdsPhone> vb = ModelItemValidationBuilder.of(req);
        vb.list(GdUpdateAdsPhone.AD_IDS)
                .check(notEmptyCollection());
        vb.list(GdUpdateAdsPhone.PERMALINK_IDS)
                .check(notEmptyCollection());
        vb.item(GdUpdateAdsPhone.PHONE_ID)
                .check(validId(), When.notNull());

        Long phoneId = req.getPhoneId();
        if (phoneId != null) {
            Map<Long, ClientPhone> existedPhones = listToMap(
                    clientPhoneService.getByPhoneIds(clientId, List.of(phoneId)), ClientPhone::getId);
            vb.item(GdUpdateAdsPhone.PHONE_ID)
                    .check(inSet(existedPhones.keySet()));
        }

        return vb.getResult();
    }

    public void validateUpdateAdsPhone(ClientId clientId, GdUpdateAdsPhone request) {
        gridValidationService.applyValidator(req -> validateUpdateAdsRequest(req, clientId), request, false);
    }

    public void validateReplaceTrackingPhone(ClientId clientId, GdDeleteClientPhone input) {
        gridValidationService.applyValidator(req -> validateReplaceTrackingPhoneRequest(clientId, req), input, false);
    }

    private ValidationResult<GdDeleteClientPhone, Defect> validateReplaceTrackingPhoneRequest(
            ClientId clientId,
            GdDeleteClientPhone input
    ) {
        ModelItemValidationBuilder<GdDeleteClientPhone> vb = ModelItemValidationBuilder.of(input);

        List<Long> phoneIds = new ArrayList<>(input.getPhoneIds());
        phoneIds.add(input.getReplacePhoneId());

        Map<Long, ClientPhone> existedPhones = listToMap(
                clientPhoneService.getByPhoneIds(clientId, phoneIds),
                ClientPhone::getId
        );

        Constraint<Long, Defect> isPhoneManual = isPhoneManual(existedPhones);

        vb.list(input.getPhoneIds(), GdDeleteClientPhone.PHONE_IDS.name())
                .checkEach(inSet(existedPhones.keySet()))
                .checkEach(isPhoneManual, When.isValid());

        vb.item(GdDeleteClientPhone.REPLACE_PHONE_ID)
                .check(inSet(existedPhones.keySet()))
                .check(isPhoneManual, When.isValid());

        return vb.getResult();
    }

}
