package ru.yandex.direct.grid.processing.service.turbolanding;

import java.util.List;
import java.util.Map;
import java.util.function.Consumer;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.turbolanding.model.TurboLanding;
import ru.yandex.direct.core.entity.turbolanding.service.TurboLandingService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.model.cliententity.GdTurbolanding;
import ru.yandex.direct.grid.processing.model.turbolanding.GdFindTurboLandings;
import ru.yandex.direct.grid.processing.model.turbolanding.GdFindTurboLandingsResult;
import ru.yandex.direct.grid.processing.service.client.converter.ClientDataConverter;

@Service
@ParametersAreNonnullByDefault
public class TurboLandingDataService {
    private static final Logger logger = LoggerFactory.getLogger(TurboLandingDataService.class);

    private final TurboLandingService turboLandingService;

    @Autowired
    public TurboLandingDataService(TurboLandingService turboLandingService) {
        this.turboLandingService = turboLandingService;
    }

    /**
     * Найти соответствующие заданным url турболендинги
     * Если найденный турболендинг принадлежит другому клиенту - возвращаем ClientId:0
     */
    GdFindTurboLandingsResult findTurboLandings(ClientId clientId, GdFindTurboLandings input) {
        Map<String, TurboLanding> foundTurboLandingsByUrl = turboLandingService.findTurboLandingsByUrl(input.getUrlList());

        return new GdFindTurboLandingsResult().withFoundTurboLandings(
                EntryStream.of(foundTurboLandingsByUrl)
                        .mapValues(ClientDataConverter::toGdTurbolanding)
                        .peekValues(unsetAlienClientId(clientId))
                        .toMap());
    }

    public List<GdTurbolanding> getTurbolandigsByClientId(ClientId clientId) {
        List<TurboLanding> foundTurboLandings = List.of();
        try {
            foundTurboLandings = turboLandingService.getTurbolandigsByClientId(clientId);
        } catch (RuntimeException e) {
            logger.error(e.getMessage());
        }

        return StreamEx.of(foundTurboLandings)
                        .map(ClientDataConverter::toGdTurbolanding)
                        .peek(unsetAlienClientId(clientId))
                        .toList();
    }

    private Consumer<GdTurbolanding> unsetAlienClientId(ClientId clientId) {
        return tl -> {
            if (!tl.getClientId().equals(clientId.asLong())) {
                tl.setClientId(0L);
            }
        };
    }
}
