package ru.yandex.direct.grid.processing.service.user;

import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.model.UsersBlockReasonType;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.core.entity.user.service.validation.BlockUserValidationService;
import ru.yandex.direct.core.entity.user.service.validation.UserValidationService;
import ru.yandex.direct.core.security.AccessDeniedException;
import ru.yandex.direct.grid.processing.model.client.GdUserInfo;
import ru.yandex.direct.grid.processing.model.user.mutation.GdBlockReasonType;
import ru.yandex.direct.grid.processing.model.user.mutation.GdBlockUser;
import ru.yandex.direct.grid.processing.model.user.mutation.GdBlockUserAction;
import ru.yandex.direct.grid.processing.model.user.mutation.GdBlockUserPayload;
import ru.yandex.direct.grid.processing.model.user.mutation.GdUpdateUserPayload;
import ru.yandex.direct.grid.processing.model.user.mutation.GdUpdateUserPhone;
import ru.yandex.direct.grid.processing.service.operator.OperatorDataService;
import ru.yandex.direct.grid.processing.service.user.validation.UserMutationValidationService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.MoreObjects.firstNonNull;
import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyList;
import static java.util.Collections.singleton;
import static org.apache.commons.collections.CollectionUtils.isNotEmpty;
import static ru.yandex.direct.core.entity.user.utils.BlockedUserUtil.logUserBlock;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isClient;
import static ru.yandex.direct.grid.processing.service.operator.OperatorAllowedActionsUtils.canBlockUser;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.pathFromStrings;

/**
 * DataService для {@link UserGraphQlService}
 */
@Service
@ParametersAreNonnullByDefault
public class UserDataService {

    private static final Logger logger = LoggerFactory.getLogger(UserDataService.class);

    private final UserService userService;
    private final GridValidationService gridValidationService;
    private final UserValidationService userValidationService;
    private final BlockUserValidationService blockUserValidationService;
    private final OperatorDataService operatorDataService;
    private final UserMutationValidationService userMutationValidationService;

    public UserDataService(
            UserService userService,
            GridValidationService gridValidationService,
            UserValidationService userValidationService,
            OperatorDataService operatorDataService,
            BlockUserValidationService blockUserValidationService,
            UserMutationValidationService userMutationValidationService) {
        this.userService = userService;
        this.gridValidationService = gridValidationService;
        this.userValidationService = userValidationService;
        this.operatorDataService = operatorDataService;
        this.blockUserValidationService = blockUserValidationService;
        this.userMutationValidationService = userMutationValidationService;
    }

    /**
     * Обновление телефона пользователя
     */
    @Nonnull
    GdUpdateUserPayload updateRepPhone(User operator, User subjectUser, GdUpdateUserPhone input) {
        User targetUser = checkOperatorRightsForChiefRep(operator, subjectUser);

        ModelChanges<User> changes = convertToChanges(targetUser, input);
        AppliedChanges<User> userAppliedChanges = changes.applyTo(checkNotNull(targetUser));

        ValidationResult<User, Defect> userVr = userValidationService.validate(userAppliedChanges);
        gridValidationService.throwGridValidationExceptionIfHasErrors(userVr);

        userService.update(userAppliedChanges);

        Long uid = targetUser.getUid();
        Map<Long, GdUserInfo> usersByIds = operatorDataService.getUsersByIds(singleton(uid));
        GdUserInfo userInfo = checkNotNull(usersByIds.get(uid), "User is missing after update");
        return new GdUpdateUserPayload().withUser(userInfo);
    }

    /**
     * Проверяем наличие у оператора прав на модификацию параметров главного представителя.
     * <p>
     * Необходимость наличия прав оператора на изменение данных клиента описана в {@link UserGraphQlService}
     * аннотацией {@link ru.yandex.direct.core.security.authorization.PreAuthorizeWrite}.
     * <p>
     * Проверки по большей части повторяют проверки из API класса {@code UpdateClientOperation}.
     *
     * @return {@link User} пользователь &ndash; главный представитель
     */
    @Nonnull
    private User checkOperatorRightsForChiefRep(User operator, User subjectUser) {
        if (operator.getRole() == RbacRole.PLACER) {
            throw new AccessDeniedException("Оператор с ролью PLACER не допускается к изменению данных клиента");
        }

        if (operator.getRole() == RbacRole.AGENCY) {
            throw new AccessDeniedException("Представители агентств не допускается к изменению данных клиента");
        }

        if (isClient(operator) && !userIsChief(operator) && !Objects.equals(operator.getUid(), subjectUser.getUid())) {
            // оператор с ролью "Клиент", не являющийся главным представителем, может получать только свои данные
            throw new AccessDeniedException("У оператора нет прав на указанного пользователя");
        }
        return subjectUser;
    }

    private boolean userIsChief(User user) {
        return Objects.equals(user.getUid(), user.getChiefUid());
    }

    /**
     * Преобразует параметры запроса в изменения core-модели пользователя
     */
    private ModelChanges<User> convertToChanges(User user, GdUpdateUserPhone input) {
        ModelChanges<User> changes = new ModelChanges<>(user.getUid(), User.class);
        changes.processNotNull(input.getPhone(), User.PHONE);
        return changes;
    }

    /**
     * Блокировка/разблокировка пользователя
     */
    GdBlockUserPayload blockUser(User operator, GdBlockUser input) {
        userMutationValidationService.validateBlockUserRequest(input);
        checkOperatorRightsForBlock(operator);
        List<User> users = getUsersByLoginOrUid(input);
        ValidationResult<List<User>, Defect> validationResult = blockUserValidationService.validateAccess(users);
        if (validationResult.hasAnyErrors()) {
            return new GdBlockUserPayload()
                    .withUserIds(emptyList())
                    .withValidationResult(gridValidationService.toGdValidationResult(validationResult,
                            pathFromStrings("users")));
        }

        if (input.getAction() == GdBlockUserAction.BLOCK) {
            UsersBlockReasonType blockReasonType = GdBlockReasonType.toSource(
                    firstNonNull(input.getBlockReasonType(), GdBlockReasonType.NOT_SET));
            Preconditions.checkNotNull(blockReasonType);
            boolean checkCampaignsCount = !firstNonNull(input.getIgnoreCampaignsCountCheck(), false);
            validationResult = blockUsers(operator, users, blockReasonType, input.getBlockComment(),
                    checkCampaignsCount);
        } else {
            validationResult = userService.unblockUsers(users);
        }
        List<Long> updatedUserIds = validationResult.hasAnyErrors() ? emptyList() : mapList(users, User::getUid);

        return new GdBlockUserPayload()
                .withUserIds(updatedUserIds)
                .withValidationResult(gridValidationService.toGdValidationResult(validationResult,
                        pathFromStrings("users")));
    }

    public ValidationResult<List<User>, Defect> blockUsers(User operator, List<User> users,
                                                           UsersBlockReasonType blockReasonType,
                                                           @Nullable String blockComment,
                                                           boolean checkCampsCount) {
        Map<Long, List<Long>> campaignIdsByClientId =
                userService.getCampaignIdsForStopping(mapList(users, User::getClientId));

        ValidationResult<List<User>, Defect> vr = userService.blockUsers(operator, users, campaignIdsByClientId,
                blockReasonType, blockComment, checkCampsCount);
        if (vr.hasAnyErrors()) {
            return vr;
        }

        users.forEach(user -> {
            Long clientId = user.getClientId().asLong();
            logUserBlock(user, operator, campaignIdsByClientId.getOrDefault(clientId, emptyList()));
            logger.info("blockUsers: operatorUid: {}, blockedClientId: {}", operator.getUid(), clientId);
        });

        return vr;
    }

    /**
     * Возвращает список пользователей с сохранением порядка относительно входных параметров: logins или uids
     * NB: Если пользователя нет в базе, то в списке будет null. На null есть валидация тут:
     * {@link BlockUserValidationService#validateAccess}
     */
    private List<User> getUsersByLoginOrUid(GdBlockUser input) {
        if (isNotEmpty(input.getUserLogins())) {
            var userByLogin = listToMap(userService.massGetUserByLogin(input.getUserLogins()), User::getLogin);
            return mapList(input.getUserLogins(), login -> userByLogin.getOrDefault(login, null));
        } else {
            var userByUid = listToMap(userService.massGetUser(input.getUserIds()), User::getUid);
            return mapList(input.getUserIds(), uid -> userByUid.getOrDefault(uid, null));
        }
    }

    private void checkOperatorRightsForBlock(User operator) {
        if (!canBlockUser(operator)) {
            throw new AccessDeniedException("Оператор не может блокировать/разблокировать пользователя");
        }
    }
}
