package ru.yandex.direct.grid.processing.service.user;

import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLMutation;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;

import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.security.authorization.PreAuthorizeRead;
import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.user.mutation.GdBlockUser;
import ru.yandex.direct.grid.processing.model.user.mutation.GdBlockUserPayload;
import ru.yandex.direct.grid.processing.model.user.mutation.GdUpdateUserPayload;
import ru.yandex.direct.grid.processing.model.user.mutation.GdUpdateUserPhone;

import static ru.yandex.direct.grid.processing.util.TextConstants.INPUT;

/**
 * Сервис для работы с пользователем.
 * <p>
 * Очень похож на {@link ru.yandex.direct.grid.processing.service.operator.OperatorGraphQlService}.
 * В перспективе хотелось бы их слить.
 */
@GridGraphQLService
@ParametersAreNonnullByDefault
public class UserGraphQlService {

    private final UserDataService userDataService;

    public UserGraphQlService(UserDataService userDataService) {
        this.userDataService = userDataService;
    }

    /**
     * Обновление телефона.
     * При необходимости можно добавить в {@link GdUpdateUserPhone} {@code nullable} параметр {@code userId}
     */
    @GraphQLNonNull
    @GraphQLMutation(
            name = "updateUserPhone",
            description = "Обновление номера у subjectUser")
    @PreAuthorizeWrite
    public GdUpdateUserPayload updateUserPhone(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = INPUT) GdUpdateUserPhone input
    ) {
        User operator = context.getOperator();
        User subjectUser = context.getSubjectUser();
        return userDataService.updateRepPhone(operator, subjectUser, input);
    }

    /**
     * Блокировка/разблокировка пользователя
     */
    @GraphQLNonNull
    @GraphQLMutation(
            name = "blockUser",
            description = "Блокировка или разблокировка пользователя")
    @PreAuthorizeRead
    public GdBlockUserPayload blockUser(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = INPUT) GdBlockUser input
    ) {
        User operator = context.getOperator();
        return userDataService.blockUser(operator, input);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "repUserName")
    @PreAuthorizeRead
    public String getRepUserName(@GraphQLRootContext GridGraphQLContext context) {
        User operator = context.getOperator();
        User subjectUser = context.getSubjectUser();

        // Есть клиент А и его представитель Б. Поддерживаем два варианта:
        // Первый: Б логинится, в ulogin'е — Б
        // Второй: А логинится, кидает ссылку Б, при открытии по ссылке в ulogin'е — А

        //noinspection ConstantConditions
        User user = operator.getClientId().equals(subjectUser.getClientId())
                ? operator
                : subjectUser;
        return user.getFio();

    }
}
