package ru.yandex.direct.grid.processing.service.userphone;

import java.util.List;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLMutation;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.userphone.GdPassportPhone;
import ru.yandex.direct.grid.processing.model.userphone.GdUpdateUserVerifiedPhonePayload;
import ru.yandex.direct.grid.processing.model.userphone.GdUpdateVerifiedUserPhoneContainer;
import ru.yandex.direct.grid.processing.model.userphone.GdUserPhonesPayload;
import ru.yandex.direct.grid.processing.service.userphone.service.UserPhoneService;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxPhone;

import static org.apache.commons.lang3.StringUtils.isEmpty;
import static ru.yandex.direct.grid.processing.util.TextConstants.INPUT;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@GridGraphQLService
public class UserPhoneGraphQlService {

    private static final Logger logger = LoggerFactory.getLogger(UserPhoneGraphQlService.class);

    private final UserService userService;
    private final UserPhoneService userPhoneService;

    @Autowired
    public UserPhoneGraphQlService(UserService userService,
                                   UserPhoneService userPhoneService) {
        this.userService = userService;
        this.userPhoneService = userPhoneService;
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "userPhones")
    public GdUserPhonesPayload getUserPhones(@GraphQLRootContext GridGraphQLContext context) {
        User subjectUser = context.getSubjectUser();
        Long subjectUserId = subjectUser.getUid();

        List<BlackboxPhone> blackboxPhones = userService.getAllBlackboxPhones(subjectUserId);

        GdPassportPhone directPhone = null;
        if (subjectUser.getVerifiedPhoneId() != null) {
            directPhone = StreamEx.of(blackboxPhones)
                    .findAny(blackboxPhone -> subjectUser.getVerifiedPhoneId().equals(blackboxPhone.getPhoneId()))
                    .map(UserPhoneUtils::toGdPassportPhone)
                    .orElse(null);
        }
        if (directPhone == null && !isEmpty(subjectUser.getPhone())) {
            directPhone = new GdPassportPhone()
                    .withPhone(subjectUser.getPhone())
                    .withIsConfirmed(false);
        }

        return new GdUserPhonesPayload()
                .withPassportPhones(mapList(blackboxPhones, UserPhoneUtils::toGdPassportPhone))
                .withDirectPhone(directPhone);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "updateUserVerifiedPhone")
    public GdUpdateUserVerifiedPhonePayload updateUserVerifiedPhone(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = INPUT) GdUpdateVerifiedUserPhoneContainer input) {
        User subjectUser = context.getSubjectUser();

        GdUpdateUserVerifiedPhonePayload payload =
                userPhoneService.updateUserVerifiedPhone(subjectUser, input.getPhoneId());

        logger.info("User {} updated verified phone", subjectUser.getUid());

        return payload;
    }
}
