package ru.yandex.direct.grid.processing.service.userphone;

import java.time.Duration;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.util.HttpUtil;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.core.entity.userphone.UserPhoneStorage;
import ru.yandex.direct.core.redis.StorageErrorException;
import ru.yandex.direct.core.service.integration.passport.PassportService;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.geobasehelper.GeoBaseHelper;
import ru.yandex.direct.grid.processing.annotations.PublicGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.userphone.GdConfirmAndBindPhoneCommitContainer;
import ru.yandex.direct.grid.processing.model.userphone.GdConfirmAndBindPhoneCommitPayload;
import ru.yandex.direct.grid.processing.model.userphone.GdConfirmAndBindPhoneSubmitContainer;
import ru.yandex.direct.grid.processing.model.userphone.GdConfirmAndBindPhoneSubmitPayload;
import ru.yandex.direct.grid.processing.model.userphone.GdNewUserFeatures;
import ru.yandex.direct.grid.processing.model.userphone.GdUserPhonesPayload;
import ru.yandex.direct.grid.processing.service.userphone.service.TestUserPhoneService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxPhone;
import ru.yandex.inside.passport.internal.api.models.phone.PhoneSubmitResponse;
import ru.yandex.inside.passport.internal.api.models.validation.ValidatePhoneNumberResponse;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static ru.yandex.direct.common.db.PpcPropertyNames.COLLECTING_VERIFIED_PHONES_DISABLED_FOR_COUNTRIES;
import static ru.yandex.direct.common.db.PpcPropertyNames.PHONE_VERIFICATION_MAX_CALLS_COUNT_PER_DAY;
import static ru.yandex.direct.grid.processing.model.userphone.GdConfirmAndBindPhoneSubmitContainer.PHONE;
import static ru.yandex.direct.grid.processing.service.userphone.validation.UserPhoneDefects.alreadyBindAndSecure;
import static ru.yandex.direct.grid.processing.service.userphone.validation.UserPhoneDefects.confirmationsLimitExceeded;
import static ru.yandex.direct.grid.processing.service.userphone.validation.UserPhoneDefects.emptyPhone;
import static ru.yandex.direct.grid.processing.service.userphone.validation.UserPhoneDefects.internalPassportError;
import static ru.yandex.direct.grid.processing.service.userphone.validation.UserPhoneDefects.invalidCode;
import static ru.yandex.direct.grid.processing.service.userphone.validation.UserPhoneDefects.invalidPhone;
import static ru.yandex.direct.grid.processing.service.userphone.validation.UserPhoneDefects.smsLimitExceeded;
import static ru.yandex.direct.grid.processing.service.userphone.validation.UserPhoneDefects.tooManyRequests;
import static ru.yandex.direct.grid.processing.service.userphone.validation.UserPhoneDefects.unexpectedError;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@PublicGraphQLService
public class UserPhonePublicGraphQlService {

    private static final Logger logger = LoggerFactory.getLogger(UserPhonePublicGraphQlService.class);

    private final FeatureService featureService;
    private final UserService userService;
    private final PassportService passportService;
    private final GridValidationResultConversionService validationResultConversionService;
    private final GeoBaseHelper geoBaseHelper;
    private final UserPhoneStorage userPhoneStorage;
    private final TestUserPhoneService testUserPhoneService;

    private final PpcProperty<Set<Long>> disabledCountryIdsProperty;
    private final PpcProperty<Integer> maxCallsCountPerDayProperty;

    @Autowired
    public UserPhonePublicGraphQlService(FeatureService featureService,
                                         UserService userService,
                                         PassportService passportService,
                                         GridValidationResultConversionService validationResultConversionService,
                                         PpcPropertiesSupport ppcPropertiesSupport,
                                         GeoBaseHelper geoBaseHelper,
                                         UserPhoneStorage userPhoneStorage,
                                         TestUserPhoneService testUserPhoneService) {
        this.featureService = featureService;
        this.userService = userService;
        this.passportService = passportService;
        this.validationResultConversionService = validationResultConversionService;
        this.geoBaseHelper = geoBaseHelper;
        this.userPhoneStorage = userPhoneStorage;
        this.testUserPhoneService = testUserPhoneService;

        this.disabledCountryIdsProperty =
                ppcPropertiesSupport.get(COLLECTING_VERIFIED_PHONES_DISABLED_FOR_COUNTRIES, Duration.ofMinutes(5));
        this.maxCallsCountPerDayProperty =
                ppcPropertiesSupport.get(PHONE_VERIFICATION_MAX_CALLS_COUNT_PER_DAY, Duration.ofMinutes(5));
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "newUserFeatures")
    public GdNewUserFeatures getNewUserFeatures(@GraphQLRootContext GridGraphQLContext context) {
        Long operatorUid = context.getOperator().getUid();
        Set<Long> disabledCountryIds = nvl(disabledCountryIdsProperty.get(), emptySet());

        boolean collectingPhonesFeatureEnabled =
                featureService.isEnabled(operatorUid, FeatureName.COLLECTING_VERIFIED_PHONES_FOR_NEW_USERS);
        boolean collectingPhoneNotRequiredFeatureEnabled =
                featureService.isEnabled(operatorUid, FeatureName.COLLECTING_VERIFIED_PHONES_NOT_REQUIRED_FOR_NEW_USERS);
        boolean isWebvisorForNewUsersForDnaEnabled =
                featureService.isEnabled(operatorUid, FeatureName.WEBVISOR_FOR_NEW_USERS_ENABLED_FOR_DNA);
        boolean isUacDesktopWelcomeEnabled =
                featureService.isEnabled(operatorUid, FeatureName.UAC_DESKTOP_WELCOME_ENABLED);
        boolean isUacMobileWelcomeEnabled =
                featureService.isEnabled(operatorUid, FeatureName.UAC_MOBILE_WELCOME_ENABLED);
        boolean isTinRequired =
                featureService.isEnabled(operatorUid, FeatureName.REQUIRED_TIN_FOR_NEW_USER);
        boolean isTinTypeRequired =
                featureService.isEnabled(operatorUid, FeatureName.REQUIRED_TIN_TYPE_FOR_NEW_USER);

        if (collectingPhonesFeatureEnabled) {
            boolean userCountryDisabled = HttpUtil.getCurrentGeoRegionId()
                    .map(geoBaseHelper::getCountryId)
                    .map(Integer::longValue)
                    .map(disabledCountryIds::contains)
                    .orElse(false);

            if (userCountryDisabled) {
                logger.info("User {} is from disabled country", operatorUid);
            }

            collectingPhonesFeatureEnabled = !userCountryDisabled;
        }

        GdNewUserFeatures userFeatures = new GdNewUserFeatures()
                .withIsCollectingVerifiedPhonesEnabled(collectingPhonesFeatureEnabled)
                .withCollectingVerifiedPhoneDisabledCountryIds(disabledCountryIds)
                .withIsWebvisorForNewUsersForDnaEnabled(isWebvisorForNewUsersForDnaEnabled)
                .withIsUacDesktopWelcomeEnabled(isUacDesktopWelcomeEnabled)
                .withIsUacMobileWelcomeEnabled(isUacMobileWelcomeEnabled)
                .withIsTinRequired(isTinRequired)
                .withIsTinTypeRequired(isTinTypeRequired);

        if (collectingPhonesFeatureEnabled) {
            userFeatures.withIsCollectingVerifiedPhonesNotRequired(collectingPhoneNotRequiredFeatureEnabled);
        }

        return userFeatures;
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "userPhones")
    public GdUserPhonesPayload getUserPhones(@GraphQLRootContext GridGraphQLContext context) {
        Long operatorUid = context.getOperator().getUid();

        List<BlackboxPhone> blackboxPhones = userService.getAllBlackboxPhones(operatorUid);

        return new GdUserPhonesPayload()
                .withPassportPhones(mapList(blackboxPhones, UserPhoneUtils::toGdPassportPhone));
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "confirmAndBindPhoneSubmit")
    public GdConfirmAndBindPhoneSubmitPayload confirmAndBindPhoneSubmit(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdConfirmAndBindPhoneSubmitContainer input) {
        Long operatorUid = context.getOperator().getUid();
        String reqId = String.valueOf(Trace.current().getParentId());
        String phone = input.getPhone();

        if (testUserPhoneService.isTestCallForSubmit(phone)) {
            return testUserPhoneService.confirmAndBindPhoneSubmit();
        }

        logger.info("User {} trying to bind phone", operatorUid);

        int callsCount = 0;
        try {
            callsCount = userPhoneStorage.incrementAndGetCallsCount(operatorUid);
        } catch (StorageErrorException e) {
            logger.warn("Redis call failed", e);
        }

        logger.info("Current calls count for user {} is {}", operatorUid, callsCount);
        if (callsCount > maxCallsCountPerDayProperty.getOrDefault(Integer.MAX_VALUE)) {
            logger.info("User {} exceeded calls limit", operatorUid);
            return new GdConfirmAndBindPhoneSubmitPayload()
                    .withValidationResult(
                            validationResultConversionService.buildGridValidationResult(
                                    ValidationResult.failed(null, tooManyRequests())));
        }

        PhoneSubmitResponse submitResponse = passportService.confirmAndBindSubmit(reqId, phone);

        if (submitResponse == null) {
            return new GdConfirmAndBindPhoneSubmitPayload()
                    .withValidationResult(
                            validationResultConversionService.buildGridValidationResult(
                                    ValidationResult.failed(null, internalPassportError())));
        }

        ValidationResult<GdConfirmAndBindPhoneSubmitContainer, Defect> vr = new ValidationResult<>(input, Defect.class);
        PathNode.Field field = new PathNode.Field(PHONE.name());

        List<String> errors = nvl(submitResponse.getErrors(), emptyList());
        if (errors.contains("number.invalid")) {
            logger.info("User {} entered invalid phone number: {}", operatorUid, phone);
            vr.addSubResult(field, ValidationResult.failed(input.getPhone(), invalidPhone()));
        } else if (errors.contains("number.empty")) {
            logger.info("User {} entered empty phone number", operatorUid);
            vr.addSubResult(field, ValidationResult.failed(input.getPhone(), emptyPhone()));
        } else if (errors.contains("sms_limit.exceeded")) {
            logger.info("User {} exceeded sms limit", operatorUid);
            vr.addSubResult(field, ValidationResult.failed(input.getPhone(), smsLimitExceeded()));
        } else if (errors.contains("phone_secure.bound_and_confirmed")) {
            Optional<BlackboxPhone> securePhone = StreamEx.of(userService.getAllBlackboxPhones(operatorUid))
                    .findAny(blackboxPhone -> blackboxPhone.getIsSecureNumber().getOrElse(false));
            Long securePhoneId = securePhone.map(BlackboxPhone::getPhoneId).orElse(null);

            logger.info("User {} trying to bind secure phone with id {}", operatorUid, securePhoneId);

            vr.addSubResult(field, ValidationResult.failed(input.getPhone(), alreadyBindAndSecure(securePhoneId)));
        } else if (!errors.isEmpty()) {
            logger.info("User {} got errors during submit {}", operatorUid, errors);
            vr.addSubResult(field, ValidationResult.failed(input.getPhone(), unexpectedError()));
        } else {
            logger.info("Sms sent to user {}", operatorUid);
        }

        String trackId = submitResponse.getTrackId();
        return new GdConfirmAndBindPhoneSubmitPayload()
                .withTrackId(trackId)
                .withValidationResult(validationResultConversionService.buildGridValidationResult(vr));
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "confirmAndBindPhoneCommit")
    public GdConfirmAndBindPhoneCommitPayload confirmAndBindPhoneCommit(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdConfirmAndBindPhoneCommitContainer input) {
        Long operatorUid = context.getOperator().getUid();
        String reqId = String.valueOf(Trace.current().getParentId());

        if (testUserPhoneService.isTestCallForCommit(input.getTrackId())) {
            return testUserPhoneService.confirmAndBindPhoneCommit(operatorUid);
        }

        logger.info("User {} trying to enter code", operatorUid);

        int callsCount = 0;
        try {
            callsCount = userPhoneStorage.incrementAndGetCallsCount(operatorUid);
        } catch (StorageErrorException e) {
            logger.warn("Redis call failed", e);
        }

        logger.info("Current calls count for user {} is {}", operatorUid, callsCount);
        if (callsCount > maxCallsCountPerDayProperty.getOrDefault(Integer.MAX_VALUE)) {
            logger.info("User {} exceeded calls limit", operatorUid);
            return new GdConfirmAndBindPhoneCommitPayload()
                    .withValidationResult(
                            validationResultConversionService.buildGridValidationResult(
                                    ValidationResult.failed(null, tooManyRequests())));
        }

        ValidatePhoneNumberResponse commitResponse =
                passportService.confirmAndBindCommit(reqId, input.getTrackId(), input.getCode());

        // Валидация; TODO(dimitrovsd): вынести в отдельный метод/модуль
        ValidationResult<GdConfirmAndBindPhoneCommitContainer, Defect> vr = new ValidationResult<>(input, Defect.class);
        PathNode.Field trackId = new PathNode.Field(GdConfirmAndBindPhoneCommitContainer.TRACK_ID.name());
        PathNode.Field code = new PathNode.Field(GdConfirmAndBindPhoneCommitContainer.CODE.name());

        if (commitResponse == null) {
            return new GdConfirmAndBindPhoneCommitPayload()
                    .withValidationResult(
                            validationResultConversionService.buildGridValidationResult(
                                    ValidationResult.failed(null, internalPassportError())));
        }

        List<String> errors = nvl(commitResponse.getErrors(), emptyList());
        if (errors.contains("code.invalid")) {
            logger.info("User {} entered invalid code", operatorUid);
            vr.addSubResult(code, ValidationResult.failed(input.getCode(), invalidCode()));
        } else if (errors.contains("confirmations_limit.exceeded")) {
            logger.info("User {} exceeded confirmation limit", operatorUid);
            vr.addSubResult(code, ValidationResult.failed(input.getCode(), confirmationsLimitExceeded()));
        } else if (!errors.isEmpty()) {
            logger.info("User {} got errors during commit {}", operatorUid, errors);
            vr = ValidationResult.failed(input, unexpectedError());
        } else if (commitResponse.getPhoneId() == null) {
            logger.info("User {} did not get phone id from passport", operatorUid);
            vr = ValidationResult.failed(input, unexpectedError());
        } else {
            logger.info("User {} entered code correctly", operatorUid);
        }

        return new GdConfirmAndBindPhoneCommitPayload()
                .withPhoneId(commitResponse.getPhoneId())
                .withValidationResult(validationResultConversionService.buildGridValidationResult(vr));
    }
}
