package ru.yandex.direct.grid.processing.service.validation;

import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.UntypedAdGroup;
import ru.yandex.direct.core.entity.banner.model.McBanner;
import ru.yandex.direct.core.entity.calltracking.model.CalltrackingSettings;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageClient;
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhone;
import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.grid.processing.model.api.GdDefect;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.DefaultPathNodeConverterProvider;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectInfo;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.PathConverter;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.validation.ValidationUtils.hasAnyErrorsOrWarnings;
import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;
import static ru.yandex.direct.grid.processing.service.banner.converter.McBannerPathConverters.NEW_MC_BANNER_BANNER_PATH_CONVERTER;
import static ru.yandex.direct.grid.processing.service.pricepackage.converter.PricePackageDataConverter.PRICE_PACKAGE_CLIENT_PATH_CONVERTER;
import static ru.yandex.direct.grid.processing.service.trackingphone.Converter.CALLTRACKING_SETTINGS_PATH_CONVERTER;
import static ru.yandex.direct.grid.processing.service.trackingphone.Converter.CLIENT_PHONE_PATH_CONVERTER;
import static ru.yandex.direct.grid.processing.service.validation.presentation.AdGroupConverters.COMMON_AD_GROUP_PATH_CONVERTER;
import static ru.yandex.direct.grid.processing.service.validation.presentation.KeywordPathConverters.KEYWORD_PATH_CONVERTER;
import static ru.yandex.direct.grid.processing.service.validation.presentation.VcardPathConverters.VCARD_PATH_CONVERTER;
import static ru.yandex.direct.validation.result.PathHelper.path;

/**
 * Преобразование {@link ValidationResult} и {@link MassResult} в структуру grid-api для фронта
 */
@Service
@ParametersAreNonnullByDefault
public class GridValidationResultConversionService {

    private final PathNodeConverterProvider gridPathNodeConverterProvider;

    @Autowired
    public GridValidationResultConversionService() {
        this.gridPathNodeConverterProvider = DefaultPathNodeConverterProvider.builder()
                .register(UntypedAdGroup.class, COMMON_AD_GROUP_PATH_CONVERTER)
                .register(Keyword.class, KEYWORD_PATH_CONVERTER)
                .register(Vcard.class, VCARD_PATH_CONVERTER)
                .register(PricePackageClient.class, PRICE_PACKAGE_CLIENT_PATH_CONVERTER)
                .register(ClientPhone.class, CLIENT_PHONE_PATH_CONVERTER)
                .register(CalltrackingSettings.class, CALLTRACKING_SETTINGS_PATH_CONVERTER)
                .register(McBanner.class, NEW_MC_BANNER_BANNER_PATH_CONVERTER)
                .build();
    }

    @Nullable
    public GdValidationResult buildGridValidationResult(@Nullable ValidationResult<?, Defect> vr) {
        return buildGridValidationResult(vr, path(), gridPathNodeConverterProvider);
    }

    @Nullable
    public GdValidationResult buildGridValidationResult(@Nullable ValidationResult<?, Defect> vr, Path prefixPath) {
        return buildGridValidationResult(vr, prefixPath, gridPathNodeConverterProvider);
    }

    @Nullable
    public static GdValidationResult buildGridValidationResultIfErrors(@Nullable ValidationResult<?, Defect> vr,
                                                                       Path prefixPath, PathNodeConverterProvider pathNodeConverterProvider) {
        if (hasValidationIssues(vr)) {
            return buildGridValidationResult(vr, prefixPath, pathNodeConverterProvider);
        }
        return null;
    }

    @Nullable
    public GdValidationResult buildGridValidationResultIfErrorsOrWarnings(@Nullable ValidationResult<?, Defect> vr,
                                                                          Path prefixPath) {
        if (hasAnyErrorsOrWarnings(vr)) {
            return buildGridValidationResult(vr, prefixPath, gridPathNodeConverterProvider);
        }
        return null;
    }

    @Nullable
    public static GdValidationResult buildGridValidationResult(@Nullable ValidationResult<?, Defect> vr,
                                                               Path prefixPath, PathNodeConverterProvider pathNodeConverterProvider) {
        if (vr == null) {
            return null;
        }
        List<GdDefect> gridErrors = StreamEx.of(vr.flattenErrors(pathNodeConverterProvider))
                .map(di -> di.convertPath(PathConverter.prepend(prefixPath)))
                .map(GridValidationResultConversionService::convert)
                // В результате конвертации путей, может получиться несколько одинаковых дефектов на одно поле.
                // Схлопнем дубликаты в один дефект
                .distinct()
                .toList();

        List<GdDefect> gridWarnings = StreamEx.of(vr.flattenWarnings(pathNodeConverterProvider))
                .map(di -> di.convertPath(PathConverter.prepend(prefixPath)))
                .map(GridValidationResultConversionService::convert)
                .distinct()
                .toList();

        return new GdValidationResult()
                .withErrors(gridErrors)
                .withWarnings(gridWarnings);
    }

    private static GdDefect convert(DefectInfo<? extends Defect> defectInfo) {
        return new GdDefect()
                .withCode(defectInfo.getDefect().defectId().getCode())
                .withPath(defectInfo.getPath().toString())
                .withParams(defectInfo.getDefect().params());
    }

    /**
     * Конвертировать validationResult в null, если это пустой объект.
     * т.к. фронт ожидает null вместо пустого объекта.
     *
     * @param vr - результат валидации
     * @return null если объект пуст
     */
    @Nullable
    public static GdValidationResult convertEmptyToNull(@Nullable GdValidationResult vr) {
        if (vr != null && vr.getErrors().isEmpty() && vr.getWarnings().isEmpty()) {
            return null;
        }
        return vr;
    }
}
