package ru.yandex.direct.grid.processing.service.validation;

import java.math.BigDecimal;
import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.function.Predicate;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.github.shyiko.mysql.binlog.GtidSet;
import com.google.common.base.CharMatcher;
import com.google.common.collect.ImmutableList;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.SetUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.grid.core.entity.deal.model.GdiDealFilter;
import ru.yandex.direct.grid.model.GdEntityStatsFilter;
import ru.yandex.direct.grid.model.GdGoalStatsFilter;
import ru.yandex.direct.grid.model.GdOfferStatsFilter;
import ru.yandex.direct.grid.model.GdOrderByParams;
import ru.yandex.direct.grid.model.GdStatRequirements;
import ru.yandex.direct.grid.model.jsonsettings.GdGetJsonSettings;
import ru.yandex.direct.grid.model.jsonsettings.GdSetJsonSettings;
import ru.yandex.direct.grid.processing.exception.GridValidationException;
import ru.yandex.direct.grid.processing.model.GdLimitOffset;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.banner.GdAdFilter;
import ru.yandex.direct.grid.processing.model.banner.GdAdsContainer;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceAdsDisplayHref;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceAdsHrefDomain;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceAdsHrefDomainInstruction;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceAdsHrefParams;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceAdsHrefParamsInstruction;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceAdsHrefTargetType;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceAdsHrefText;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceAdsTargetType;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceCallouts;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceCalloutsInstruction;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceHrefTextInstruction;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceOptions;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceText;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceTextInstruction;
import ru.yandex.direct.grid.processing.model.campaign.GdCampaignFilter;
import ru.yandex.direct.grid.processing.model.campaign.GdCampaignsContainer;
import ru.yandex.direct.grid.processing.model.cashback.GdCashbackRewardsDetailsInput;
import ru.yandex.direct.grid.processing.model.client.GdCheckClientMutationState;
import ru.yandex.direct.grid.processing.model.client.GdClientSearchRequest;
import ru.yandex.direct.grid.processing.model.client.GdGetClientMutationId;
import ru.yandex.direct.grid.processing.model.cliententity.mutation.GdAddClientMeasurerAccount;
import ru.yandex.direct.grid.processing.model.common.GdOrderingItemWithParams;
import ru.yandex.direct.grid.processing.model.group.GdAdGroupFilter;
import ru.yandex.direct.grid.processing.model.group.GdAdGroupsContainer;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddAdGroupMinusKeywords;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddAdGroupMinusKeywordsItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddAdGroupsMinusKeywords;
import ru.yandex.direct.grid.processing.model.group.mutation.GdRemoveAdGroupsMinusKeywords;
import ru.yandex.direct.grid.processing.model.group.mutation.GdReplaceAdGroupsMinusKeywords;
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFilterFilter;
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFiltersContainer;
import ru.yandex.direct.grid.processing.model.strategy.query.GdPackageStrategiesContainer;
import ru.yandex.direct.grid.processing.model.strategy.query.GdPackageStrategyFilter;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.utils.CommonUtils;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static com.google.common.base.CharMatcher.anyOf;
import static ru.yandex.direct.core.entity.banner.type.displayhref.BannerWithDisplayHrefValidatorProvider.displayHrefValidator;
import static ru.yandex.direct.core.entity.banner.type.href.BannerWithHrefConstraints.domainMatchesAllowedPattern;
import static ru.yandex.direct.core.entity.cashback.CashbackConstants.DETALIZATION_MAX_LENGTH;
import static ru.yandex.direct.core.entity.cashback.CashbackConstants.DETALIZATION_MIN_LENGTH;
import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;
import static ru.yandex.direct.core.validation.constraints.Constraints.validLogin;
import static ru.yandex.direct.grid.processing.model.banner.mutation.GdReplacementMode.DELETE;
import static ru.yandex.direct.grid.processing.model.banner.mutation.GdReplacementMode.FIND_AND_REPLACE;
import static ru.yandex.direct.grid.processing.model.banner.mutation.GdReplacementMode.REPLACE_ALL;
import static ru.yandex.direct.grid.processing.service.client.validation.ClientDataValidationService.ADD_CLIENT_MEASURER_ACCOUNT_VALIDATOR;
import static ru.yandex.direct.grid.processing.service.jsonsettings.JsonSettingsValidationUtils.allowedIdType;
import static ru.yandex.direct.grid.processing.service.jsonsettings.JsonSettingsValidationUtils.getJsonSettingsUnionValidator;
import static ru.yandex.direct.grid.processing.service.jsonsettings.JsonSettingsValidationUtils.validJsonPath;
import static ru.yandex.direct.grid.processing.service.validation.GridDefectDefinitions.datesFromNotAfterTo;
import static ru.yandex.direct.grid.processing.service.validation.GridDefectDefinitions.invalidFindAndReplaceBannersHrefParamsText;
import static ru.yandex.direct.grid.processing.service.validation.GridDefectDefinitions.mustContainNonNullProps;
import static ru.yandex.direct.grid.processing.service.validation.GridDefectDefinitions.mutuallyExclusive;
import static ru.yandex.direct.grid.processing.service.validation.GridDefectDefinitions.sameNullState;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.validation.Predicates.each;
import static ru.yandex.direct.validation.Predicates.not;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.eachInSetNotNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.eachNotNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.DateConstraints.isNotOlderThan;
import static ru.yandex.direct.validation.constraint.NumberConstraints.inRange;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notLessThan;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.constraint.StringConstraints.validHref;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;
import static ru.yandex.direct.validation.result.PathHelper.emptyPath;
import static ru.yandex.direct.validation.result.PathHelper.index;
import static ru.yandex.direct.validation.result.ValidationResult.transferSubNodesWithIssues;

/**
 * Сервис, предназначенный для валидации всех поступающих в гриды в запросах внешних данных. Здесь не валидируется то,
 * что уже должно быть проверено GraphQL-фреимворком
 */
@ParametersAreNonnullByDefault
@Service
@SuppressWarnings("rawtypes")
public class GridValidationService {  //todo DIRECT-83062: Разбить GridValidationService на классы

    public static final Duration MAX_TO_STAT_PERIOD = Duration.ofDays(365);
    public static final Duration MAX_FROM_STAT_PERIOD = Duration.ofDays(365);

    private static final CharMatcher ALLOW_BANNERS_HREF_PARAMS_KEY_MATCHER = anyOf("&=#/?");
    private static final CharMatcher ALLOW_BANNERS_HREF_PARAMS_VALUE_MATCHER = anyOf("&#");
    public static final int MAX_FIND_AND_REPLACE_BANNERS_HREF_PARAMS = 512;


    private final GridValidationResultConversionService validationResultConversionService;

    @Autowired
    public GridValidationService(GridValidationResultConversionService validationResultConversionService) {
        this.validationResultConversionService = validationResultConversionService;
    }

    public static final Validator<Set<Long>, Defect> IDS_COLLECTION_VALIDATOR = ids -> {
        ItemValidationBuilder<Set<Long>, Defect> vb = ItemValidationBuilder.of(ids);
        vb.check(Constraint.fromPredicate(each(CommonUtils::isValidId), CommonDefects.validId()),
                When.notNull());
        return vb.getResult();
    };

    public static final Validator<GdLimitOffset, Defect> LIMIT_OFFSET_VALIDATOR = gdLimitOffset -> {
        ModelItemValidationBuilder<GdLimitOffset> vb = ModelItemValidationBuilder.of(gdLimitOffset);
        vb.item(GdLimitOffset.LIMIT)
                .check(notLessThan(1));
        vb.item(GdLimitOffset.OFFSET)
                .check(notLessThan(0));
        return vb.getResult();
    };

    public static final Validator<GdStatRequirements, Defect> STAT_REQUIREMENTS_VALIDATOR = statReqs -> {
        ModelItemValidationBuilder<GdStatRequirements> vb = ModelItemValidationBuilder.of(statReqs);
        vb.check(notNull());
        vb.check(checkMutuallyExclusiveFieldGroups(
                ImmutableList.of(GdStatRequirements.FROM, GdStatRequirements.TO),
                ImmutableList.of(GdStatRequirements.PRESET),
                mutuallyExclusive()), When.notNull());
        vb.check(checkBothFieldsNullOrNotNull(GdStatRequirements.FROM, GdStatRequirements.TO,
                sameNullState()), When.notNull());
        vb.item(GdStatRequirements.FROM)
                .check(isNotOlderThan(MAX_FROM_STAT_PERIOD.toDays()), When.notNull());
        vb.item(GdStatRequirements.TO)
                .check(isNotOlderThan(MAX_TO_STAT_PERIOD.toDays()), When.notNull());
        vb.check(fromFieldIsNotAfterToField(), When.isValid());
        return vb.getResult();
    };

    public static final Validator<GdEntityStatsFilter, Defect> STATS_FILTER_VALIDATOR = filter -> {
        ModelItemValidationBuilder<GdEntityStatsFilter> vb = ModelItemValidationBuilder.of(filter);
        vb.item(GdEntityStatsFilter.MIN_COST)
                .check(notLessThan(BigDecimal.ZERO), When.notNull());
        vb.item(GdEntityStatsFilter.MAX_COST)
                .check(notLessThan(BigDecimal.ZERO), When.notNull());
        vb.item(GdEntityStatsFilter.MIN_SHOWS)
                .check(notLessThan(0L), When.notNull());
        vb.item(GdEntityStatsFilter.MAX_SHOWS)
                .check(notLessThan(0L), When.notNull());
        vb.item(GdEntityStatsFilter.MIN_CLICKS)
                .check(notLessThan(0L), When.notNull());
        vb.item(GdEntityStatsFilter.MAX_CLICKS)
                .check(notLessThan(0L), When.notNull());
        vb.item(GdEntityStatsFilter.MIN_CTR)
                .check(notLessThan(BigDecimal.ZERO), When.notNull());
        vb.item(GdEntityStatsFilter.MAX_CTR)
                .check(notLessThan(BigDecimal.ZERO), When.notNull());
        vb.item(GdEntityStatsFilter.MIN_AVG_CLICK_COST)
                .check(notLessThan(BigDecimal.ZERO), When.notNull());
        vb.item(GdEntityStatsFilter.MAX_AVG_CLICK_COST)
                .check(notLessThan(BigDecimal.ZERO), When.notNull());
        vb.item(GdEntityStatsFilter.MIN_AVG_SHOW_POSITION)
                .check(notLessThan(BigDecimal.ZERO), When.notNull());
        vb.item(GdEntityStatsFilter.MAX_AVG_SHOW_POSITION)
                .check(notLessThan(BigDecimal.ZERO), When.notNull());
        vb.item(GdEntityStatsFilter.MIN_AVG_CLICK_POSITION)
                .check(notLessThan(BigDecimal.ZERO), When.notNull());
        vb.item(GdEntityStatsFilter.MAX_AVG_CLICK_POSITION)
                .check(notLessThan(BigDecimal.ZERO), When.notNull());
        vb.item(GdEntityStatsFilter.MIN_BOUNCE_RATE)
                .check(notLessThan(BigDecimal.ZERO), When.notNull());
        vb.item(GdEntityStatsFilter.MAX_BOUNCE_RATE)
                .check(notLessThan(BigDecimal.ZERO), When.notNull());
        vb.item(GdEntityStatsFilter.MIN_CONVERSION_RATE)
                .check(notLessThan(BigDecimal.ZERO), When.notNull());
        vb.item(GdEntityStatsFilter.MAX_CONVERSION_RATE)
                .check(notLessThan(BigDecimal.ZERO), When.notNull());
        vb.item(GdEntityStatsFilter.MIN_AVG_GOAL_COST)
                .check(notLessThan(BigDecimal.ZERO), When.notNull());
        vb.item(GdEntityStatsFilter.MAX_AVG_GOAL_COST)
                .check(notLessThan(BigDecimal.ZERO), When.notNull());
        vb.item(GdEntityStatsFilter.MIN_GOALS)
                .check(notLessThan(0L), When.notNull());
        vb.item(GdEntityStatsFilter.MAX_GOALS)
                .check(notLessThan(0L), When.notNull());
        vb.item(GdEntityStatsFilter.MIN_AVG_DEPTH)
                .check(notLessThan(BigDecimal.ZERO), When.notNull());
        vb.item(GdEntityStatsFilter.MAX_AVG_DEPTH)
                .check(notLessThan(BigDecimal.ZERO), When.notNull());
        return vb.getResult();
    };

    public static final Validator<GdOfferStatsFilter, Defect> OFFER_STATS_FILTER_VALIDATOR = filter -> {
        ModelItemValidationBuilder<GdOfferStatsFilter> vb = ModelItemValidationBuilder.of(filter);
        vb.item(GdOfferStatsFilter.MIN_SHOWS)
                .check(notLessThan(0L), When.notNull());
        vb.item(GdOfferStatsFilter.MAX_SHOWS)
                .check(notLessThan(0L), When.notNull());
        vb.item(GdOfferStatsFilter.MIN_CLICKS)
                .check(notLessThan(0L), When.notNull());
        vb.item(GdOfferStatsFilter.MAX_CLICKS)
                .check(notLessThan(0L), When.notNull());
        vb.item(GdOfferStatsFilter.MIN_CTR)
                .check(notLessThan(BigDecimal.ZERO), When.notNull());
        vb.item(GdOfferStatsFilter.MAX_CTR)
                .check(notLessThan(BigDecimal.ZERO), When.notNull());
        vb.item(GdOfferStatsFilter.MIN_COST)
                .check(notLessThan(BigDecimal.ZERO), When.notNull());
        vb.item(GdOfferStatsFilter.MAX_COST)
                .check(notLessThan(BigDecimal.ZERO), When.notNull());
        vb.item(GdOfferStatsFilter.MIN_CARTS)
                .check(notLessThan(0L), When.notNull());
        vb.item(GdOfferStatsFilter.MAX_CARTS)
                .check(notLessThan(0L), When.notNull());
        vb.item(GdOfferStatsFilter.MIN_PURCHASES)
                .check(notLessThan(0L), When.notNull());
        vb.item(GdOfferStatsFilter.MAX_PURCHASES)
                .check(notLessThan(0L), When.notNull());
        vb.item(GdOfferStatsFilter.MIN_AVG_CLICK_COST)
                .check(notLessThan(BigDecimal.ZERO), When.notNull());
        vb.item(GdOfferStatsFilter.MAX_AVG_CLICK_COST)
                .check(notLessThan(BigDecimal.ZERO), When.notNull());
        vb.item(GdOfferStatsFilter.MIN_AVG_PRODUCT_PRICE)
                .check(notLessThan(BigDecimal.ZERO), When.notNull());
        vb.item(GdOfferStatsFilter.MAX_AVG_PRODUCT_PRICE)
                .check(notLessThan(BigDecimal.ZERO), When.notNull());
        vb.item(GdOfferStatsFilter.MIN_AVG_PURCHASE_REVENUE)
                .check(notLessThan(BigDecimal.ZERO), When.notNull());
        vb.item(GdOfferStatsFilter.MAX_AVG_PURCHASE_REVENUE)
                .check(notLessThan(BigDecimal.ZERO), When.notNull());
        return vb.getResult();
    };

    private static final Validator<GdCampaignFilter, Defect> CAMPAIGN_FILTER_VALIDATOR = filter -> {
        ModelItemValidationBuilder<GdCampaignFilter> vb = ModelItemValidationBuilder.of(filter);
        vb.item(GdCampaignFilter.CAMPAIGN_ID_IN)
                .checkBy(IDS_COLLECTION_VALIDATOR, When.notNull());
        vb.item(GdCampaignFilter.CAMPAIGN_ID_NOT_IN)
                .checkBy(IDS_COLLECTION_VALIDATOR, When.notNull());
        vb.item(GdCampaignFilter.STATS)
                .checkBy(STATS_FILTER_VALIDATOR, When.notNull());
        return vb.getResult();
    };

    private static final Validator<GdPackageStrategyFilter, Defect> PACKAGE_STRATEGY_FILTER_VALIDATOR = filter -> {
        ModelItemValidationBuilder<GdPackageStrategyFilter> vb = ModelItemValidationBuilder.of(filter);
        vb.item(GdPackageStrategyFilter.STRATEGY_ID_IN)
                .checkBy(IDS_COLLECTION_VALIDATOR, When.notNull());
        vb.item(GdPackageStrategyFilter.STRATEGY_ID_NOT_IN)
                .checkBy(IDS_COLLECTION_VALIDATOR, When.notNull());
        return vb.getResult();
    };

    private static final Validator<GdCampaignsContainer, Defect> CAMPAIGNS_CONTAINER_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdCampaignsContainer> vb = ModelItemValidationBuilder.of(req);

                vb.item(GdCampaignsContainer.FILTER)
                        .check(notNull(), When.isTrue(req.getFilterKey() == null));
                vb.item(GdCampaignsContainer.FILTER_KEY)
                        .check(notNull(), When.isTrue(req.getFilter() == null));

                vb.item(GdCampaignsContainer.FILTER)
                        .checkBy(CAMPAIGN_FILTER_VALIDATOR, When.notNull());
                vb.item(GdCampaignsContainer.FILTER_KEY)
                        .check(notBlank(), When.notNull());

                vb.item(GdCampaignsContainer.STAT_REQUIREMENTS)
                        .checkBy(STAT_REQUIREMENTS_VALIDATOR);
                vb.item(GdCampaignsContainer.LIMIT_OFFSET)
                        .checkBy(LIMIT_OFFSET_VALIDATOR);
                vb.item(GdCampaignsContainer.CACHE_KEY)
                        .check(notBlank(), When.notNull());

                vb.item(GdCampaignsContainer.ORDER_BY)
                        .checkByFunction(orderByList ->
                                        checkOrderByForGoalsHasGoalIdsInRequest(orderByList, req.getStatRequirements()),
                                When.notNull());

                return vb.getResult();
            };

    private static final Validator<GdPackageStrategiesContainer, Defect> PACKAGE_STRATEGIES_CONTAINER_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdPackageStrategiesContainer> vb = ModelItemValidationBuilder.of(req);

                vb.item(GdPackageStrategiesContainer.FILTER)
                        .check(isNull(), When.isTrue(req.getFilterKey() != null));

                vb.item(GdPackageStrategiesContainer.FILTER)
                        .checkBy(PACKAGE_STRATEGY_FILTER_VALIDATOR, When.notNull());
                vb.item(GdPackageStrategiesContainer.FILTER_KEY)
                        .check(notBlank(), When.notNull());

                vb.item(GdPackageStrategiesContainer.LIMIT_OFFSET)
                        .checkBy(LIMIT_OFFSET_VALIDATOR);
                vb.item(GdPackageStrategiesContainer.CACHE_KEY)
                        .check(notBlank(), When.notNull());

                return vb.getResult();
            };


    private static final Validator<GdAdGroupFilter, Defect> AD_GROUP_FILTER_VALIDATOR = filter -> {
        ModelItemValidationBuilder<GdAdGroupFilter> vb = ModelItemValidationBuilder.of(filter);
        vb.item(GdAdGroupFilter.CAMPAIGN_ID_IN)
                .checkBy(IDS_COLLECTION_VALIDATOR, When.notNull());

        vb.item(GdAdGroupFilter.AD_GROUP_ID_IN)
                .checkBy(IDS_COLLECTION_VALIDATOR, When.notNull());
        vb.item(GdAdGroupFilter.AD_GROUP_ID_NOT_IN)
                .checkBy(IDS_COLLECTION_VALIDATOR, When.notNull());
        vb.item(GdAdGroupFilter.STATS)
                .checkBy(STATS_FILTER_VALIDATOR, When.notNull());
        vb.item(GdAdGroupFilter.LIBRARY_MW_ID_IN)
                .checkBy(IDS_COLLECTION_VALIDATOR, When.notNull());
        return vb.getResult();
    };

    /**
     * Проверяет, что если есть сортировка по полям для целей, то в запросе для этих целей должен быть
     * id в {@link GdStatRequirements}
     * id для целей нужны, чтобы в запросе в YT могли делать джойны к нужным таблицам
     */
    @Nullable
    public static Defect checkOrderByForGoalsHasGoalIdsInRequest(List<? extends GdOrderingItemWithParams> orderByList,
                                                                 GdStatRequirements gdStatRequirements) {
        Set<Long> orderByGoalIds = StreamEx.of(orderByList)
                .map(GdOrderingItemWithParams::getParams)
                .nonNull()
                .map(GdOrderByParams::getGoalId)
                .nonNull()
                .toSet();
        Set<Long> requestGoalIds = nvl(gdStatRequirements.getGoalIds(), Collections.emptySet());

        Set<Long> difference = SetUtils.difference(orderByGoalIds, requestGoalIds);
        return difference.isEmpty() ? null : GridDefectDefinitions.invalidGoalIdsFromOrderBy(difference);
    }

    private static final Validator<GdAdGroupsContainer, Defect> AD_GROUPS_CONTAINER_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdAdGroupsContainer> vb = ModelItemValidationBuilder.of(req);

                vb.item(GdAdGroupsContainer.FILTER)
                        .check(notNull(), When.isTrue(req.getFilterKey() == null));
                vb.item(GdAdGroupsContainer.FILTER_KEY)
                        .check(notNull(), When.isTrue(req.getFilter() == null));

                vb.item(GdAdGroupsContainer.FILTER)
                        .checkBy(AD_GROUP_FILTER_VALIDATOR, When.notNull());
                vb.item(GdAdGroupsContainer.FILTER_KEY)
                        .check(notBlank(), When.notNull());

                vb.item(GdAdGroupsContainer.STAT_REQUIREMENTS)
                        .checkBy(STAT_REQUIREMENTS_VALIDATOR);
                vb.item(GdAdGroupsContainer.LIMIT_OFFSET)
                        .checkBy(LIMIT_OFFSET_VALIDATOR);
                vb.item(GdAdGroupsContainer.CACHE_KEY)
                        .check(notBlank(), When.notNull());

                vb.item(GdAdGroupsContainer.ORDER_BY)
                        .checkByFunction(orderByList ->
                                        checkOrderByForGoalsHasGoalIdsInRequest(orderByList, req.getStatRequirements()),
                                When.notNull());

                return vb.getResult();
            };

    private static final Validator<GdAdFilter, Defect> AD_FILTER_VALIDATOR = filter -> {
        ModelItemValidationBuilder<GdAdFilter> vb = ModelItemValidationBuilder.of(filter);
        vb.item(GdAdFilter.CAMPAIGN_ID_IN)
                .checkBy(IDS_COLLECTION_VALIDATOR, When.notNull());
        vb.item(GdAdFilter.AD_GROUP_ID_IN)
                .checkBy(IDS_COLLECTION_VALIDATOR, When.notNull());

        vb.item(GdAdFilter.AD_ID_IN)
                .checkBy(IDS_COLLECTION_VALIDATOR, When.notNull());
        vb.item(GdAdFilter.AD_ID_NOT_IN)
                .checkBy(IDS_COLLECTION_VALIDATOR, When.notNull());
        vb.item(GdAdFilter.STATS)
                .checkBy(STATS_FILTER_VALIDATOR, When.notNull());
        return vb.getResult();
    };
    private static final Validator<GdAdsContainer, Defect> ADS_CONTAINER_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdAdsContainer> vb = ModelItemValidationBuilder.of(req);

                vb.item(GdAdsContainer.FILTER)
                        .check(notNull(), When.isTrue(req.getFilterKey() == null));
                vb.item(GdAdsContainer.FILTER_KEY)
                        .check(notNull(), When.isTrue(req.getFilter() == null));

                vb.item(GdAdsContainer.FILTER)
                        .checkBy(AD_FILTER_VALIDATOR, When.notNull());
                vb.item(GdAdsContainer.FILTER_KEY)
                        .check(notBlank(), When.notNull());

                vb.item(GdAdsContainer.STAT_REQUIREMENTS)
                        .checkBy(STAT_REQUIREMENTS_VALIDATOR);
                vb.item(GdAdsContainer.LIMIT_OFFSET)
                        .checkBy(LIMIT_OFFSET_VALIDATOR);
                vb.item(GdAdsContainer.CACHE_KEY)
                        .check(notBlank(), When.notNull());

                vb.item(GdAdsContainer.ORDER_BY)
                        .checkByFunction(orderByList ->
                                        checkOrderByForGoalsHasGoalIdsInRequest(orderByList, req.getStatRequirements()),
                                When.notNull());

                return vb.getResult();
            };

    private static final Validator<GdiDealFilter, Defect> DEAL_FILTER_VALIDATOR = filter -> {
        ModelItemValidationBuilder<GdiDealFilter> vb = ModelItemValidationBuilder.of(filter);
        vb.item(GdiDealFilter.DEAL_ID_EXCLUDED)
                .checkBy(IDS_COLLECTION_VALIDATOR, When.notNull());
        vb.item(GdiDealFilter.DEAL_ID_INCLUDED)
                .checkBy(IDS_COLLECTION_VALIDATOR, When.notNull());
        return vb.getResult();
    };

    private static final Validator<GdClientSearchRequest, Defect> CLIENT_SEARCH_REQUEST_VALIDATOR_NOT_NULL =
            req -> {
                ModelItemValidationBuilder<GdClientSearchRequest> vb = ModelItemValidationBuilder.of(req);
                vb.check(checkMutuallyExclusiveFields(List.of(
                        GdClientSearchRequest.ID, GdClientSearchRequest.USER_ID, GdClientSearchRequest.LOGIN)));
                vb.item(GdClientSearchRequest.ID)
                        .check(validId(), When.notNull());
                vb.item(GdClientSearchRequest.USER_ID)
                        .check(validId(), When.notNull());
                vb.item(GdClientSearchRequest.LOGIN)
                        .check(validLogin(), When.notNull());
                return vb.getResult();
            };

    public static final Validator<GdClientSearchRequest, Defect> CLIENT_SEARCH_REQUEST_VALIDATOR = req -> {
        ModelItemValidationBuilder<GdClientSearchRequest> vb = ModelItemValidationBuilder.of(req);
        vb.check(notNull());
        vb.checkBy(CLIENT_SEARCH_REQUEST_VALIDATOR_NOT_NULL, When.isValid());
        return vb.getResult();
    };

    private static final Validator<GdGoalStatsFilter, Defect> GOAL_STATS_FILTER_VALIDATOR =
            filter -> {
                ModelItemValidationBuilder<GdGoalStatsFilter> vb = ModelItemValidationBuilder.of(filter);
                if (filter == null) {
                    return vb.getResult();
                }
                vb.item(GdGoalStatsFilter.GOAL_ID)
                        .check(notNull());
                vb.item(GdGoalStatsFilter.MIN_GOALS)
                        .check(notLessThan(0L));
                vb.item(GdGoalStatsFilter.MIN_CONVERSION_RATE)
                        .check(notLessThan(BigDecimal.ZERO));
                vb.item(GdGoalStatsFilter.MIN_COST_PER_ACTION)
                        .check(notLessThan(BigDecimal.ZERO));
                return vb.getResult();
            };

    private static final Validator<List<GdGoalStatsFilter>, Defect> GOAL_LIST_STATS_FILTER_VALIDATOR =
            filter -> {
                ListValidationBuilder<GdGoalStatsFilter, Defect> vb = ListValidationBuilder.of(filter);
                vb.checkEachBy(GOAL_STATS_FILTER_VALIDATOR);
                return vb.getResult();
            };

    private static final Validator<GdSmartFilterFilter, Defect> SMART_FILTER_FILTER_VALIDATOR =
            filter -> {
                ModelItemValidationBuilder<GdSmartFilterFilter> vb = ModelItemValidationBuilder.of(filter);
                vb.item(GdSmartFilterFilter.CAMPAIGN_ID_IN)
                        .checkBy(IDS_COLLECTION_VALIDATOR, When.notNull());
                vb.item(GdSmartFilterFilter.AD_GROUP_ID_IN)
                        .checkBy(IDS_COLLECTION_VALIDATOR, When.notNull());
                vb.item(GdSmartFilterFilter.CAMPAIGN_ID_IN)
                        .checkBy(IDS_COLLECTION_VALIDATOR, When.notNull());

                vb.item(GdSmartFilterFilter.MIN_PRICE_CPA)
                        .check(notLessThan(BigDecimal.ZERO));
                vb.item(GdSmartFilterFilter.MIN_PRICE_CPC)
                        .check(notLessThan(BigDecimal.ZERO));
                vb.item(GdSmartFilterFilter.AUTOBUDGET_PRIORITY_IN)
                        .check(eachInSetNotNull());
                vb.item(GdSmartFilterFilter.STATS)
                        .checkBy(STATS_FILTER_VALIDATOR, When.notNull());
                vb.item(GdSmartFilterFilter.GOAL_STATS)
                        .checkBy(GOAL_LIST_STATS_FILTER_VALIDATOR);
                return vb.getResult();
            };

    private static final Validator<GdSmartFiltersContainer, Defect> SMART_FILTER_CONTAINER_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdSmartFiltersContainer> vb = ModelItemValidationBuilder.of(req);

                vb.item(GdSmartFiltersContainer.FILTER)
                        .check(notNull(), When.isTrue(req.getFilterKey() == null));
                vb.item(GdSmartFiltersContainer.FILTER_KEY)
                        .check(notNull(), When.isTrue(req.getFilter() == null));

                vb.item(GdSmartFiltersContainer.FILTER)
                        .checkBy(SMART_FILTER_FILTER_VALIDATOR, When.notNull());
                vb.item(GdSmartFiltersContainer.FILTER_KEY)
                        .check(notBlank(), When.notNull());

                vb.item(GdSmartFiltersContainer.STAT_REQUIREMENTS)
                        .checkBy(STAT_REQUIREMENTS_VALIDATOR, When.notNull());
                vb.item(GdSmartFiltersContainer.LIMIT_OFFSET)
                        .checkBy(LIMIT_OFFSET_VALIDATOR, When.notNull());
                vb.item(GdSmartFiltersContainer.CACHE_KEY)
                        .check(notBlank(), When.notNull());

                vb.item(GdSmartFiltersContainer.ORDER_BY)
                        .checkByFunction(orderByList ->
                                        checkOrderByForGoalsHasGoalIdsInRequest(orderByList, req.getStatRequirements()),
                                When.notNull());

                return vb.getResult();
            };

    private static final Validator<List<Long>, Defect> AD_IDS_VALIDATOR = req -> {
        ListValidationBuilder<Long, Defect> vb = ListValidationBuilder.of(req);

        vb
                .check(notNull())
                .check(notEmptyCollection())
                .checkEach(notNull(), When.isValid())
                .checkEach(validId(), When.isValid())
                .checkEach(unique(), When.isValid());
        return vb.getResult();
    };

    private static final Validator<Set<GdFindAndReplaceAdsHrefTargetType>, Defect> ADS_HREF_TARGET_TYPES_VALIDATOR =
            targetTypes -> ItemValidationBuilder.<Set<GdFindAndReplaceAdsHrefTargetType>, Defect>of(targetTypes)
                    .check(notEmptyCollection())
                    .check(eachInSetNotNull())
                    .getResult();

    private static final Validator<Set<GdFindAndReplaceAdsTargetType>, Defect> ADS_TEXT_TARGET_TYPES_VALIDATOR =
            targetTypes -> ItemValidationBuilder.<Set<GdFindAndReplaceAdsTargetType>, Defect>of(targetTypes)
                    .check(notEmptyCollection())
                    .check(eachInSetNotNull())
                    .getResult();

    private static final Validator<GdFindAndReplaceAdsHrefParamsInstruction, Defect>
            FIND_AND_REPLACE_AD_HREF_PARAMS_INSTRUCTION_VALIDATOR =
            gdFindAndReplaceBannersHrefParamsInstruction -> {
                ModelItemValidationBuilder<GdFindAndReplaceAdsHrefParamsInstruction> vb =
                        ModelItemValidationBuilder.of(gdFindAndReplaceBannersHrefParamsInstruction);

                vb.check(checkFieldsContainNonNullValue(
                        ImmutableList.of(
                                GdFindAndReplaceAdsHrefParamsInstruction.REPLACE_KEY,
                                GdFindAndReplaceAdsHrefParamsInstruction.REPLACE_VALUE)), When.isValid());

                vb.item(GdFindAndReplaceAdsHrefParamsInstruction.SEARCH_KEY)
                        .check(notNull())
                        .check(notBlank())
                        .check(maxStringLength(MAX_FIND_AND_REPLACE_BANNERS_HREF_PARAMS))
                        .check(bannersHrefParamsKeyCharsAreAllowed(), When.isValid());
                vb.item(GdFindAndReplaceAdsHrefParamsInstruction.REPLACE_KEY)
                        .check(notBlank())
                        .check(maxStringLength(MAX_FIND_AND_REPLACE_BANNERS_HREF_PARAMS))
                        .check(bannersHrefParamsKeyCharsAreAllowed(), When.isValid());
                vb.item(GdFindAndReplaceAdsHrefParamsInstruction.REPLACE_VALUE)
                        .check(maxStringLength(MAX_FIND_AND_REPLACE_BANNERS_HREF_PARAMS))
                        .check(bannersHrefParamsValueCharsAreAllowed(), When.isValid());
                return vb.getResult();
            };

    private static final Validator<GdFindAndReplaceAdsHrefParams, Defect>
            FIND_AND_REPLACE_ADS_HREF_PARAMS_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdFindAndReplaceAdsHrefParams> vb = ModelItemValidationBuilder.of(req);

                vb.item(GdFindAndReplaceAdsHrefParams.AD_IDS).checkBy(AD_IDS_VALIDATOR);

                vb.item(GdFindAndReplaceAdsHrefParams.REPLACE_INSTRUCTION)
                        .check(notNull())
                        .checkBy(FIND_AND_REPLACE_AD_HREF_PARAMS_INSTRUCTION_VALIDATOR, When.notNull());

                vb.item(GdFindAndReplaceAdsHrefParams.TARGET_TYPES)
                        .check(notNull())
                        .checkBy(ADS_HREF_TARGET_TYPES_VALIDATOR);

                return vb.getResult();
            };


    private static final Validator<GdFindAndReplaceAdsHrefDomainInstruction, Defect>
            FIND_AND_REPLACE_ADS_HREF_DOMAIN_INSTRUCTION_VALIDATOR =
            instruction -> {
                ModelItemValidationBuilder<GdFindAndReplaceAdsHrefDomainInstruction> vb =
                        ModelItemValidationBuilder.of(instruction);

                vb.list(GdFindAndReplaceAdsHrefDomainInstruction.SEARCH)
                        .check(notNull())
                        .check(notEmptyCollection())
                        .check(eachNotNull())
                        .checkEach(notBlank())
                        .checkEach(domainMatchesAllowedPattern(), When.isValidAnd(When.notNull()));

                vb.item(GdFindAndReplaceAdsHrefDomainInstruction.REPLACE)
                        .check(notNull())
                        .check(notBlank())
                        .check(domainMatchesAllowedPattern(), When.isValidAnd(When.notNull()));
                return vb.getResult();
            };

    private static final Validator<GdFindAndReplaceOptions, Defect> FIND_AND_REPLACE_OPTIONS_VALIDATOR =
            options -> {
                ModelItemValidationBuilder<GdFindAndReplaceOptions> vb = ModelItemValidationBuilder.of(options);

                vb.item(GdFindAndReplaceOptions.CASE_SENSITIVE)
                        .check(notNull());

                vb.item(GdFindAndReplaceOptions.LINK_REPLACEMENT_MODE)
                        .check(notNull());

                return vb.getResult();
            };

    private static final Validator<GdFindAndReplaceTextInstruction, Defect>
            FIND_AND_REPLACE_BANNER_TEXT_INSTRUCTION_VALIDATOR =
            instruction -> {
                ModelItemValidationBuilder<GdFindAndReplaceTextInstruction> vb =
                        ModelItemValidationBuilder.of(instruction);
                var replacementMode = instruction.getOptions().getReplacementMode();

                vb.item(GdFindAndReplaceTextInstruction.SEARCH)
                        .check(notNull(), When.isTrue(instruction.getReplace() == null &&
                                (replacementMode != REPLACE_ALL && replacementMode != DELETE)))
                        .check(notNull(), When.isTrue(replacementMode == FIND_AND_REPLACE));

                vb.item(GdFindAndReplaceTextInstruction.REPLACE)
                        .check(notNull(), When.isTrue(instruction.getSearch() == null &&
                                (replacementMode != REPLACE_ALL && replacementMode != DELETE)))
                        .check(notNull(), When.isTrue(replacementMode != FIND_AND_REPLACE && replacementMode != DELETE
                                && replacementMode != REPLACE_ALL));

                vb.item(GdFindAndReplaceTextInstruction.OPTIONS)
                        .check(notNull())
                        .checkBy(FIND_AND_REPLACE_OPTIONS_VALIDATOR);

                return vb.getResult();
            };

    private static final Validator<GdFindAndReplaceAdsHrefDomain, Defect>
            FIND_AND_REPLACE_ADS_HREF_DOMAIN_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdFindAndReplaceAdsHrefDomain> vb = ModelItemValidationBuilder.of(req);

                vb.item(GdFindAndReplaceAdsHrefDomain.AD_IDS).checkBy(AD_IDS_VALIDATOR);

                vb.item(GdFindAndReplaceAdsHrefDomain.TARGET_TYPES)
                        .check(notNull())
                        .checkBy(ADS_HREF_TARGET_TYPES_VALIDATOR);

                vb.item(GdFindAndReplaceAdsHrefDomain.REPLACE_INSTRUCTION)
                        .check(notNull())
                        .checkBy(FIND_AND_REPLACE_ADS_HREF_DOMAIN_INSTRUCTION_VALIDATOR, When.notNull());

                return vb.getResult();
            };

    private static final Validator<GdFindAndReplaceHrefTextInstruction, Defect>
            FIND_AND_REPLACE_BANNER_HREF_TEXT_INSTRUCTION_VALIDATOR =
            gdFindAndReplaceBannersHrefParamsInstruction -> {
                ModelItemValidationBuilder<GdFindAndReplaceHrefTextInstruction> vb =
                        ModelItemValidationBuilder.of(gdFindAndReplaceBannersHrefParamsInstruction);

                vb.item(GdFindAndReplaceHrefTextInstruction.HREF_PARTS)
                        .check(notNull())
                        .check(notEmptyCollection())
                        .check(eachInSetNotNull());

                vb.item(GdFindAndReplaceHrefTextInstruction.SEARCH)
                        .check(notBlank());
                vb.item(GdFindAndReplaceHrefTextInstruction.REPLACE)
                        .check(notNull())
                        .check(notBlank());
                return vb.getResult();
            };

    private static final Validator<GdFindAndReplaceAdsHrefText, Defect>
            FIND_AND_REPLACE_BANNER_HREF_TEXT_VALIDATOR = req -> {
        ModelItemValidationBuilder<GdFindAndReplaceAdsHrefText> vb = ModelItemValidationBuilder.of(req);

        vb.item(GdFindAndReplaceAdsHrefText.REPLACE_INSTRUCTION)
                .check(notNull())
                .checkBy(FIND_AND_REPLACE_BANNER_HREF_TEXT_INSTRUCTION_VALIDATOR, When.isValid());

        vb.item(GdFindAndReplaceAdsHrefText.TARGET_TYPES)
                .check(notNull())
                .checkBy(ADS_HREF_TARGET_TYPES_VALIDATOR);

        return vb.getResult();
    };

    private static final Validator<GdFindAndReplaceAdsDisplayHref, Defect>
            FIND_AND_REPLACE_BANNER_DISPLAY_HREF_VALIDATOR = req -> {
        ModelItemValidationBuilder<GdFindAndReplaceAdsDisplayHref> vb = ModelItemValidationBuilder.of(req);

        vb.item(GdFindAndReplaceAdsDisplayHref.AD_IDS).checkBy(AD_IDS_VALIDATOR);
        vb.item(GdFindAndReplaceAdsDisplayHref.NEW_DISPLAY_HREF).checkBy(displayHrefValidator(), When.notNull());
        vb.list(GdFindAndReplaceAdsDisplayHref.SEARCH_DISPLAY_HREFS)
                .checkEachBy(displayHrefValidator(), When.notNull());

        return vb.getResult();
    };

    private static final Validator<GdFindAndReplaceText, Defect>
            FIND_AND_REPLACE_BANNER_TEXT_VALIDATOR = req -> {
        ModelItemValidationBuilder<GdFindAndReplaceText> vb = ModelItemValidationBuilder.of(req);

        vb.item(GdFindAndReplaceText.AD_IDS).checkBy(AD_IDS_VALIDATOR);

        vb.item(GdFindAndReplaceText.REPLACE_INSTRUCTION)
                .check(notNull())
                .checkBy(FIND_AND_REPLACE_BANNER_TEXT_INSTRUCTION_VALIDATOR, When.isValid());

        vb.item(GdFindAndReplaceText.TARGET_TYPES)
                .check(notNull())
                .checkBy(ADS_TEXT_TARGET_TYPES_VALIDATOR);

        return vb.getResult();
    };

    private static final Validator<GdFindAndReplaceCalloutsInstruction, Defect>
            FIND_AND_REPLACE_BANNER_CALLOUTS_INSTRUCTION_VALIDATOR =
            instruction -> {
                ModelItemValidationBuilder<GdFindAndReplaceCalloutsInstruction> vb =
                        ModelItemValidationBuilder.of(instruction);

                vb.list(GdFindAndReplaceCalloutsInstruction.SEARCH)
                        .check(notNull())
                        .checkEach(validId());

                vb.list(GdFindAndReplaceCalloutsInstruction.REPLACE)
                        .check(notNull())
                        .checkEach(validId());

                return vb.getResult();
            };

    private static final Validator<GdFindAndReplaceCallouts, Defect>
            FIND_AND_REPLACE_BANNER_CALLOUTS_VALIDATOR = req -> {
        ModelItemValidationBuilder<GdFindAndReplaceCallouts> vb = ModelItemValidationBuilder.of(req);

        vb.item(GdFindAndReplaceCallouts.AD_IDS).checkBy(AD_IDS_VALIDATOR);

        vb.item(GdFindAndReplaceCallouts.ACTION)
                .check(notNull());

        vb.item(GdFindAndReplaceCallouts.REPLACE_INSTRUCTION)
                .check(notNull())
                .checkBy(FIND_AND_REPLACE_BANNER_CALLOUTS_INSTRUCTION_VALIDATOR, When.isValid());

        return vb.getResult();
    };

    private static final Validator<GdAddAdGroupMinusKeywordsItem, Defect> ADD_AD_GROUP_MINUS_KEYWORDS_ITEM_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdAddAdGroupMinusKeywordsItem> vb = ModelItemValidationBuilder.of(req);

                vb.item(GdAddAdGroupMinusKeywordsItem.AD_GROUP_ID)
                        .check(validId());
                vb.list(GdAddAdGroupMinusKeywordsItem.MINUS_KEYWORDS)
                        .check(notEmptyCollection());

                return vb.getResult();
            };

    private static final Validator<GdAddAdGroupsMinusKeywords, Defect> ADD_AD_GROUPS_MINUS_KEYWORDS_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdAddAdGroupsMinusKeywords> vb = ModelItemValidationBuilder.of(req);

                vb.list(GdAddAdGroupsMinusKeywords.AD_GROUP_IDS)
                        .check(notEmptyCollection())
                        .checkEach(validId());
                vb.list(GdAddAdGroupsMinusKeywords.MINUS_KEYWORDS)
                        .check(notEmptyCollection());

                return vb.getResult();
            };

    private static final Validator<GdReplaceAdGroupsMinusKeywords, Defect> REPLACE_AD_GROUPS_MINUS_KEYWORDS_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdReplaceAdGroupsMinusKeywords> vb = ModelItemValidationBuilder.of(req);

                vb.list(GdReplaceAdGroupsMinusKeywords.AD_GROUP_IDS)
                        .check(notEmptyCollection())
                        .checkEach(validId());
                vb.list(GdReplaceAdGroupsMinusKeywords.MINUS_KEYWORDS)
                        .check(notNull());

                return vb.getResult();
            };

    private static final Validator<GdRemoveAdGroupsMinusKeywords, Defect> REMOVE_AD_GROUPS_MINUS_KEYWORDS_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdRemoveAdGroupsMinusKeywords> vb = ModelItemValidationBuilder.of(req);

                vb.list(GdRemoveAdGroupsMinusKeywords.AD_GROUP_IDS)
                        .check(notEmptyCollection())
                        .checkEach(validId());
                vb.list(GdRemoveAdGroupsMinusKeywords.MINUS_KEYWORDS)
                        .check(notEmptyCollection());

                return vb.getResult();
            };

    private static final Validator<GdAddAdGroupMinusKeywords, Defect> ADD_AD_GROUP_MINUS_KEYWORDS_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdAddAdGroupMinusKeywords> vb = ModelItemValidationBuilder.of(req);

                vb.list(GdAddAdGroupMinusKeywords.ADD_ITEMS)
                        .check(notEmptyCollection())
                        .checkEachBy(ADD_AD_GROUP_MINUS_KEYWORDS_ITEM_VALIDATOR, When.isValid());

                return vb.getResult();
            };

    private static final Validator<GdGetClientMutationId, Defect> GET_CLIENT_MUTATION_ID_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdGetClientMutationId> vb = ModelItemValidationBuilder.of(req);

                vb.item(GdGetClientMutationId.LOGIN)
                        .check(validLogin());

                return vb.getResult();
            };

    private static final Validator<GdGetJsonSettings, Defect> GET_JSON_SETTINGS_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdGetJsonSettings> vb = ModelItemValidationBuilder.of(req);
                vb.item(GdGetJsonSettings.JSON_PATH)
                        .check(notEmptyCollection());
                vb.list(GdGetJsonSettings.JSON_PATH)
                        .checkEach(validJsonPath());

                return vb.getResult();
            };

    private static final Validator<GdSetJsonSettings, Defect> SET_JSON_SETTINGS_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdSetJsonSettings> vb = ModelItemValidationBuilder.of(req);
                vb.item(GdSetJsonSettings.UPDATE_ITEMS)
                        .check(notEmptyCollection());
                vb.list(GdSetJsonSettings.UPDATE_ITEMS)
                        .checkEachBy(getJsonSettingsUnionValidator());

                return vb.getResult();
            };

    private static final Validator<GdSetJsonSettings, Defect> SET_JSON_SETTINGS_CLIENT_ID_NULL_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdSetJsonSettings> vb = ModelItemValidationBuilder.of(req);
                vb.item(GdSetJsonSettings.ID_TYPE)
                        .check(allowedIdType());

                return vb.getResult();
            };

    private static final Validator<GdGetJsonSettings, Defect> GET_JSON_SETTINGS_CLIENT_ID_NULL_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdGetJsonSettings> vb = ModelItemValidationBuilder.of(req);
                vb.item(GdGetJsonSettings.ID_TYPE)
                        .check(allowedIdType());

                return vb.getResult();
            };

    private static final Validator<GdCheckClientMutationState, Defect> CHECK_CLIENT_MUTATION_STATE_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdCheckClientMutationState> vb = ModelItemValidationBuilder.of(req);

                vb.item(GdCheckClientMutationState.LOGIN)
                        .check(validLogin());
                vb.item(GdCheckClientMutationState.MUTATION_ID)
                        .check(validGtidSet());

                return vb.getResult();
            };

    private static final Validator<GdCashbackRewardsDetailsInput, Defect> CASHBACK_REWARDS_DETAILS_INPUT_DEFECT_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdCashbackRewardsDetailsInput> vb = ModelItemValidationBuilder.of(req);
                vb.item(GdCashbackRewardsDetailsInput.PERIOD)
                        .check(inRange(DETALIZATION_MIN_LENGTH, DETALIZATION_MAX_LENGTH));
                return vb.getResult();
            };

    public static <M extends Model> Constraint<M, Defect> checkMutuallyExclusiveFields(
            List<ModelProperty<? super M, ?>> fields) {
        return Constraint.fromPredicate(
                obj -> StreamEx.of(fields)
                        .filter(f -> f.get(obj) != null)
                        .count() == 1,
                mutuallyExclusive());
    }

    public static <M extends Model> Constraint<M, Defect> checkNotMoreThanOneFieldSet(
            List<ModelProperty<? super M, ?>> fields) {
        return Constraint.fromPredicate(
                obj -> StreamEx.of(fields)
                        .filter(f -> f.get(obj) != null)
                        .count() <= 1,
                mutuallyExclusive());
    }

    private static <M extends Model> Constraint<M, Defect> checkMutuallyExclusiveFieldGroups(
            Collection<ModelProperty<? super M, ?>> first,
            Collection<ModelProperty<? super M, ?>> second, Defect defect) {
        return Constraint.fromPredicate(t -> (hasOnlyNullProps(t, first) && hasOnlyNonNullProps(t, second)) ||
                        (hasOnlyNonNullProps(t, first) && hasOnlyNullProps(t, second)),
                defect);
    }

    private static <M extends Model> boolean hasOnlyNullProps(M obj, Collection<ModelProperty<? super M, ?>> props) {
        return props.stream().map(p -> p.get(obj)).noneMatch(Objects::nonNull);
    }

    private static <M extends Model> boolean hasOnlyNonNullProps(M obj, Collection<ModelProperty<? super M, ?>> props) {
        return props.stream().map(p -> p.get(obj)).noneMatch(not(Objects::nonNull));
    }

    private static <M extends Model> boolean hasOneNonNullProps(M obj, Collection<ModelProperty<? super M, ?>> props) {
        return props.stream()
                .map(p -> p.get(obj))
                .anyMatch(Objects::nonNull);
    }

    /**
     * Проверить, что оба переданных поля или одновременно заданы или одновременно не заданы
     */
    private static <M extends Model> Constraint<M, Defect> checkBothFieldsNullOrNotNull(
            ModelProperty<? super M, ?> first, ModelProperty<? super M, ?> second, Defect defect) {
        Predicate<M> pr =
                t -> (first.get(t) == null && second.get(t) == null) || (first.get(t) != null && second.get(t) != null);
        return Constraint.fromPredicate(pr, defect);
    }

    /**
     * Проверить, что из переданных полей хотя-бы одно имеет значение не null
     */
    public static <M extends Model> Constraint<M, Defect> checkFieldsContainNonNullValue(
            Collection<ModelProperty<? super M, ?>> properties) {
        Predicate<M> pr = t -> hasOneNonNullProps(t, properties);
        return Constraint.fromPredicate(pr, mustContainNonNullProps());
    }

    private static Constraint<String, Defect> validGtidSet() {
        Predicate<String> pr = gtidSetAsString -> {
            try {
                GtidSet gtidSet = new GtidSet(gtidSetAsString);
                return !gtidSet.getUUIDSets().isEmpty();
            } catch (Exception e) {
                //do nothing
            }

            return false;
        };

        return Constraint.fromPredicate(pr, invalidValue());
    }

    private static Constraint<GdStatRequirements, Defect> fromFieldIsNotAfterToField() {
        Predicate<GdStatRequirements> predicate = sr -> (sr.getFrom() == null || sr.getTo() == null) ||
                !sr.getTo().isBefore(sr.getFrom());
        return Constraint.fromPredicate(predicate, datesFromNotAfterTo());
    }

    public static Validator<String, Defect> hrefIsValid() {
        return href -> {
            ItemValidationBuilder<String, Defect> vb = ItemValidationBuilder.of(href);
            vb.check(notNull())
                    .check(notBlank())
                    .check(validHref(), When.isValid());
            return vb.getResult();
        };
    }

    /**
     * Проверка допустимых символов в ключе пар-ра ссылки
     */
    static Constraint<String, Defect> bannersHrefParamsKeyCharsAreAllowed() {
        return fromPredicate(ALLOW_BANNERS_HREF_PARAMS_KEY_MATCHER::matchesNoneOf,
                invalidFindAndReplaceBannersHrefParamsText());
    }

    /**
     * Проверка допустимых символов в значении пар-ра ссылки
     */
    static Constraint<String, Defect> bannersHrefParamsValueCharsAreAllowed() {
        return fromPredicate(ALLOW_BANNERS_HREF_PARAMS_VALUE_MATCHER::matchesNoneOf,
                invalidFindAndReplaceBannersHrefParamsText());
    }

    public <T> void applyValidator(Validator<T, Defect> validator, @Nullable T obj, boolean nullable) {
        applyValidator(validator, obj, nullable, emptyPath());
    }

    /**
     * Применить валидатор к объекту
     *
     * @param validator  - валидатор объекта obj
     * @param obj        - валидируемый объект
     * @param nullable   - является ли null допустимым значением obj
     * @param pathPrefix - префикс добавляемый к path в ошибках валидации
     * @param <T>        - тип валидируемого объекта
     */
    public <T> void applyValidator(Validator<T, Defect> validator, @Nullable T obj, boolean nullable, Path pathPrefix) {
        if (obj == null) {
            if (nullable) {
                // Это состояние допустимо, другая валидация бессмысленна
                return;
            } else {
                // Объект не должен быть null, сразу ругаемся
                ValidationResult<T, Defect> failed =
                        ValidationResult.failed(null, CommonDefects.notNull());
                throwGridValidationException(failed, pathPrefix);
            }
        }
        ValidationResult<T, Defect> validationResult = validator.apply(obj);
        throwGridValidationExceptionIfHasErrors(validationResult, pathPrefix);
    }

    public void throwGridValidationExceptionIfHasErrors(ValidationResult<?, Defect> validationResult) {
        throwGridValidationExceptionIfHasErrors(validationResult, emptyPath());
    }

    private void throwGridValidationExceptionIfHasErrors(ValidationResult<?, Defect> validationResult, Path path) {
        if (validationResult.hasAnyErrors()) {
            throwGridValidationException(validationResult, path);
        }
    }

    public void throwGridValidationExceptionIfHasErrors(Result<?> result, Path path) {
        if (hasValidationIssues(result) || !result.isSuccessful()) {
            throwGridValidationException(result.getValidationResult(), path);
        }
    }

    private void throwGridValidationException(ValidationResult<?, Defect> validationResult, Path path) {
        throw new GridValidationException(
                validationResultConversionService.buildGridValidationResult(validationResult, path));
    }

    @Nullable
    public GdValidationResult getValidationResult(Result<?> result, Path path) {
        return toGdValidationResult(result.getValidationResult(), path);
    }

    @Nullable
    public GdValidationResult toGdValidationResult(ValidationResult<?, Defect> validationResult, Path path) {
        if (hasValidationIssues(validationResult)) {
            return validationResultConversionService.buildGridValidationResult(validationResult, path);
        }
        return null;
    }

    public void validateIdsCollection(@Nullable Set<Long> ids) {
        applyValidator(IDS_COLLECTION_VALIDATOR, ids, true);
    }

    public void validateLimitOffset(@Nullable GdLimitOffset limitOffset) {
        applyValidator(LIMIT_OFFSET_VALIDATOR, limitOffset, true);
    }

    @Deprecated
    void validateStatRequirements(@Nullable GdStatRequirements statRequirements) {
        applyValidator(STAT_REQUIREMENTS_VALIDATOR, statRequirements, true);
    }

    @Deprecated
    void validateCampaignFilter(@Nullable GdCampaignFilter filter) {
        applyValidator(CAMPAIGN_FILTER_VALIDATOR, filter, true);
    }

    @Deprecated
    void validateAdGroupFilter(@Nullable GdAdGroupFilter filter) {
        applyValidator(AD_GROUP_FILTER_VALIDATOR, filter, true);
    }

    @Deprecated
    void validateAdFilter(@Nullable GdAdFilter bannerFilter) {
        applyValidator(AD_FILTER_VALIDATOR, bannerFilter, true);
    }

    public void validateDealFilter(@Nullable GdiDealFilter dealFilter) {
        applyValidator(DEAL_FILTER_VALIDATOR, dealFilter, true);
    }

    public void validateClientSearchRequest(GdClientSearchRequest searchRequest) {
        applyValidator(CLIENT_SEARCH_REQUEST_VALIDATOR, searchRequest, false);
    }

    public void validateAddAdGroupMinusKeywordsRequest(GdAddAdGroupMinusKeywords request) {
        applyValidator(ADD_AD_GROUP_MINUS_KEYWORDS_VALIDATOR, request, false);
    }

    public void validateAddAdGroupsMinusKeywordsRequest(GdAddAdGroupsMinusKeywords request) {
        applyValidator(ADD_AD_GROUPS_MINUS_KEYWORDS_VALIDATOR, request, false);
    }

    public void validateReplaceAdGroupsMinusKeywordsRequest(GdReplaceAdGroupsMinusKeywords request) {
        applyValidator(REPLACE_AD_GROUPS_MINUS_KEYWORDS_VALIDATOR, request, false);
    }

    public void validateRemoveAdGroupsMinusKeywordsRequest(GdRemoveAdGroupsMinusKeywords request) {
        applyValidator(REMOVE_AD_GROUPS_MINUS_KEYWORDS_VALIDATOR, request, false);
    }

    public void validateGetClientMutationIdRequest(GdGetClientMutationId input) {
        applyValidator(GET_CLIENT_MUTATION_ID_VALIDATOR, input, false);
    }

    public void validateGetJsonSettings(GdGetJsonSettings input) {
        applyValidator(GET_JSON_SETTINGS_VALIDATOR, input, false);
    }

    public void validateSetJsonSettings(GdSetJsonSettings input) {
        applyValidator(SET_JSON_SETTINGS_VALIDATOR, input, false);
    }

    public void validateSetJsonSettingsClientIdIsNull(GdSetJsonSettings input) {
        applyValidator(SET_JSON_SETTINGS_CLIENT_ID_NULL_VALIDATOR, input, false);
    }

    public void validateGetJsonSettingsClientIdIsNull(GdGetJsonSettings input) {
        applyValidator(GET_JSON_SETTINGS_CLIENT_ID_NULL_VALIDATOR, input, false);
    }

    public void validateCheckClientMutationStateRequest(GdCheckClientMutationState input) {
        applyValidator(CHECK_CLIENT_MUTATION_STATE_VALIDATOR, input, false);
    }

    public void validateGdCampaignsContainer(GdCampaignsContainer input) {
        applyValidator(CAMPAIGNS_CONTAINER_VALIDATOR, input, false);
    }

    public void validateGdPackagiesStrategyContainer(GdPackageStrategiesContainer input) {
        applyValidator(PACKAGE_STRATEGIES_CONTAINER_VALIDATOR, input, false);
    }

    public void validateGdAdGroupsContainer(GdAdGroupsContainer input) {
        applyValidator(AD_GROUPS_CONTAINER_VALIDATOR, input, false);
    }

    public void validateGdAdsContainer(GdAdsContainer input) {
        applyValidator(ADS_CONTAINER_VALIDATOR, input, false);
    }

    public void validateGdSmartFilterContainer(GdSmartFiltersContainer input) {
        applyValidator(SMART_FILTER_CONTAINER_VALIDATOR, input, false);
    }

    public void validateFindAndReplaceAdHrefParamsRequest(
            GdFindAndReplaceAdsHrefParams findAndReplaceBannersHrefParams, Path pathPrefix) {
        applyValidator(FIND_AND_REPLACE_ADS_HREF_PARAMS_VALIDATOR, findAndReplaceBannersHrefParams, false,
                pathPrefix);
    }

    public void validateAdIds(List<Long> adIds, Path pathPrefix) {
        applyValidator(AD_IDS_VALIDATOR, adIds, false, pathPrefix);
    }

    public void validateFindAndReplaceBannerHrefDomainRequest(
            GdFindAndReplaceAdsHrefDomain findAndReplaceBannersHrefDomain, Path pathPrefix) {
        applyValidator(FIND_AND_REPLACE_ADS_HREF_DOMAIN_VALIDATOR, findAndReplaceBannersHrefDomain, false,
                pathPrefix);
    }

    public void validateFindAndReplaceBannerHrefTextRequest(GdFindAndReplaceAdsHrefText request, Path pathPrefix) {
        applyValidator(FIND_AND_REPLACE_BANNER_HREF_TEXT_VALIDATOR, request, false, pathPrefix);
    }

    public void validateFindAndReplaceBannerDisplayHrefRequest(GdFindAndReplaceAdsDisplayHref request,
                                                               Path pathPrefix) {
        applyValidator(FIND_AND_REPLACE_BANNER_DISPLAY_HREF_VALIDATOR, request, false, pathPrefix);
    }

    public void validateFindAndReplaceBannerTextRequest(GdFindAndReplaceText request, Path prefix) {
        applyValidator(FIND_AND_REPLACE_BANNER_TEXT_VALIDATOR, request, false, prefix);
    }

    public void validateFindAndReplaceBannerCalloutsRequest(GdFindAndReplaceCallouts request, Path prefix) {
        applyValidator(FIND_AND_REPLACE_BANNER_CALLOUTS_VALIDATOR, request, false, prefix);
    }

    public void validateClientMeasurerAccount(GdAddClientMeasurerAccount gdAddClientMeasurerAccount) {
        applyValidator(ADD_CLIENT_MEASURER_ACCOUNT_VALIDATOR, gdAddClientMeasurerAccount, false);
    }

    public void validateCashbackRewardsDetailsRequest(GdCashbackRewardsDetailsInput request) {
        applyValidator(CASHBACK_REWARDS_DETAILS_INPUT_DEFECT_VALIDATOR, request, false);
    }

    /**
     * Собирает верхнеуровневый ValidationResult на основе ValidationResult'ов Result'ов MassResult'а
     */
    public static MassResult<Long> getMassResultWithAggregatedValidationResult(MassResult<Long> massResult,
                                                                               List<Long> keywordIds) {
        ValidationResult<List<Long>, Defect> mergedVr = new ValidationResult<>(keywordIds);
        for (int i = 0; i < massResult.getResult().size(); i++) {
            ValidationResult<?, Defect> localVr = massResult.getResult().get(i).getValidationResult();
            if (null != localVr) {
                transferSubNodesWithIssues(localVr, mergedVr.getOrCreateSubValidationResult(index(i),
                        mergedVr.getValue().get(i)));
            }
        }
        return new MassResult<>(massResult.getResult(), mergedVr, massResult.getState());
    }

}
