package ru.yandex.direct.grid.processing.service.validation.presentation;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.EntryStream;

import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.PathHelper;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.PathNodeConverter;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.path;

/**
 * Конвертирует {@link PathNode} в соответствии с заданными правилами подстановки.
 * По умолчанию конвертирует PathNode в пустой PathNode, если отсутствует правила подстановки для пути
 * <p>
 * Пример использования:
 * <pre>PathNodeConverter converter = SkipByDefaultMappingPathNodeConverter.builder()
 *     .replace("priceContext", "ContextBid")
 *     .replace("retargetings", "AudienceTargets")
 *     .build();
 * </pre>
 *
 * @see PathNode
 * @see PathNodeConverterProvider
 */
@ParametersAreNonnullByDefault
public class SkipByDefaultMappingPathNodeConverter implements PathNodeConverter {

    private final ImmutableMap<PathNode.Field, Path> singleItemDict;

    private SkipByDefaultMappingPathNodeConverter(ImmutableMap<PathNode.Field, Path> pathDict) {
        this.singleItemDict = pathDict;
    }

    public static SkipByDefaultMappingPathNodeConverter emptyConverter() {
        return new SkipByDefaultMappingPathNodeConverter(ImmutableMap.of());
    }

    /**
     * Конвертер удаляющий все поля, кроме указанных
     */
    public static SkipByDefaultMappingPathNodeConverter skipAllFieldsConverterExcept(ModelProperty<?, ?>... fields) {
        return builder().keep(fields)
                .build();
    }

    /**
     * Конвертер удаляющий все поля, кроме указанных
     */
    public static SkipByDefaultMappingPathNodeConverter skipAllFieldsConverterExcept(String... fields) {
        return builder().keep(fields)
                .build();
    }

    public static SkipByDefaultMappingPathNodeConverter.Builder builder() {
        return new SkipByDefaultMappingPathNodeConverter.Builder();
    }

    @Override
    public Path convert(PathNode.Field fieldName) {
        if (!singleItemDict.containsKey(fieldName)) {
            return path();
        } else {
            return singleItemDict.get(fieldName);
        }
    }

    @Override
    public Path convert(PathNode.Field fieldName, PathNode.Index sourceIndex) {
        if (!singleItemDict.containsKey(fieldName)) {
            return path();
        } else {
            return path(sourceIndex);
        }
    }

    public static class Builder {
        private final Map<String, List<String>> singleItemDict = new HashMap<>();

        private Builder() {
        }

        /**
         * Преобразование одного элемента исходного пути в один элемент результирующего пути.
         *
         * @param sourceFieldName исходное название
         * @param destFieldName   во что преобразуется
         */
        public SkipByDefaultMappingPathNodeConverter.Builder replace(String sourceFieldName, String destFieldName) {
            checkArgument(!singleItemDict.containsKey(sourceFieldName), "dictionary already contains %s",
                    sourceFieldName);
            singleItemDict.put(sourceFieldName, singletonList(destFieldName));
            return this;
        }

        /**
         * Преобразование одного элемента исходного пути в один элемент результирующего пути.
         *
         * @param sourceField исходное название
         * @param destField   во что преобразуется
         */
        public SkipByDefaultMappingPathNodeConverter.Builder replace(ModelProperty<?, ?> sourceField,
                                                                     ModelProperty<?, ?> destField) {
            return replace(sourceField.name(), destField.name());
        }

        /**
         * Преобразование одного элемента исходного пути в один элемент результирующего пути.
         *
         * @param sourceField исходное название
         * @param destField   во что преобразуется
         */
        public SkipByDefaultMappingPathNodeConverter.Builder replace(ModelProperty<?, ?> sourceField,
                                                                     String destField) {
            return replace(sourceField.name(), destField);
        }

        /**
         * Преобразование одного элемента исходного пути в несколько элементов результирующего пути.
         * <p>
         * Используется, если нужно добавить в путь промежуточные элементы.
         * Например, исходный путь: {@code banner.creativeId},
         * а в результате хочется получить {@code banner.videoResources.creativeId}.
         *
         * @param sourceFieldName исходное название
         * @param destFieldNames  во что преобразуется
         */
        public SkipByDefaultMappingPathNodeConverter.Builder replace(String sourceFieldName,
                                                                     List<String> destFieldNames) {
            checkArgument(!singleItemDict.containsKey(sourceFieldName), "dictionary already contains %s",
                    sourceFieldName);
            singleItemDict.put(sourceFieldName, destFieldNames);
            return this;
        }

        /**
         * Преобразование переданных полей как есть в результирующий путь
         */
        public SkipByDefaultMappingPathNodeConverter.Builder keep(ModelProperty<?, ?>... fields) {
            Arrays.stream(fields)
                    .forEach(field -> this.replace(field, field));
            return this;
        }

        /**
         * Преобразование переданных полей как есть в результирующий путь
         */
        public SkipByDefaultMappingPathNodeConverter.Builder keep(String... fields) {
            Arrays.stream(fields)
                    .forEach(field -> this.replace(field, field));
            return this;
        }

        public SkipByDefaultMappingPathNodeConverter build() {
            Map<PathNode.Field, Path> singleItemDictTmp = EntryStream.of(singleItemDict)
                    .mapKeys(PathHelper::field)
                    .mapValues(names -> mapList(names, PathHelper::field))
                    .mapValues(Path::new)
                    .toMap();
            return new SkipByDefaultMappingPathNodeConverter(ImmutableMap.copyOf(singleItemDictTmp));
        }
    }

}
