package ru.yandex.direct.grid.processing.service.welcome;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.model.AgencyClientRelation;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.AgencyClientRelationService;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.currency.service.CurrencyService;
import ru.yandex.direct.core.entity.geo.model.GeoRegion;
import ru.yandex.direct.core.entity.geo.model.GeoRegionType;
import ru.yandex.direct.core.entity.geo.repository.IGeoRegionRepository;
import ru.yandex.direct.core.entity.geo.service.CurrentGeoService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.BlackboxUserService;
import ru.yandex.direct.core.service.integration.balance.BalanceService;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.model.placement.GdRegionDesc;
import ru.yandex.direct.grid.processing.model.welcome.GdGetWelcomePageDataPayload;
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.Region;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxAddress;
import ru.yandex.misc.email.Email;

import static ru.yandex.direct.core.entity.user.service.validation.UserDefects.userAssociatedWithAnotherClient;
import static ru.yandex.direct.core.entity.user.service.validation.UserDefects.userHasNoAvailableCurrencies;
import static ru.yandex.direct.core.entity.user.service.validation.UserDefects.userNotFound;
import static ru.yandex.direct.regions.Region.BY_REGION_ID;
import static ru.yandex.direct.regions.Region.KAZAKHSTAN_REGION_ID;
import static ru.yandex.direct.regions.Region.RUSSIA_REGION_ID;
import static ru.yandex.direct.regions.Region.TURKEY_REGION_ID;
import static ru.yandex.direct.regions.Region.UZBEKISTAN_REGION_ID;

@Service
@ParametersAreNonnullByDefault
public class WelcomePageDataService {
    private static final Logger logger = LoggerFactory.getLogger(WelcomePageDataService.class);
    // см. GeoTools.pm::MAIN_COUNTRIES
    private static final List<Long> MAIN_COUNTRIES = List.of(RUSSIA_REGION_ID, BY_REGION_ID,
            KAZAKHSTAN_REGION_ID, TURKEY_REGION_ID, UZBEKISTAN_REGION_ID);

    private final RbacService rbacService;
    private final ClientService clientService;
    private final BalanceService balanceService;
    private final CurrencyService currencyService;
    private final ClientGeoService clientGeoService;
    private final CurrentGeoService currentGeoService;
    private final BlackboxUserService blackboxUserService;
    private final AgencyClientRelationService agencyClientRelationService;
    private final GridValidationResultConversionService validationResultConverter;
    private final IGeoRegionRepository geoRegionRepository;

    public WelcomePageDataService(RbacService rbacService,
                                  ClientService clientService,
                                  BalanceService balanceService,
                                  CurrencyService currencyService,
                                  ClientGeoService clientGeoService,
                                  CurrentGeoService currentGeoService,
                                  BlackboxUserService blackboxUserService,
                                  AgencyClientRelationService agencyClientRelationService,
                                  GridValidationResultConversionService validationResultConverter,
                                  IGeoRegionRepository geoRegionRepository) {
        this.rbacService = rbacService;
        this.clientService = clientService;
        this.balanceService = balanceService;
        this.currencyService = currencyService;
        this.clientGeoService = clientGeoService;
        this.currentGeoService = currentGeoService;
        this.blackboxUserService = blackboxUserService;
        this.agencyClientRelationService = agencyClientRelationService;
        this.validationResultConverter = validationResultConverter;
        this.geoRegionRepository = geoRegionRepository;
    }

    /**
     * см DoCmd.pm::cmd_chooseCountryCurrency
     */
    public GdGetWelcomePageDataPayload getWelcomePageData(User user) {
        Long uid = user.getUid();
        if (uid == null || uid == 0L) {
            logger.warn("No authenticated user found");
            return errorPayload(userNotFound());
        }
        ClientId clientId = user.getClientId();
        if (clientId == null) {
            clientId = balanceService.findClientIdByUid(uid).orElse(null);
        }
        boolean isNewClient = user.getRole() == RbacRole.EMPTY;
        if (isNewClient && clientId != null) {
            Long clientChiefUid = rbacService.getChiefByClientIdOptional(clientId).orElse(null);
            if (clientChiefUid != null && !clientChiefUid.equals(uid)) {
                // к нам пришёл новый (для нас) представитель уже существующего клиента
                // т.е. представителя клиенту добавили в обход нас в Балансе
                logger.error("User {} ({}) is representative of existing client {} with chief user {}",
                        uid, user.getLogin(), clientId, clientChiefUid);
                return errorPayload(userAssociatedWithAnotherClient());
            }
        }
        ClientId clientFirstAgencyId = null;
        if (clientId != null) {
            clientFirstAgencyId = agencyClientRelationService.getUnarchivedBindedAgencies(Set.of(clientId))
                    .stream().map(AgencyClientRelation::getAgencyClientId)
                    .max(Comparator.comparingLong(ClientId::asLong))
                    .orElse(null);
            if (clientFirstAgencyId != null) {
                logger.warn("client agency not null, equal to {}", clientFirstAgencyId);
            }
        }

        Multimap<Long, CurrencyCode> balanceCountryCurrencies = null;
        if (isNewClient && clientId != null) {
            balanceCountryCurrencies = getBalanceCountryCurrencies(clientId, clientFirstAgencyId);
            if (balanceCountryCurrencies.isEmpty()) {
                // так может получаться, если на балансовые данные наложили свою логику и ничего не осталось
                logger.error("No country currencies found in Balance for client={} and agencyId={}",
                        clientId, clientFirstAgencyId);
                return errorPayload(userHasNoAvailableCurrencies());
            }
        }

        Map<Long, Set<CurrencyCode>> allowedCountryCurrencies = currencyService.getAllowedCountryCurrencies(
                balanceCountryCurrencies, uid, clientFirstAgencyId);
        boolean countryAlreadySet = balanceCountryCurrencies != null && balanceCountryCurrencies.keys().size() == 1;
        boolean countryChooseEnable = isNewClient && !countryAlreadySet;
        Long clientCountry = suggestClientCountry(balanceCountryCurrencies, clientId, isNewClient);
        List<String> emails = suggestEmails(user);

        return new GdGetWelcomePageDataPayload()
                .withCurrenciesByCountry(allowedCountryCurrencies)
                // выбирать страну и валюту можно только до момента, когда мы создали пользователя в Балансе
                // страну можно "довыбрать" и для созданных (если есть из чего выбирать)
                .withCountryCurrencyChooseEnabled(clientId == null || countryChooseEnable)
                .withClientCountry(clientCountry)
                .withMainCountries(MAIN_COUNTRIES)
                .withCountries(getCountries(balanceCountryCurrencies, clientCountry))
                .withClientCurrency(StreamEx.of(allowedCountryCurrencies.getOrDefault(clientCountry, Set.of()))
                        .findFirst()
                        .orElse(null))
                .withEmails(emails);
    }

    private Multimap<Long, CurrencyCode> getBalanceCountryCurrencies(ClientId clientId,
                                                                     @Nullable ClientId agencyId) {
        Multimap<Long, CurrencyCode> balanceCountryCurrencies =
                balanceService.getClientCountryCurrencies(clientId, agencyId);
        // см. Currencies.pm::check_contry_currency_restrictions
        return Multimaps.filterEntries(balanceCountryCurrencies,
                e -> !e.getKey().equals(BY_REGION_ID) || e.getValue() == CurrencyCode.BYN);
    }

    @Nullable
    private Long suggestClientCountry(@Nullable Multimap<Long, CurrencyCode> balanceCountryCurrencies,
                                      @Nullable ClientId clientId, boolean isNewClient) {
        if (balanceCountryCurrencies != null && balanceCountryCurrencies.keys().size() == 1) {
            return balanceCountryCurrencies.keys().iterator().next();
        }
        if (clientId != null) {
            Client client = clientService.getClient(clientId);
            if (client != null) {
                return client.getCountryRegionId();
            }
        }
        if (isNewClient) {
            return currentGeoService.getCurrentCountryRegionId().orElse(null);
        }
        return null;
    }

    private List<String> suggestEmails(User user) {
        if (StringUtils.isNotEmpty(user.getEmail())) {
            return List.of(user.getEmail());
        }
        try {
            List<BlackboxAddress> blackboxAddresses = blackboxUserService.getAllEmailsByUid(user.getUid());
            return StreamEx.of(blackboxAddresses)
                    .filter(a -> a.isDefault() || !a.isNative())
                    .sortedBy(BlackboxAddress::isNative)
                    .map(BlackboxAddress::getEmail)
                    .map(Email::getEmail)
                    .toList();
        } catch (RuntimeException e) {
            logger.error("Error while getting emails from blackbox for uid={}", user.getUid(), e);
            return List.of();
        }
    }

    /**
     * Если из Баланса пришли страны, то берем их.
     * Иначе берем все страны из Директового гео-дерева (а не из
     * {@link CurrencyService#getAllowedCountryCurrencies(Multimap, Long, ClientId)},
     * поскольку этот метод возвращает данные на основе таблицы ppcdict.country_currencies,
     * в которой нет стран без валют оплаты, например Украины).
     */
    private List<GdRegionDesc> getCountries(@Nullable Multimap<Long, CurrencyCode> balanceCountryCurrencies,
                                            @Nullable Long clientCountry) {
        if (balanceCountryCurrencies != null) {
            return geoRegionRepository.getGeoRegionsByIds(balanceCountryCurrencies.keys())
                    .stream()
                    .map(this::toGdRegionDesc)
                    .collect(Collectors.toList());
        }

        GeoTree geoTree = clientGeoService.getClientTranslocalGeoTree(clientCountry);
        return EntryStream.of(geoTree.getRegions())
                .filterValues(region -> region.getType() == GeoRegionType.COUNTRY.getTypedValue())
                .values()
                .map(this::toGdRegionDesc)
                .toList();
    }

    private GdRegionDesc toGdRegionDesc(Region region) {
        return new GdRegionDesc()
                .withId(region.getId())
                .withNameRu(region.getNameRu())
                .withNameEn(region.getNameEn())
                .withNameTr(region.getNameTr())
                .withNameUa(region.getNameUa());
    }

    private GdRegionDesc toGdRegionDesc(GeoRegion region) {
        return new GdRegionDesc()
                .withId(region.getId())
                .withNameRu(region.getName())
                .withNameEn(region.getEname())
                .withNameTr(region.getTrname())
                .withNameUa(region.getUaname());
    }

    private GdGetWelcomePageDataPayload errorPayload(Defect defect) {
        return new GdGetWelcomePageDataPayload().withValidationResult(
                validationResultConverter.buildGridValidationResult(
                        ValidationResult.failed(null, defect)));
    }
}
