package ru.yandex.direct.grid.processing.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;

import ru.yandex.direct.validation.Predicates;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Хелпер для построения аггрегаций по коллекции однородных объектов
 *
 * @param <S> класс элемента коллекции
 * @param <F> класс контейнера результата
 */
@ParametersAreNonnullByDefault
public class AggregationStatsCalculator<S, F> implements Function<Collection<S>, F> {
    @ParametersAreNonnullByDefault
    private static class AggregationChain<S, F, V, I> {
        private final BiConsumer<F, V> valueConsumer;
        private final Supplier<V> initialValueSupplier;
        private final Function<S, I> checkValueExtractor;
        private final BiFunction<V, I, V> reduceFunction;
        private final Predicate<V> aggregationFinishedTest;

        private AggregationChain(BiConsumer<F, V> valueConsumer, Supplier<V> initialValueSupplier,
                                 Function<S, I> checkValueExtractor, BiFunction<V, I, V> reduceFunction,
                                 @Nullable Predicate<V> aggregationFinishedTest) {
            this.valueConsumer = valueConsumer;
            this.initialValueSupplier = initialValueSupplier;
            this.checkValueExtractor = checkValueExtractor;
            this.reduceFunction = reduceFunction;
            this.aggregationFinishedTest = aggregationFinishedTest;
        }

        private AggregationNode<S, F, V, I> createNode() {
            return new AggregationNode<>(this);
        }
    }

    @ParametersAreNonnullByDefault
    private static class AggregationNode<S, F, V, I> implements Function<S, Boolean> {
        private final AggregationChain<S, F, V, I> aggregationChain;

        private V value;

        private AggregationNode(AggregationChain<S, F, V, I> aggregationChain) {
            this.aggregationChain = aggregationChain;

            value = aggregationChain.initialValueSupplier.get();
        }

        public Boolean apply(S source) {
            I checkValue = aggregationChain.checkValueExtractor.apply(source);
            value = aggregationChain.reduceFunction.apply(value, checkValue);

            // Возвращаем истину, если значение еще может измениться
            return aggregationChain.aggregationFinishedTest == null ||
                    !aggregationChain.aggregationFinishedTest.test(value);
        }

        private void addData(F featureContainer) {
            aggregationChain.valueConsumer.accept(featureContainer, value);
        }
    }

    @ParametersAreNonnullByDefault
    public static class Builder<S, F> {
        private final Supplier<F> resultContainerSupplier;
        private final List<AggregationChain<S, F, ?, ?>> aggregationChains = new ArrayList<>();

        private Builder(Supplier<F> resultContainerSupplier) {
            this.resultContainerSupplier = resultContainerSupplier;
        }

        /**
         * Добавить произвольную аггрегацию
         *
         * @param valueConsumer           сеттер результата в контейнер
         * @param initialValueSupplier    поставщик значения по-умолчанию
         * @param checkValueExtractor     функция, получающая аггрегируемое значение из элемента обрабатываемой
         *                                коллекции
         * @param reduceFunction          функция, объединяющая результат предыдущей аггрегации с новым полученным
         *                                значением
         * @param aggregationFinishedTest предикат, проверяющий текущее значение аггрегации и возвращающий истину,
         *                                если оно не может измениться. Применяется для оптимизации
         */
        public <V, I> Builder<S, F> aggregate(BiConsumer<F, V> valueConsumer, Supplier<V> initialValueSupplier,
                                              Function<S, I> checkValueExtractor, BiFunction<V, I, V> reduceFunction,
                                              @Nullable Predicate<V> aggregationFinishedTest) {
            aggregationChains
                    .add(new AggregationChain<>(valueConsumer, initialValueSupplier, checkValueExtractor,
                            reduceFunction, aggregationFinishedTest));
            return this;
        }

        /**
         * Добавить произвольную аггрегацию
         *
         * @param valueConsumer        сеттер результата в контейнер
         * @param initialValueSupplier поставщик значения по-умолчанию
         * @param checkValueExtractor  функция, получающая аггрегируемое значение из элемента обрабатываемой коллекции
         * @param reduceFunction       функция, объединяющая результат предыдущей аггрегации с новым полученным
         *                             значением
         */
        public <V, I> Builder<S, F> aggregate(BiConsumer<F, V> valueConsumer, Supplier<V> initialValueSupplier,
                                              Function<S, I> checkValueExtractor, BiFunction<V, I, V> reduceFunction) {
            return aggregate(valueConsumer, initialValueSupplier, checkValueExtractor, reduceFunction, null);
        }

        /**
         * Добавить произвольную аггрегацию, значение по-умолчанию у которой null
         *
         * @param valueConsumer       сеттер результата в контейнер
         * @param checkValueExtractor функция, получающая аггрегируемое значение из элемента обрабатываемой коллекции
         * @param reduceFunction      функция, объединяющая результат предыдущей аггрегации с новым полученным значением
         */
        public <V, I> Builder<S, F> aggregate(BiConsumer<F, V> valueConsumer,
                                              Function<S, I> checkValueExtractor, BiFunction<V, I, V> reduceFunction,
                                              Predicate<V> aggregationFinishedTest) {
            return aggregate(valueConsumer, () -> null, checkValueExtractor, reduceFunction, aggregationFinishedTest);
        }

        /**
         * Добавить произвольную аггрегацию, значение по-умолчанию у которой null
         *
         * @param valueConsumer       сеттер результата в контейнер
         * @param checkValueExtractor функция, получающая аггрегируемое значение из элемента обрабатываемой коллекции
         * @param reduceFunction      функция, объединяющая результат предыдущей аггрегации с новым полученным значением
         */
        public <V, I> Builder<S, F> aggregate(BiConsumer<F, V> valueConsumer,
                                              Function<S, I> checkValueExtractor, BiFunction<V, I, V> reduceFunction) {
            return aggregate(valueConsumer, () -> null, checkValueExtractor, reduceFunction);
        }

        /**
         * Добавить аггрегацию, возвращающую набор всех значений поля, полученных из элементов списка
         *
         * @param valueConsumer       сеттер результата в контейнер
         * @param checkValueExtractor функция, получающая аггрегируемое значение из элемента обрабатываемой коллекции
         */
        public <V> Builder<S, F> valuesSet(BiConsumer<F, Set<V>> valueConsumer, Function<S, V> checkValueExtractor) {
            return valuesSet(valueConsumer, checkValueExtractor, Predicates.ignore());
        }

        /**
         * Добавить аггрегацию, возвращающую набор всех значений поля, полученных из элементов списка
         *
         * @param valueConsumer       сеттер результата в контейнер
         * @param checkValueExtractor функция, получающая аггрегируемое значение из элемента обрабатываемой коллекции
         */
        public <V> Builder<S, F> valuesSet(BiConsumer<F, Set<V>> valueConsumer, Function<S, V> checkValueExtractor,
                                           Predicate<V> addPredicate) {
            return aggregate(valueConsumer, HashSet::new, checkValueExtractor, (aggr, v) -> {
                if (addPredicate.test(v)) {
                    aggr.add(v);
                }
                return aggr;
            });
        }

        /**
         * Добавить аггрегацию, возвращающую кол-во значений поля, полученных из элементов списка удовлетворяющих
         * переданному условию
         *
         * @param valueConsumer       сеттер результата в контейнер
         * @param checkValueExtractor функция, получающая аггрегируемое значение из элемента обрабатываемой коллекции
         * @param valuePredicate      предикат, проверяющий полученное значение на соответствие условию
         */
        public <V> Builder<S, F> valuesCount(BiConsumer<F, Integer> valueConsumer,
                                             Function<S, V> checkValueExtractor,
                                             Predicate<V> valuePredicate) {
            return aggregate(valueConsumer, () -> 0, checkValueExtractor,
                    (sum, value) -> sum + (valuePredicate.test(value) ? 1 : 0));
        }

        /**
         * Добавить аггрегацию, возвращающую истину, если хоть одно из полученных значений соответствует переданному
         * условию
         *
         * @param valueConsumer       сеттер результата в контейнер
         * @param checkValueExtractor функция, получающая аггрегируемое значение из элемента обрабатываемой коллекции
         * @param valuePredicate      предикат, проверяющий полученное значение на соответствие условию
         */
        public <V> Builder<S, F> hasValue(BiConsumer<F, Boolean> valueConsumer, Function<S, V> checkValueExtractor,
                                          Predicate<V> valuePredicate) {
            return hasValue(valueConsumer, s -> valuePredicate.test(checkValueExtractor.apply(s)));
        }

        /**
         * Добавить аггрегацию, возвращающую истину, если хоть одно из полученных значений истинно
         *
         * @param valueConsumer       сеттер результата в контейнер
         * @param checkValueExtractor функция, получающая аггрегируемое значение из элемента обрабатываемой коллекции
         */
        public Builder<S, F> hasValue(BiConsumer<F, Boolean> valueConsumer, Function<S, Boolean> checkValueExtractor) {
            return aggregate(valueConsumer, () -> false, checkValueExtractor, (aggr, v) -> aggr || v, v -> v);
        }

        public AggregationStatsCalculator<S, F> build() {
            return new AggregationStatsCalculator<>(resultContainerSupplier, aggregationChains);
        }
    }

    private final Supplier<F> resultContainerSupplier;
    private final List<AggregationChain<S, F, ?, ?>> aggregationChains;

    private AggregationStatsCalculator(Supplier<F> resultContainerSupplier,
                                       List<AggregationChain<S, F, ?, ?>> aggregationChains) {
        this.resultContainerSupplier = resultContainerSupplier;
        this.aggregationChains = ImmutableList.copyOf(aggregationChains);
    }

    /**
     * @param resultContainerSupplier поставщик контейнера, в который складывается информация об аггрегации
     */
    public static <S, F> Builder<S, F> builder(Supplier<F> resultContainerSupplier) {
        return new Builder<>(resultContainerSupplier);
    }

    @Override
    public F apply(Collection<S> source) {
        F featureContainer = resultContainerSupplier.get();

        for (AggregationNode<S, F, ?, ?> node : mapList(aggregationChains, AggregationChain::createNode)) {
            for (S item : source) {
                // Если значение не может измениться, переходим к расчету следующего
                if (!node.apply(item)) {
                    break;
                }
            }
            node.addData(featureContainer);
        }

        mapList(aggregationChains, AggregationChain::createNode).stream()
                .peek(n -> calcAggregation(source, n))
                .forEach(n -> n.addData(featureContainer));

        return featureContainer;
    }

    private boolean calcAggregation(Collection<S> source, AggregationNode<S, F, ?, ?> n) {
        return source.stream()
                // Последовательно передаем элементы на подсчет результата
                .map(n)
                // Пока не поймем, что нам больше не надо это делать
                .filter(s -> false)
                .findFirst()
                .orElse(false);
    }
}
