package ru.yandex.direct.grid.processing.util;

import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import one.util.streamex.EntryStream;

import ru.yandex.direct.feature.FeatureName;

@ParametersAreNonnullByDefault
public class GdEntitySet<T>  {

    private final ImmutableSet<T> globalEntities;
    private final ImmutableMap<FeatureName, T> entitiesByFeature;

    private GdEntitySet(ImmutableSet<T> globalEntities,
                        ImmutableMap<FeatureName, T> entitiesByFeature) {
        this.globalEntities = globalEntities;
        this.entitiesByFeature = entitiesByFeature;
    }

    public static <T> GdEntitySet.Builder<T> newBuilder() {
        return new Builder<T>();
    }

    public ImmutableSet<T> getAll(Set<String> availableFeatures) {
        Set<T> featureEntities = EntryStream.of(entitiesByFeature)
                .mapKeys(FeatureName::getName)
                .filterKeys(availableFeatures::contains)
                .values()
                .toSet();
        return ImmutableSet.<T>builder()
                .addAll(globalEntities)
                .addAll(featureEntities)
                .build();
    }

    public ImmutableSet<T> getGlobal() {
        return globalEntities;
    }

    public ImmutableMap<FeatureName, T> getByFeature() {
        return entitiesByFeature;
    }

    public static class Builder<T> {

        ImmutableSet.Builder<T> globalEntities = ImmutableSet.builder();
        ImmutableMap.Builder<FeatureName, T> entitiesByFeature = ImmutableMap.builder();

        public Builder<T> add(T type) {
            globalEntities.add(type);
            return this;
        }

        public Builder<T> addIf(T type, FeatureName feature) {
            entitiesByFeature.put(feature, type);
            return this;
        }

        public GdEntitySet<T> build() {
            return new GdEntitySet<>(globalEntities.build(), entitiesByFeature.build());
        }

    }

}
