package ru.yandex.direct.grid.processing.util;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.grid.processing.model.common.GdiOperationResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.PathHelper;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.result.ValidationResult.ValidationResultTransformer;

/**
 * Сортирует GdiOperationResult
 */
@ParametersAreNonnullByDefault
public class OperationResultSorter {

    private OperationResultSorter() {
    }

    /**
     * Сортирует список rowset на основе validationResult в порядке: 1. ошибки  2. предупреждения 3. без ошибок
     * т.к. validationResult должен ссылаться на rowset, то нужно изменить и индексы в validationResult.subResults
     */
    public static <T, R> GdiOperationResult<T, R> sortByValidationResult(GdiOperationResult<T, R> sourceOperationResult) {
        // разделить sourceRowset на три части errorList - ошибки валидации, warningList - предупреждения,
        // successList - без ошибок.
        // переложить элементы из sourceValidationResult.subResults в sortedSubResults под новым ключом-индексом,
        // так чтобы subResults ссылался на sortedRowset

        List<T> sourceRowset = sourceOperationResult.getRowset();
        ValidationResult<List<R>, Defect> sourceValidationResult = sourceOperationResult.getValidationResult();

        Map<PathNode, ValidationResult<?, Defect>> sourceSubResults = sourceValidationResult.getSubResults();


        List<T> errorList = new ArrayList<>();
        List<T> warningList = new ArrayList<>();
        List<T> successList = new ArrayList<>();

        List<ValidationResult<?, Defect>> errorValidationResults = new ArrayList<>();
        List<ValidationResult<?, Defect>> warningValidationResults = new ArrayList<>();

        for (int i = 0; i < sourceRowset.size(); i++) {
            T item = sourceRowset.get(i);
            ValidationResult<?, Defect> vr = sourceSubResults.get(PathHelper.index(i));

            if (vr != null && vr.hasAnyErrors()) {
                errorList.add(item);
                errorValidationResults.add(vr);
            } else if (vr != null && vr.hasAnyWarnings()) {
                warningList.add(item);
                warningValidationResults.add(vr);
            } else {
                successList.add(item);
            }
        }

        List<T> sortedRowset = joinLists(errorList, warningList, successList);
        Map<PathNode, ValidationResult<?, Defect>> sortedSubResults =
                joinSubResults(errorValidationResults, warningValidationResults);

        ValidationResult<List<R>, Defect> sortedValidationResult =
                createValidationResultBy(sourceValidationResult, sortedSubResults);

        return new GdiOperationResult<>(sortedRowset, sortedValidationResult);
    }

    /**
     * Объединить списки errorValidationResults и warningValidationResults в мапу subResults под индексами так
     * чтобы первыми шли errorValidationResults, а warningValidationResults шли со свдигом индексов.
     */
    private static Map<PathNode, ValidationResult<?, Defect>> joinSubResults(
            List<ValidationResult<?, Defect>> errorVRs,
            List<ValidationResult<?, Defect>> warningVRs) {
        Map<PathNode, ValidationResult<?, Defect>> sortedSubResults = new HashMap<>();

        for (int i = 0; i < errorVRs.size(); i++) {
            sortedSubResults.put(PathHelper.index(i), errorVRs.get(i));
        }

        int indexShift = errorVRs.size();
        for (int i = 0; i < warningVRs.size(); i++) {
            sortedSubResults.put(PathHelper.index(indexShift + i), warningVRs.get(i));
        }
        return sortedSubResults;
    }

    /**
     * Объединить списки в перечисленном порядке
     */
    private static <T> List<T> joinLists(List<T> first, List<T> second, List<T> third) {
        List<T> sortedRowset = new ArrayList<>();
        sortedRowset.addAll(first);
        sortedRowset.addAll(second);
        sortedRowset.addAll(third);
        return sortedRowset;
    }

    /**
     * Создать новый ValidationResult на основе sourceValidationResult с замененным newSubResults.
     * Порядок в value не меняется.
     *
     * @param sourceValidationResult - исходный объект валидации
     * @param newSubResults          - новый subResults, который будет заменен в sourceValidationResult
     * @return - новый ValidationResult
     */
    private static <R> ValidationResult<List<R>, Defect> createValidationResultBy(
            ValidationResult<List<R>, Defect> sourceValidationResult,
            Map<PathNode, ValidationResult<?, Defect>> newSubResults) {
        ValidationResultTransformer<Defect> replaceSubResultTransformer =
                new ValidationResultTransformer<Defect>() {
                    @ParametersAreNonnullByDefault
                    @Override
                    public Map<PathNode, ValidationResult<?, Defect>> transformSubResults(Path path, Map subResults) {
                        return newSubResults;
                    }
                };

        return sourceValidationResult.transformUnchecked(replaceSubResultTransformer);
    }
}
