package ru.yandex.direct.grid.processing.util;

import java.util.Collections;
import java.util.List;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;

@ParametersAreNonnullByDefault
public class ResponseConverter {

    /**
     * Преборазует результаты массовой операции по правилу:
     * удаляются все неуспешные результаты, к ним применяется mapper
     * Возвращает пустой список, если {@code massResult.getResult() == null}
     *
     * @param massResult - результат операции
     * @param mapper     - функция преобразования для успешных результатов
     * @param <T>        - тип результата операции
     * @param <R>        - тип преобразованных элементов
     * @return список преобразованных элементов из massResult
     */
    public static <T, R> List<R> getSuccessfullyResults(MassResult<T> massResult, Function<T, R> mapper) {
        if (massResult.getResult() == null) {
            return Collections.emptyList();
        }

        return StreamEx.of(massResult.getResult())
                .nonNull()
                .filter(Result::isSuccessful)
                .map(Result::getResult)
                .map(mapper)
                .toList();
    }

    /**
     * Преборазует результаты массовой операции по правилу:
     * к успешным результатам применяется mapper, для остальных в результирующем списке будет null
     * Возвращает пустой список, если {@code massResult.getResult() == null}
     *
     * @param massResult - результат операции
     * @param mapper     - функция преобразования для успешных результатов
     * @param <T>        - тип результата операции
     * @param <R>        - тип преобразованных элементов
     * @return список преобразованных элементов из massResult
     */
    public static <T, R> List<R> getResults(MassResult<T> massResult, Function<T, R> mapper) {
        if (massResult.getResult() == null) {
            return Collections.emptyList();
        }

        return StreamEx.of(massResult.getResult())
                .nonNull()
                .map(result -> result.isSuccessful() && result.getResult() != null
                        ? mapper.apply(result.getResult())
                        : null)
                .toList();
    }

}
