package ru.yandex.direct.grid.processing.util;

import java.util.Collections;
import java.util.List;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectInfo;

import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapAndFilterList;

@ParametersAreNonnullByDefault
public class ResultConverterHelper {

    public static List<Long> getSuccessfullyUpdatedIds(MassResult<? extends ModelWithId> result) {
        return getSuccessfullyUpdatedIds(result, ModelWithId::getId);
    }

    public static <T> List<Long> getSuccessfullyUpdatedIds(MassResult<T> result, Function<T, Long> mapperToId) {
        if (result.getResult() == null) {
            return Collections.emptyList();
        }

        return StreamEx.of(result.getResult())
                .filter(Result::isSuccessful)
                .map(Result::getResult)
                .map(mapperToId)
                .toList();
    }

    public static <T> int getCountOfTrueBooleanValues(Iterable<? extends T> source,
                                                      Function<? super T, Boolean> mapper) {
        return mapAndFilterList(source, mapper, Boolean::booleanValue).size();
    }

    public static int getCountOfTrueBooleanValues(List<Boolean> booleans) {
        return filterList(booleans, Boolean::booleanValue).size();
    }

    public static List<Long> getUpdatedIdsByDefectInWarns(MassResult<? extends ModelWithId> massResult, Defect defect) {
        if (massResult.getResult() == null) {
            return Collections.emptyList();
        }

        return StreamEx.of(massResult.getResult())
                .filter(r -> isResultContainsDefectInWarnings(r, defect))
                .map(Result::getResult)
                .map(ModelWithId::getId)
                .toList();
    }

    private static boolean isResultContainsDefectInWarnings(Result<? extends ModelWithId> result, Defect defect) {
        return StreamEx.of(result.getWarnings())
                .map(DefectInfo::getDefect)
                .has(defect);
    }

}
