package ru.yandex.direct.grid.processing.util;

import java.math.BigDecimal;
import java.time.Instant;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.direct.grid.core.entity.model.GdiEntityStats;
import ru.yandex.direct.grid.core.entity.model.GdiEntityStatsFilter;
import ru.yandex.direct.grid.core.entity.model.GdiGoalStats;
import ru.yandex.direct.grid.core.entity.model.GdiGoalStatsFilter;
import ru.yandex.direct.grid.core.entity.model.GdiOfferStats;
import ru.yandex.direct.grid.core.entity.model.GdiOfferStatsFilter;
import ru.yandex.direct.grid.core.util.filters.FilterProcessor;
import ru.yandex.direct.grid.core.util.stats.GridStatNew;
import ru.yandex.direct.grid.model.GdEntityStats;
import ru.yandex.direct.grid.model.GdEntityStatsFilter;
import ru.yandex.direct.grid.model.GdGoalStats;
import ru.yandex.direct.grid.model.GdGoalStatsFilter;
import ru.yandex.direct.grid.model.GdOfferStats;
import ru.yandex.direct.grid.model.GdOfferStatsFilter;
import ru.yandex.direct.grid.model.GdStatPreset;
import ru.yandex.direct.grid.model.GdStatRequirements;
import ru.yandex.direct.grid.model.campaign.GdCampaign;
import ru.yandex.direct.grid.model.campaign.GdCampaignType;
import ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter;
import ru.yandex.direct.grid.model.entity.recommendation.GdiRecommendationType;
import ru.yandex.direct.grid.processing.service.campaign.CampaignServiceUtils;

import static java.math.BigDecimal.ZERO;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.groupingBy;
import static ru.yandex.direct.grid.core.util.filters.FilterProvider.greaterOrEqual;
import static ru.yandex.direct.grid.core.util.filters.FilterProvider.lessOrEqual;
import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toCampaignType;
import static ru.yandex.direct.grid.processing.util.StatCalculationHelper.mapToBigDecimalAndSum;
import static ru.yandex.direct.grid.processing.util.StatCalculationHelper.mapToLongAndSum;
import static ru.yandex.direct.grid.processing.util.StatCalculationHelper.percent;
import static ru.yandex.direct.grid.processing.util.StatCalculationHelper.ratio;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.DateTimeUtils.MSK;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.NumberUtils.isZero;

@ParametersAreNonnullByDefault
public class StatHelper {

    public static final GdGoalStats EMPTY_GOAL_STAT =
            new GdGoalStats().withGoals(0L).withConversionRate(ZERO).withCostPerAction(ZERO);

    private static final GdStatPreset DEFAULT_PRESET = GdStatPreset.TODAY;

    private static final FilterProcessor.Builder<GdEntityStatsFilter, GdEntityStats> STATS_FILTER_PROCESSOR_BUILDER =
            new FilterProcessor.Builder<GdEntityStatsFilter, GdEntityStats>()
                    .withFilter(GdEntityStatsFilter::getMinCost, lessOrEqual(GdEntityStats::getCost))
                    .withFilter(GdEntityStatsFilter::getMaxCost, greaterOrEqual(GdEntityStats::getCost))
                    .withFilter(GdEntityStatsFilter::getMinCostWithTax, lessOrEqual(GdEntityStats::getCostWithTax))
                    .withFilter(GdEntityStatsFilter::getMaxCostWithTax, greaterOrEqual(GdEntityStats::getCostWithTax))
                    .withFilter(GdEntityStatsFilter::getMinShows, lessOrEqual(GdEntityStats::getShows))
                    .withFilter(GdEntityStatsFilter::getMaxShows, greaterOrEqual(GdEntityStats::getShows))
                    .withFilter(GdEntityStatsFilter::getMinClicks, lessOrEqual(GdEntityStats::getClicks))
                    .withFilter(GdEntityStatsFilter::getMaxClicks, greaterOrEqual(GdEntityStats::getClicks))
                    .withFilter(GdEntityStatsFilter::getMinCtr, lessOrEqual(GdEntityStats::getCtr))
                    .withFilter(GdEntityStatsFilter::getMaxCtr, greaterOrEqual(GdEntityStats::getCtr))
                    .withFilter(GdEntityStatsFilter::getMinAvgClickCost,
                            lessOrEqual(GdEntityStats::getAvgClickCost))
                    .withFilter(GdEntityStatsFilter::getMaxAvgClickCost,
                            greaterOrEqual(GdEntityStats::getAvgClickCost))
                    .withFilter(GdEntityStatsFilter::getMinAvgShowPosition,
                            lessOrEqual(GdEntityStats::getAvgShowPosition))
                    .withFilter(GdEntityStatsFilter::getMaxAvgShowPosition,
                            greaterOrEqual(GdEntityStats::getAvgShowPosition))
                    .withFilter(GdEntityStatsFilter::getMinAvgClickPosition,
                            lessOrEqual(GdEntityStats::getAvgClickPosition))
                    .withFilter(GdEntityStatsFilter::getMaxAvgClickPosition,
                            greaterOrEqual(GdEntityStats::getAvgClickPosition))
                    .withFilter(GdEntityStatsFilter::getMinBounceRate,
                            lessOrEqual(GdEntityStats::getBounceRate))
                    .withFilter(GdEntityStatsFilter::getMaxBounceRate,
                            greaterOrEqual(GdEntityStats::getBounceRate))
                    .withFilter(GdEntityStatsFilter::getMinConversionRate,
                            lessOrEqual(GdEntityStats::getConversionRate))
                    .withFilter(GdEntityStatsFilter::getMaxConversionRate,
                            greaterOrEqual(GdEntityStats::getConversionRate))
                    .withFilter(GdEntityStatsFilter::getMinAvgGoalCost,
                            lessOrEqual(GdEntityStats::getAvgGoalCost))
                    .withFilter(GdEntityStatsFilter::getMaxAvgGoalCost,
                            greaterOrEqual(GdEntityStats::getAvgGoalCost))
                    .withFilter(GdEntityStatsFilter::getMinGoals, lessOrEqual(GdEntityStats::getGoals))
                    .withFilter(GdEntityStatsFilter::getMaxGoals, greaterOrEqual(GdEntityStats::getGoals))
                    .withFilter(GdEntityStatsFilter::getMinCpmPrice, lessOrEqual(GdEntityStats::getCpmPrice))
                    .withFilter(GdEntityStatsFilter::getMaxCpmPrice, greaterOrEqual(GdEntityStats::getCpmPrice))
                    .withFilter(GdEntityStatsFilter::getMinAvgDepth, lessOrEqual(GdEntityStats::getAvgDepth))
                    .withFilter(GdEntityStatsFilter::getMaxAvgDepth,
                            greaterOrEqual(GdEntityStats::getAvgDepth))
                    .withFilter(GdEntityStatsFilter::getMinProfitability,
                            lessOrEqual(GdEntityStats::getProfitability))
                    .withFilter(GdEntityStatsFilter::getMaxProfitability,
                            greaterOrEqual(GdEntityStats::getProfitability))
                    .withFilter(GdEntityStatsFilter::getMinCrr,
                            lessOrEqual(GdEntityStats::getCrr))
                    .withFilter(GdEntityStatsFilter::getMaxCrr,
                            greaterOrEqual(GdEntityStats::getCrr))
                    .withFilter(GdEntityStatsFilter::getMinRevenue, lessOrEqual(GdEntityStats::getRevenue))
                    .withFilter(GdEntityStatsFilter::getMaxRevenue, greaterOrEqual(GdEntityStats::getRevenue));

    private static final FilterProcessor<GdEntityStatsFilter, GdEntityStats> STATS_FILTER_PROCESSOR =
            STATS_FILTER_PROCESSOR_BUILDER.build();

    /**
     * Обработчик фильтров по статистике кампании
     */
    public static final FilterProcessor<GdEntityStatsFilter, GdCampaign> CAMPAIGN_STATS_FILTER_PROCESSOR =
            STATS_FILTER_PROCESSOR_BUILDER.build(GdCampaign::getStats);

    /**
     * Обработчик фильтров по статистике целей
     */
    private static final FilterProcessor<GdGoalStatsFilter, GdGoalStats> GOAL_STAT_FILTER_PROCESSOR =
            new FilterProcessor.Builder<GdGoalStatsFilter, GdGoalStats>()
                    .withFilter(GdGoalStatsFilter::getMinGoals, lessOrEqual(GdGoalStats::getGoals))
                    .withFilter(GdGoalStatsFilter::getMaxGoals, greaterOrEqual(GdGoalStats::getGoals))
                    .withFilter(GdGoalStatsFilter::getMinConversionRate, lessOrEqual(GdGoalStats::getConversionRate))
                    .withFilter(GdGoalStatsFilter::getMaxConversionRate, greaterOrEqual(GdGoalStats::getConversionRate))
                    .withFilter(GdGoalStatsFilter::getMinCostPerAction, lessOrEqual(GdGoalStats::getCostPerAction))
                    .withFilter(GdGoalStatsFilter::getMaxCostPerAction, greaterOrEqual(GdGoalStats::getCostPerAction))
                    .build();

    private StatHelper() {
    }

    public static GdEntityStats internalStatsToOuter(GdiEntityStats stats,
                                                     GdCampaignType campaignType) {
        return internalStatsToOuter(stats, campaignType, false);
    }

    /**
     * Конвертировать статистику во внутреннем представлении в статистику во внешнем представлении. Преполагает, что
     * поля внутренней статистики не могут быть null
     * <p>
     * Поля cost, avgClickCost обрабатываются специальным образом в зависимости от типа кампании
     */
    public static GdEntityStats internalStatsToOuter(GdiEntityStats stats,
                                                     GdCampaignType campaignType,
                                                     boolean cpcAndCpmOnOneGridEnabled) {
        GdEntityStats result = convertInternalStatsToOuter(stats);

        if (cpcAndCpmOnOneGridEnabled) {
            if (CampaignServiceUtils.CPC_CAMPAIGN_TYPES.contains(toCampaignType(campaignType))) {
                result.withCpmPrice(null);
            } else if (CampaignServiceUtils.CPM_CAMPAIGN_TYPES.contains(toCampaignType(campaignType))) {
                result.withAvgClickCost(null);
            }
        }

        if (campaignType == GdCampaignType.INTERNAL_DISTRIB || campaignType == GdCampaignType.INTERNAL_FREE) {
            if (isZero(result.getCost())) {
                result.setCost(null);
            }

            if (isZero(result.getCostWithTax())) {
                result.setCostWithTax(null);
            }

            if (isZero(result.getAvgClickCost())) {
                result.setAvgClickCost(null);
            }
        }

        return result;
    }

    /**
     * Конвертировать сводку статистики для внешнего представления.
     * Не обрабатывает специальным образом поля cost, avgClickCost (в сводке их значения не важны)
     */
    private static GdEntityStats internalSummaryStatsToOuter(GdiEntityStats stats) {
        return convertInternalStatsToOuter(stats);
    }

    public static GdEntityStats convertInternalStatsToOuter(GdiEntityStats stats) {
        return new GdEntityStats()
                .withClicks(stats.getClicks().longValue())
                .withShows(stats.getShows().longValue())
                .withCost(stats.getCost())
                .withCostWithTax(stats.getCostWithTax())
                .withCpmPrice(stats.getCpmPrice())
                .withRevenue(stats.getRevenue())
                .withGoals(stats.getGoals().longValue())
                .withCtr(stats.getCtr())
                .withBounceRate(stats.getBounceRate())
                .withAvgClickCost(stats.getAvgClickCost())
                .withAvgShowPosition(stats.getAvgShowPosition())
                .withAvgClickPosition(stats.getAvgClickPosition())
                .withAvgGoalCost(stats.getAvgGoalCost())
                .withAvgDepth(stats.getAvgDepth())
                .withCrr(stats.getCrr())
                .withConversionRate(stats.getConversionRate())
                .withProfitability(stats.getProfitability())
                .withDay(stats.getDay());
    }

    public static GdGoalStats internalGoalStatToOuter(GdiGoalStats goalStats) {
        return new GdGoalStats()
                .withGoalId(goalStats.getGoalId())
                .withGoals(goalStats.getGoals())
                .withConversionRate(goalStats.getConversionRate())
                .withCostPerAction(goalStats.getCostPerAction());
    }

    public static GdOfferStats internalOfferStatsToOuter(GdiOfferStats stats) {
        return new GdOfferStats()
                .withShows(stats.getShows().longValue())
                .withClicks(stats.getClicks().longValue())
                .withCtr(stats.getCtr())
                .withCost(stats.getCost())
                .withCostWithTax(stats.getCostWithTax())
                .withRevenue(stats.getRevenue())
                .withCrr(stats.getCrr())
                .withCarts(ifNotNull(stats.getCarts(), BigDecimal::longValue))
                .withPurchases(ifNotNull(stats.getPurchases(), BigDecimal::longValue))
                .withAvgClickCost(stats.getAvgClickCost())
                .withAvgProductPrice(stats.getAvgProductPrice())
                .withAvgPurchaseRevenue(stats.getAvgPurchaseRevenue())
                .withAutobudgetGoals(stats.getAutobudgetGoals().longValue())
                .withMeaningfulGoals(stats.getMeaningfulGoals().longValue())
                .withMetrikaStatUrlByCounterId(stats.getMetrikaStatUrlByCounterId());
    }

    @Nullable
    public static GdiEntityStatsFilter toInternalStatsFilter(@Nullable GdEntityStatsFilter filter) {
        if (filter == null) {
            return null;
        }

        return new GdiEntityStatsFilter()
                .withMinCost(filter.getMinCost())
                .withMaxCost(filter.getMaxCost())
                .withMinCostWithTax(filter.getMinCostWithTax())
                .withMaxCostWithTax(filter.getMaxCostWithTax())
                .withMinShows(filter.getMinShows())
                .withMaxShows(filter.getMaxShows())
                .withMinClicks(filter.getMinClicks())
                .withMaxClicks(filter.getMaxClicks())
                .withMinCtr(filter.getMinCtr())
                .withMaxCtr(filter.getMaxCtr())
                .withMinAvgClickCost(filter.getMinAvgClickCost())
                .withMaxAvgClickCost(filter.getMaxAvgClickCost())
                .withMinAvgShowPosition(filter.getMinAvgShowPosition())
                .withMaxAvgShowPosition(filter.getMaxAvgShowPosition())
                .withMinAvgClickPosition(filter.getMinAvgClickPosition())
                .withMaxAvgClickPosition(filter.getMaxAvgClickPosition())
                .withMinBounceRate(filter.getMinBounceRate())
                .withMaxBounceRate(filter.getMaxBounceRate())
                .withMinConversionRate(filter.getMinConversionRate())
                .withMaxConversionRate(filter.getMaxConversionRate())
                .withMinAvgGoalCost(filter.getMinAvgGoalCost())
                .withMaxAvgGoalCost(filter.getMaxAvgGoalCost())
                .withMinGoals(filter.getMinGoals())
                .withMaxGoals(filter.getMaxGoals())
                .withMinCpmPrice(filter.getMinCpmPrice())
                .withMaxCpmPrice(filter.getMaxCpmPrice())
                .withMinAvgDepth(filter.getMinAvgDepth())
                .withMaxAvgDepth(filter.getMaxAvgDepth())
                .withMinProfitability(filter.getMinProfitability())
                .withMaxProfitability(filter.getMaxProfitability())
                .withMinCrr(filter.getMinCrr())
                .withMaxCrr(filter.getMaxCrr())
                .withMinRevenue(filter.getMinRevenue())
                .withMaxRevenue(filter.getMaxRevenue());
    }

    @Nullable
    public static GdiGoalStatsFilter toInternalGoalStatsFilter(@Nullable GdGoalStatsFilter filter) {
        if (filter == null) {
            return null;
        }

        return new GdiGoalStatsFilter()
                .withGoalId(filter.getGoalId())
                .withMinGoals(filter.getMinGoals())
                .withMaxGoals(filter.getMaxGoals())
                .withMinConversionRate(filter.getMinConversionRate())
                .withMaxConversionRate(filter.getMaxConversionRate())
                .withMinCostPerAction(filter.getMinCostPerAction())
                .withMaxCostPerAction(filter.getMaxCostPerAction());
    }

    @Nullable
    public static GdiOfferStatsFilter toInternalOfferStatsFilter(@Nullable GdOfferStatsFilter filter) {
        if (filter == null) {
            return null;
        }

        return new GdiOfferStatsFilter()
                .withMinShows(ifNotNull(filter.getMinShows(), BigDecimal::valueOf))
                .withMaxShows(ifNotNull(filter.getMaxShows(), BigDecimal::valueOf))
                .withMinClicks(ifNotNull(filter.getMinClicks(), BigDecimal::valueOf))
                .withMaxClicks(ifNotNull(filter.getMaxClicks(), BigDecimal::valueOf))
                .withMinCtr(filter.getMinCtr())
                .withMaxCtr(filter.getMaxCtr())
                .withMinCost(filter.getMinCost())
                .withMaxCost(filter.getMaxCost())
                .withMinCostWithTax(filter.getMinCostWithTax())
                .withMaxCostWithTax(filter.getMaxCostWithTax())
                .withMinRevenue(filter.getMinRevenue())
                .withMaxRevenue(filter.getMaxRevenue())
                .withMinCrr(filter.getMinCrr())
                .withMaxCrr(filter.getMaxCrr())
                .withMinCarts(ifNotNull(filter.getMinCarts(), BigDecimal::valueOf))
                .withMaxCarts(ifNotNull(filter.getMaxCarts(), BigDecimal::valueOf))
                .withMinPurchases(ifNotNull(filter.getMinPurchases(), BigDecimal::valueOf))
                .withMaxPurchases(ifNotNull(filter.getMaxPurchases(), BigDecimal::valueOf))
                .withMinAvgClickCost(filter.getMinAvgClickCost())
                .withMaxAvgClickCost(filter.getMaxAvgClickCost())
                .withMinAvgProductPrice(filter.getMinAvgProductPrice())
                .withMaxAvgProductPrice(filter.getMaxAvgProductPrice())
                .withMinAvgPurchaseRevenue(filter.getMinAvgProductPrice())
                .withMaxAvgPurchaseRevenue(filter.getMaxAvgPurchaseRevenue())
                .withMinAutobudgetGoals(ifNotNull(filter.getMinAutobudgetGoals(), BigDecimal::valueOf))
                .withMaxAutobudgetGoals(ifNotNull(filter.getMaxAutobudgetGoals(), BigDecimal::valueOf))
                .withMinMeaningfulGoals(ifNotNull(filter.getMinMeaningfulGoals(), BigDecimal::valueOf))
                .withMaxMeaningfulGoals(ifNotNull(filter.getMaxMeaningfulGoals(), BigDecimal::valueOf));
    }

    /**
     * Посчитать суммарные значения (строка ИТОГО) для статистики.
     */
    public static GdEntityStats calcTotalStats(Collection<GdEntityStats> stats) {
        List<GdEntityStats> statsWithShows =
                filterList(stats, gdEntityStats -> (gdEntityStats != null) && (gdEntityStats.getShows() > 0));
        GdEntityStats total = internalSummaryStatsToOuter(GridStatNew.addZeros(new GdiEntityStats()))
                .withClicks(mapToLongAndSum(statsWithShows, GdEntityStats::getClicks))
                .withShows(mapToLongAndSum(statsWithShows, GdEntityStats::getShows))
                .withGoals(mapToLongAndSum(statsWithShows, GdEntityStats::getGoals))
                .withCost(mapToBigDecimalAndSum(statsWithShows, GdEntityStats::getCost))
                .withCostWithTax(mapToBigDecimalAndSum(statsWithShows, GdEntityStats::getCostWithTax))
                .withRevenue(mapToBigDecimalAndSum(statsWithShows, GdEntityStats::getRevenue));
        total.withCtr(percent(BigDecimal.valueOf(total.getClicks()), BigDecimal.valueOf(total.getShows())));
        total.withAvgClickCost(calcAvgClickCost(total.getCost(), total.getClicks()));
        total.withAvgGoalCost(ratio(total.getCost(), BigDecimal.valueOf(total.getGoals())));
        total.withBounceRate(ratio(calcTotalBounce(statsWithShows), BigDecimal.valueOf(total.getClicks())));
        total.withConversionRate(percent(BigDecimal.valueOf(total.getGoals()), BigDecimal.valueOf(total.getClicks())));
        total.withProfitability(ratio(total.getRevenue().subtract(total.getCost()), total.getCost()));
        total.withCrr(percent(total.getCost(), total.getRevenue()));
        return total;
    }

    public static void recalcTotalStatsForUnitedGrid(GdEntityStats totalStats,
                                                     Map<GdCampaignType, List<GdEntityStats>> campaignTypeToStats) {
        totalStats.setAvgClickCost(calcAvgClickCost(campaignTypeToStats));
        // Итоговый cpmPrice не отдается на фронт (всегда 0)
    }

    /**
     * Считает AvgClickCost в зависимости от типа кампании
     * AvgClickCost = 0 для CPM кампаний
     */
    private static BigDecimal calcAvgClickCost(Map<GdCampaignType, List<GdEntityStats>> campaignTypeToStats) {
        List<GdEntityStats> statsWithShows = EntryStream.of(campaignTypeToStats)
                .mapKeys(CampaignDataConverter::toCampaignType)
                .removeKeys(CampaignServiceUtils.CPM_CAMPAIGN_TYPES::contains)
                .values()
                .flatMap(StreamEx::of)
                .filter(gdEntityStats -> gdEntityStats != null && gdEntityStats.getShows() > 0)
                .toList();
        BigDecimal cost = mapToBigDecimalAndSum(statsWithShows, GdEntityStats::getCost);
        Long clicks = mapToLongAndSum(statsWithShows, GdEntityStats::getClicks);
        return calcAvgClickCost(cost, clicks);
    }

    private static BigDecimal calcAvgClickCost(BigDecimal cost, Long clicks) {
        return ratio(cost, BigDecimal.valueOf(clicks));
    }

    public static List<GdGoalStats> calcTotalGoalStats(
            GdEntityStats totalStats, @Nullable List<List<GdGoalStats>> goalStatsList) {
        if (goalStatsList == null) {
            return emptyList();
        }

        Map<Long, List<GdGoalStats>> groupedGoalStats =
                goalStatsList.stream()
                        .filter(Objects::nonNull)
                        .flatMap(List::stream)
                        .filter(Objects::nonNull)
                        .collect(groupingBy(GdGoalStats::getGoalId));

        var result = new ArrayList<GdGoalStats>();
        for (var entity : groupedGoalStats.entrySet()) {
            var goals = entity.getValue().stream().mapToLong(GdGoalStats::getGoals).sum();
            var stats = new GdGoalStats()
                    .withGoalId(entity.getKey())
                    .withGoals(goals)
                    .withCostPerAction(nvl(ratio(totalStats.getCost(), BigDecimal.valueOf(goals)), ZERO))
                    .withConversionRate(nvl(percent(BigDecimal.valueOf(goals),
                            BigDecimal.valueOf(totalStats.getClicks())), ZERO));
            result.add(stats);
        }

        return result;
    }

    private static BigDecimal calcTotalBounce(List<GdEntityStats> statsWithShows) {
        return StreamEx.of(statsWithShows)
                .filter(s -> s.getClicks() != null && s.getBounceRate() != null)
                .map(s -> s.getBounceRate().multiply(BigDecimal.valueOf(s.getClicks())))
                .foldLeft(ZERO, BigDecimal::add);
    }

    public static GdStatRequirements normalizeStatRequirements(
            @Nullable GdStatRequirements srcRequirements,
            Instant currentInstant
    ) {
        if (srcRequirements == null) {
            var defaultRequirements = new GdStatRequirements()
                    .withPreset(DEFAULT_PRESET);
            return normalizeStatRequirements(defaultRequirements, currentInstant, null);
        }
        if (srcRequirements.getFrom() == null && srcRequirements.getPreset() == null) {
            srcRequirements.withPreset(DEFAULT_PRESET);
        }
        return normalizeStatRequirements(srcRequirements, currentInstant, null);
    }

    public static GdStatRequirements normalizeStatRequirements(
            GdStatRequirements statRequirements,
            Instant currentInstant,
            @Nullable Set<GdiRecommendationType> recommendationFilter
    ) {
        GdStatPreset preset = statRequirements.getPreset();
        GdStatPreset statsByDaysPreset = statRequirements.getStatsByDaysPreset();
        if (preset == null && statsByDaysPreset == null) {
            return statRequirements;
        }

        GdStatRequirements result = new GdStatRequirements()
                .withFrom(statRequirements.getFrom())
                .withTo(statRequirements.getTo())
                .withStatsByDaysFrom(statRequirements.getStatsByDaysFrom())
                .withStatsByDaysTo(statRequirements.getStatsByDaysTo())
                .withGoalIds(statRequirements.getGoalIds())
                .withUseCampaignGoalIds(statRequirements.getUseCampaignGoalIds())
                .withIsFlat(statRequirements.getIsFlat());
        if (preset != null) {
            Pair<LocalDate, LocalDate> dayPeriod = getDayPeriod(preset, currentInstant, recommendationFilter);
            result
                    .withFrom(dayPeriod.getLeft())
                    .withTo(dayPeriod.getRight());
        }
        if (statsByDaysPreset != null) {
            Pair<LocalDate, LocalDate> dayPeriod =
                    getDayPeriod(statsByDaysPreset, currentInstant, recommendationFilter);
            result
                    .withStatsByDaysFrom(dayPeriod.getLeft())
                    .withStatsByDaysTo(dayPeriod.getRight());
        }
        return result;
    }

    public static Pair<LocalDate, LocalDate> getDayPeriod(GdStatPreset preset, Instant currentInstant,
                                                          @Nullable Set<GdiRecommendationType> recommendationFilter) {
        LocalDate today = currentInstant.atZone(MSK).toLocalDate();
        switch (preset) {
            case TODAY:
                return Pair.of(today, today);
            case YESTERDAY:
                LocalDate yesterday = today.minusDays(1);
                return Pair.of(yesterday, yesterday);
            case RECOMMENDATIONS_PERIOD:
            case LAST_7DAYS:
                LocalDate from = today.minusDays(6);
                LocalDate to = today;
                if (preset == GdStatPreset.RECOMMENDATIONS_PERIOD && recommendationFilter != null) {
                    if (recommendationFilter.contains(GdiRecommendationType.switchOnAutotargeting)) {
                        from = from.minusDays(1);
                        to = to.minusDays(1);
                    } else if (recommendationFilter.contains(GdiRecommendationType.dailyBudget)
                            || recommendationFilter.contains(GdiRecommendationType.removePagesFromBlackListOfACampaign)) {
                        from = from.minusDays(1);
                        to = to.minusDays(1);
                    }
                }
                return Pair.of(from, to);
            case LAST_30DAYS:
                return Pair.of(today.minusDays(29), today);
            case LAST_90DAYS:
                return Pair.of(today.minusDays(89), today);
            case LAST_365DAYS:
                return Pair.of(today.minusDays(364), today);
            case CURRENT_WEEK:
            case PREVIOUS_WEEK:
                // Перемещаем текущую дату на понедельник
                LocalDate weekStart = today.minusDays(today.getDayOfWeek().getValue() - 1L);
                if (preset == GdStatPreset.PREVIOUS_WEEK) {
                    // Отнимаем ещё одну неделю для получения предыдущей
                    weekStart = weekStart.minusWeeks(1);
                }
                return Pair.of(weekStart, weekStart.plusDays(6));
            case CURRENT_MONTH:
            case PREVIOUS_MONTH:
                // Перемещаем текущую дату на первое число
                LocalDate monthStart = today.minusDays(today.getDayOfMonth() - 1L);
                if (preset == GdStatPreset.PREVIOUS_MONTH) {
                    // Отнимаем ещё месяц для получения предыдущего
                    monthStart = monthStart.minusMonths(1);
                }
                return Pair.of(monthStart, monthStart.plusMonths(1).minusDays(1));
            case LAST_6MONTHS:
                return Pair.of(today.minusMonths(6), today);
        }
        throw new IllegalArgumentException("Unsupported date preset: " + preset);
    }

    public static boolean applyEntityStatsFilter(@Nullable GdEntityStatsFilter statsFilter, GdEntityStats stats) {
        return STATS_FILTER_PROCESSOR.test(statsFilter, stats);
    }

    public static boolean applyGoalsStatFilters(@Nullable List<GdGoalStatsFilter> goalStatFilters,
                                                GdCampaign campaign) {
        return applyGoalsStatFilters(goalStatFilters, campaign.getGoalStats());
    }

    public static boolean applyGoalsStatFilters(@Nullable List<GdGoalStatsFilter> goalStatFilters,
                                                List<GdGoalStats> goalStats) {
        if (goalStatFilters != null) {
            for (GdGoalStatsFilter goalStatFilter : goalStatFilters) {
                GdGoalStats goalStat = goalStats.stream()
                        .filter(c -> c.getGoalId().equals(goalStatFilter.getGoalId()))
                        .findFirst()
                        .orElse(EMPTY_GOAL_STAT);

                if (!GOAL_STAT_FILTER_PROCESSOR.test(goalStatFilter, goalStat)) {
                    return false;
                }
            }
        }
        return true;
    }
}
