package ru.yandex.direct.grid.processing.util.findandreplace;

import java.util.Set;
import java.util.regex.Pattern;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.banner.type.href.BannersUrlHelper;
import ru.yandex.direct.grid.model.findandreplace.ChangeMode;
import ru.yandex.direct.grid.model.findandreplace.ReplaceRule;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceAdsHrefDomainInstruction;

import static ru.yandex.direct.utils.FunctionalUtils.listToSet;


/**
 * Заменяет домен в href на newDomain, если текущий домен содержится в searchDomains.
 * Игнорирует регистр домена при поиске
 */
@ParametersAreNonnullByDefault
public class BannerHrefDomainReplaceRule implements ReplaceRule {
    private static final Logger logger = LoggerFactory.getLogger(BannerHrefDomainReplaceRule.class);

    private final Set<String> searchDomains;
    private final String newDomain;
    private final ChangeMode changeMode;
    private final BannersUrlHelper bannersUrlHelper;

    public BannerHrefDomainReplaceRule(GdFindAndReplaceAdsHrefDomainInstruction replaceInstruction,
                                       BannersUrlHelper bannersUrlHelper) {
        this.newDomain = replaceInstruction.getReplace();
        this.searchDomains = listToSet(replaceInstruction.getSearch(), String::toLowerCase);
        this.bannersUrlHelper = bannersUrlHelper;
        this.changeMode = ChangeMode.REPLACE;
    }

    public BannerHrefDomainReplaceRule(@Nullable String searchDomain, @Nullable String newDomain, ChangeMode changeMode,
                                       BannersUrlHelper bannersUrlHelper) {
        this.searchDomains = searchDomain != null ? Set.of(searchDomain) : null;
        this.newDomain = newDomain == null ? "" : newDomain;
        this.changeMode = changeMode;
        this.bannersUrlHelper = bannersUrlHelper;
    }

    @Nullable
    @Override
    public String apply(String href) {
        if (href == null) {
            return null;
        }

        String currentDomain = bannersUrlHelper.extractHostFromHrefWithWwwOrNull(href);
        if (currentDomain != null && isReplaceableDomain(currentDomain)) {
            String newHref = replace(href, currentDomain, newDomain);
            logger.debug("Old href: {}, new href: {}", href, newHref);
            return newHref;
        }

        return href;
    }

    private String replace(String value, String currentDomain, String replacement) {
        switch (changeMode) {
            case PREFIX:
                return value.replaceFirst(Pattern.quote(currentDomain), replacement + currentDomain);
            case POSTFIX:
                return value.replaceFirst(Pattern.quote(currentDomain), currentDomain + replacement);
            case REPLACE:
            default:
                return value.replaceFirst(Pattern.quote(currentDomain), replacement);
        }
    }

    private boolean isReplaceableDomain(String domain) {
        return searchDomains == null || searchDomains.contains(domain.toLowerCase());
    }
}
