package ru.yandex.direct.grid.processing.util.findandreplace;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.Nullable;

import ru.yandex.direct.grid.model.findandreplace.ChangeMode;
import ru.yandex.direct.grid.model.findandreplace.ReplaceRule;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceAdsHrefParamsInstruction;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.validation.constraint.StringConstraints.DOMAIN_PATTERN;
import static ru.yandex.direct.validation.constraint.StringConstraints.PORT_PATTERN;
import static ru.yandex.direct.validation.constraint.StringConstraints.PROTOCOL_PATTERN;

public class BannerHrefParamsReplaceRule implements ReplaceRule {

    private static final Logger logger = LoggerFactory.getLogger(BannerHrefParamsReplaceRule.class);

    private static final String PATH_PATTERN = "([^?#]*)";

    private static final String QUERY_PATTERN = "([^#]*)";

    private static final String LAST_PATTERN = "(.*)";

    private static final Pattern HREF_PATTERN = Pattern.compile("^"
            + PROTOCOL_PATTERN   // protocol
            + DOMAIN_PATTERN // domain, group #1
            + PORT_PATTERN // port
            + PATH_PATTERN + "(\\?" + QUERY_PATTERN + ")?" + "(#" + LAST_PATTERN + ")?" // path and query parts
            + "\\z", Pattern.UNICODE_CHARACTER_CLASS);

    private final String search;
    private final String replace;
    private final ChangeMode changeMode;

    public BannerHrefParamsReplaceRule(GdFindAndReplaceAdsHrefParamsInstruction instruction) {
        search = instruction.getSearchKey();
        replace = instruction.getReplaceValue();
        this.changeMode = ChangeMode.REPLACE;
    }

    public BannerHrefParamsReplaceRule(
            @Nullable String search,
            @Nullable String replace,
            ChangeMode changeMode) {
        this.search = search;
        this.replace = replace;
        this.changeMode = changeMode;
    }

    @Override
    @Nullable
    public String apply(String hrefToChange) {
        if (hrefToChange == null) {
            return null;
        }
        Matcher matcher = HREF_PATTERN.matcher(hrefToChange);

        if (!matcher.matches()) {
            return hrefToChange;
        }

        if(search == null && replace == null) {
            String query = matcher.group(3);
            if(query != null) {
                return new StringBuilder(hrefToChange)
                        .replace(matcher.start(3), matcher.end(3), "")
                        .toString();
            } else {
                return hrefToChange;
            }
        }

        String query = matcher.group(4);

        String replacedQuery = replaceParams(query);
        int paramsStart = paramsStart(hrefToChange);
        return new StringBuilder(hrefToChange)
                .replace(query == null ? paramsStart : matcher.start(4),
                        query == null ? paramsStart : matcher.end(4), replacedQuery)
                .toString();
    }

    public String replaceParams(String value) {
        if (value == null) {
            return search != null ? null : replace;
        }
        String newQuery = replace(value, nvl(search, value), nvl(replace, ""));
        logger.debug("Old href: {}, new href: {}", value, newQuery);
        return newQuery;
    }

    private String replace(String value, String search, String replacement) {
        switch (changeMode) {
            case PREFIX:
                return value.replaceFirst(Pattern.quote(search), replacement + search);
            case POSTFIX:
                return value.replaceFirst(Pattern.quote(search), search + replacement);
            case REPLACE:
            default:
                return value.replaceFirst(Pattern.quote(search), replacement);
        }
    }

    private int paramsStart(String query) {
        Matcher matcher = HREF_PATTERN.matcher(query);
        if (!matcher.matches()) {
            return query.length();
        }
        for (int i = 3; i >= 0; i--) {
            String s = matcher.group(i);
            if (s != null) {
                return matcher.end(i);
            }
        }
        return query.length();
    }
}
