package ru.yandex.direct.grid.processing.util.findandreplace;

import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.Nullable;

import ru.yandex.direct.grid.model.findandreplace.ReplaceRule;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceHrefTextHrefPart;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdFindAndReplaceHrefTextInstruction;


public class BannerHrefTextReplaceRule implements ReplaceRule {
    private static final Pattern HREF_PATTERN = Pattern.compile("^([^?#]*)" + "(.*)\\z");
    private static final String REPLACE_ALL_PATTERN = "*";

    private final ReplaceParts replaceParts;
    private final ReplaceMode replaceMode;
    private final String search;
    private final String replace;

    public BannerHrefTextReplaceRule(GdFindAndReplaceHrefTextInstruction instruction) {
        Set<GdFindAndReplaceHrefTextHrefPart> hrefParts = instruction.getHrefParts();
        if (hrefParts.contains(GdFindAndReplaceHrefTextHrefPart.PROTOCOL_DOMAIN_PATH) &&
                hrefParts.contains(GdFindAndReplaceHrefTextHrefPart.QUERY_AND_FRAGMENT)) {
            replaceParts = ReplaceParts.FULL;
        } else if (hrefParts.contains(GdFindAndReplaceHrefTextHrefPart.PROTOCOL_DOMAIN_PATH)) {
            replaceParts = ReplaceParts.PROTOCOL_DOMAIN_PATH;
        } else if (hrefParts.contains(GdFindAndReplaceHrefTextHrefPart.QUERY_AND_FRAGMENT)) {
            replaceParts = ReplaceParts.QUERY_AND_FRAGMENT;
        } else {
            throw new UnsupportedOperationException("Unknown href parts for replace");
        }
        search = instruction.getSearch();
        if (search == null || search.equals(REPLACE_ALL_PATTERN)) {
            replaceMode = ReplaceMode.REPLACE_ALL;
        } else {
            replaceMode = ReplaceMode.FIND_AND_REPLACE_SUBSTRING;
        }

        replace = instruction.getReplace();
    }

    public BannerHrefTextReplaceRule(ReplaceParts replaceParts, @Nullable String search, @Nullable String replace) {
        this.replaceParts = replaceParts;
        this.search = search;
        if (search == null || search.equals(REPLACE_ALL_PATTERN)) {
            replaceMode = ReplaceMode.REPLACE_ALL;
        } else {
            replaceMode = ReplaceMode.FIND_AND_REPLACE_SUBSTRING;
        }
        this.replace = replace;
    }

    @Nullable
    @Override
    public String apply(String value) {
        if (replaceMode == ReplaceMode.FIND_AND_REPLACE_SUBSTRING) {
            switch (replaceParts) {
                case FULL:
                    return findAndReplaceInAllHrefApplyForSubstring(value);
                case PROTOCOL_DOMAIN_PATH:
                    return findAndReplaceInProtocolDomainPathApplyForSubstring(value);
                case QUERY_AND_FRAGMENT:
                    return findAndReplaceInQueryAndFragmentApplyForSubstring(value);
                default:
                    return null;
            }
        } else {
            switch (replaceParts) {
                case FULL:
                    return findAndReplaceInAllHrefApplyFull();
                case PROTOCOL_DOMAIN_PATH:
                    return findAndReplaceInProtocolDomainPathApplyFull(value);
                case QUERY_AND_FRAGMENT:
                    return findAndReplaceInQueryAndFragmentApplyFull(value);
                default:
                    return null;
            }
        }
    }

    @Nullable
    private String findAndReplaceInAllHrefApplyFull() {
        return replace;
    }

    @Nullable
    private String findAndReplaceInAllHrefApplyForSubstring(@Nullable String hrefToChange) {
        return hrefToChange == null ? null : hrefToChange.replace(search, blank(replace));
    }

    @Nullable
    private String findAndReplaceInProtocolDomainPathApplyFull(@Nullable String hrefToChange) {
        if (hrefToChange == null) {
            return null;
        }
        Matcher matcher = HREF_PATTERN.matcher(hrefToChange);
        if (!matcher.matches()) {
            return hrefToChange;
        }
        return blank(replace) + matcher.group(2);
    }

    @Nullable
    private String findAndReplaceInProtocolDomainPathApplyForSubstring(@Nullable String hrefToChange) {
        if (hrefToChange == null) {
            return null;
        }
        Matcher matcher = HREF_PATTERN.matcher(hrefToChange);
        if (!matcher.matches()) {
            return hrefToChange;
        }
        String textForFindAndReplace = matcher.group(1);
        if (!textForFindAndReplace.contains(search)) {
            return hrefToChange;
        }
        return textForFindAndReplace.replace(search, blank(replace)) + matcher.group(2);
    }

    @Nullable
    private String findAndReplaceInQueryAndFragmentApplyFull(@Nullable String hrefToChange) {
        if (hrefToChange == null) {
            return null;
        }
        Matcher matcher = HREF_PATTERN.matcher(hrefToChange);
        if (!matcher.matches()) {
            return hrefToChange;
        }
        return matcher.group(1) + blank(replace);
    }

    @Nullable
    private String findAndReplaceInQueryAndFragmentApplyForSubstring(@Nullable String hrefToChange) {
        if (hrefToChange == null) {
            return null;
        }
        Matcher matcher = HREF_PATTERN.matcher(hrefToChange);
        if (!matcher.matches()) {
            return hrefToChange;
        }
        String textForFindAndReplace = matcher.group(2);
        if (!textForFindAndReplace.contains(search)) {
            return hrefToChange;
        }
        return matcher.group(1) + textForFindAndReplace.replace(search, blank(replace));
    }

    /**
     * Какую часть ссылки меняем:
     * <ul>
     *     <li>{@code FULL} — меняем всю ссылку (полностью или по подстроке)
     *     <li>{@code PROTOCOL_DOMAIN_PATH} — меняем часть с протоколом, доменом и путём (до параметров)
     *     <li>{@code QUERY_AND_FRAGMENT} — параметры ссылки
     * </ul>
     */
    public enum ReplaceParts {
        PROTOCOL_DOMAIN_PATH,
        QUERY_AND_FRAGMENT,
        FULL
    }

    /**
     * Режим замены:
     * <ul>
     *     <li>{@code FIND_AND_REPLACE_SUBSTRING} — ищем подстроку по {@code search} и меняем только её
     *     <li>{@code REPLACE_ALL} — меняем полностью значение
     * </ul>
     */
    private enum ReplaceMode {
        FIND_AND_REPLACE_SUBSTRING,
        REPLACE_ALL,
    }

    private static String blank(@Nullable String str) {
        return str == null ? "" : str;
    }
}
