package ru.yandex.direct.grid.processing.util.findandreplace;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.grid.model.findandreplace.ChangeMode;
import ru.yandex.direct.grid.model.findandreplace.FindAndReplaceParams;
import ru.yandex.direct.grid.model.findandreplace.ReplaceRule;
import ru.yandex.direct.grid.model.findandreplace.SearchOptions;

import static java.lang.String.format;

@ParametersAreNonnullByDefault
public class PatternReplaceRule implements ReplaceRule {
    private static final String WHOLE_WORDS_PATTERN = "(\\b)(%s)(\\b)";
    private static final String SPECIAL_SYMBOLS = "[.$^+*?!()\\[\\]|\\\\]";
    private static final String ANY_WHITESPACE_INCLUDING_NBSP_REGEX = "[\\p{javaWhitespace}\u00A0\u2007\u202F]";
    private static final String ANY_WHITESPACE_INCLUDING_NBSP = "[\\\\p{javaWhitespace}\\\\u00A0\\\\u2007\\\\u202F]";

    private final Pattern searchPattern;
    private final String replacePattern;

    PatternReplaceRule(FindAndReplaceParams<?> findAndReplaceParams) {
        this.searchPattern = getPattern(findAndReplaceParams.getSearchText(), findAndReplaceParams.getSearchOptions());
        this.replacePattern =
                getReplacePattern(findAndReplaceParams.getChangeText(), findAndReplaceParams.getChangeMode(),
                        findAndReplaceParams.getSearchOptions());
    }

    @Override
    @Nullable
    public String apply(String value) {
        if (value == null) {
            return null;
        }
        Matcher matcher = searchPattern.matcher(value);
        if (matcher.find()) {
            return searchPattern.matcher(value).replaceAll(replacePattern);
        }
        return null;
    }

    private static Pattern getPattern(String searchText, SearchOptions searchOptions) {
        String pattern = escapeSpecialSymbols(searchText);
        pattern = matchAllSpaces(pattern);

        if (searchOptions.isOnlyWholeWords()) {
            pattern = format(WHOLE_WORDS_PATTERN, pattern);
        }

        if (searchOptions.isMatchCase()) {
            return Pattern.compile(pattern);
        } else {
            return Pattern.compile(pattern, Pattern.CASE_INSENSITIVE | Pattern.UNICODE_CASE);
        }
    }

    /**
     * Добавляет в паттерн поиска для замены правило, для учета всех пробельных символов
     */
    private static String matchAllSpaces(String pattern) {
        return pattern.replaceAll(ANY_WHITESPACE_INCLUDING_NBSP_REGEX, ANY_WHITESPACE_INCLUDING_NBSP);
    }

    static String escapeSpecialSymbols(String searchText) {
        return searchText.replaceAll(SPECIAL_SYMBOLS, "\\\\$0");
    }

    private static String getReplacePattern(String changeText, ChangeMode changeMode, SearchOptions searchOptions) {
        switch (changeMode) {
            case REPLACE:
                return searchOptions.isOnlyWholeWords() ? "$1" + changeText : changeText;
            case PREFIX:
                return searchOptions.isOnlyWholeWords() ? format("$1%s$2", changeText) : changeText + "$0";
            case POSTFIX:
                return "$0" + changeText;
            default:
                throw new IllegalStateException("Unknown replace mode");
        }
    }
}
