package ru.yandex.direct.configuration;

import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.List;

import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Lazy;

import ru.yandex.direct.common.configuration.TracingConfiguration;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.config.EssentialConfiguration;
import ru.yandex.direct.hourglass.HourglassProperties;
import ru.yandex.direct.hourglass.InstanceId;
import ru.yandex.direct.hourglass.InstanceMeta;
import ru.yandex.direct.hourglass.MonitoringWriter;
import ru.yandex.direct.hourglass.RandomChooser;
import ru.yandex.direct.hourglass.SchedulerInstancePinger;
import ru.yandex.direct.hourglass.SchedulerService;
import ru.yandex.direct.hourglass.TaskThreadPool;
import ru.yandex.direct.hourglass.implementations.InstanceIdImpl;
import ru.yandex.direct.hourglass.implementations.InstanceMetaImpl;
import ru.yandex.direct.hourglass.implementations.MD5Hash;
import ru.yandex.direct.hourglass.implementations.SchedulerInstancePingerImpl;
import ru.yandex.direct.hourglass.implementations.TaskThreadPoolImpl;
import ru.yandex.direct.hourglass.implementations.ThreadsHierarchy;
import ru.yandex.direct.hourglass.implementations.randomchoosers.RendezvousRandomChooser;
import ru.yandex.direct.hourglass.implementations.updateschedule.MainScheduleVersionExtractorImpl;
import ru.yandex.direct.hourglass.implementations.updateschedule.ScheduleUpdateServiceImpl;
import ru.yandex.direct.hourglass.storage.PrimaryId;
import ru.yandex.direct.hourglass.storage.Storage;
import ru.yandex.direct.hourglass.updateschedule.MainScheduleVersionExtractor;
import ru.yandex.direct.hourglass.updateschedule.ScheduleUpdateService;
import ru.yandex.direct.hourglass.updateschedule.SchedulerInstancesRepository;
import ru.yandex.direct.juggler.JugglerSender;
import ru.yandex.direct.liveresource.LiveResourceFactory;
import ru.yandex.direct.scheduler.JobInterceptorsList;
import ru.yandex.direct.scheduler.configuration.SchedulerConfiguration;
import ru.yandex.direct.scheduler.hourglass.ScheduleRecordListProvider;
import ru.yandex.direct.scheduler.hourglass.TaskFactory;
import ru.yandex.direct.scheduler.hourglass.TaskListProvider;
import ru.yandex.direct.scheduler.hourglass.implementations.JobScheduleInfoFactory;
import ru.yandex.direct.scheduler.hourglass.implementations.ScheduleInfoProcessor;
import ru.yandex.direct.scheduler.hourglass.implementations.ScheduleRecordListProviderImpl;
import ru.yandex.direct.scheduler.hourglass.implementations.SchedulerTracingWrapper;
import ru.yandex.direct.scheduler.hourglass.implementations.TaskFactoryImpl;
import ru.yandex.direct.scheduler.hourglass.implementations.TaskListProviderImpl;
import ru.yandex.direct.scheduler.hourglass.implementations.TaskParameterizer;
import ru.yandex.direct.scheduler.hourglass.implementations.monitoring.MonitoringWriterImpl;
import ru.yandex.direct.scheduler.hourglass.implementations.monitoring.SchedulerPingListener;
import ru.yandex.direct.scheduler.hourglass.implementations.monitoring.SchedulerPingMonitoring;
import ru.yandex.direct.scheduler.hourglass.implementations.monitoring.juggler.HourglassJugglerSender;
import ru.yandex.direct.scheduler.hourglass.implementations.schedule.ScheduleInfoConverterImpl;
import ru.yandex.direct.scheduler.hourglass.implementations.schedule.modifiers.NextRunModifierFactoryImpl;
import ru.yandex.direct.scheduler.hourglass.implementations.schedule.strategies.NextRunStrategiesFactoryImpl;
import ru.yandex.direct.scheduler.hourglass.schedule.ScheduleInfoConverter;
import ru.yandex.direct.scheduler.hourglass.schedule.modifiers.NextRunModifier;
import ru.yandex.direct.scheduler.hourglass.schedule.modifiers.NextRunModifierFactory;
import ru.yandex.direct.scheduler.hourglass.schedule.strategies.NextRunCalcStrategy;
import ru.yandex.direct.scheduler.hourglass.schedule.strategies.NextRunStrategiesFactory;
import ru.yandex.direct.scheduler.support.JobExecutingMonitor;
import ru.yandex.direct.tracing.TraceHelper;
import ru.yandex.direct.utils.SystemUtils;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static ru.yandex.direct.solomon.SolomonUtils.SOLOMON_REGISTRY;

/**
 * Общая часть для директо-специфичной конфигурации hourglass.
 * Снаружи (в том приложении, куда подключается) нужно определить бины:
 * <ul>
 *     <li>VERSION_BEAN_NAME, т.к. здесь задана пустая строка для получения доступов на чтение расписания</li>
 *     <li>INSTANCES_REPOSITORY_BEAN_NAME (можно получить приведение типа из HourglassYdbStorageConfiguration</li>
 *     <li>STORAGE_BEAN_NAME (аналогично)</li>
 *     <li>JobInterceptorsList</li>
 *     <li>JugglerSender</li>
 *     <li>TaskParameterizer</li>
 * </ul>
 * После чего в приложении можно будет создать собственно бин HourglassScheduler
 */
@Lazy
@Configuration
@Import({
        EssentialConfiguration.class,
        SchedulerConfiguration.class,
        TracingConfiguration.class,
})
public class DirectHourglassConfiguration {

    public static final String STORAGE_BEAN_NAME = "hourglassStorage";
    public static final String INSTANCES_REPOSITORY_BEAN_NAME = "schedulerInstancesRepository";
    public static final String VERSION_BEAN_NAME = "hourglassSchedulerVersionBeanName";
    public static final String SCHEDULER_CONFIG_BRANCH = "scheduler";

    @Bean
    public InstanceId instanceId(DirectConfig directConfig) {
        return directConfig.hasPath("hourglass.name") ? new InstanceIdImpl(directConfig.getString("hourglass.name"))
                : new InstanceIdImpl();
    }

    @Bean(VERSION_BEAN_NAME)
    public String version() {
        return "";
    }

    @Bean
    public HourglassJugglerSender hourglassJugglerSender(JugglerSender jugglerSender) {
        return new HourglassJugglerSender(jugglerSender);
    }

    @Bean
    public MainScheduleVersionExtractor mainScheduleVersionExtractor(
            @Value("${scheduler.version_file}") String mainVersionPath) {
        var liveResource = LiveResourceFactory.get(mainVersionPath);
        return new MainScheduleVersionExtractorImpl(liveResource::getContent);
    }

    @Bean
    public InstanceMeta instanceMeta() {
        var host = SystemUtils.shortCloudHostname();
        var startTime = LocalDateTime.now().toString();
        return new InstanceMetaImpl(host, startTime);
    }

    @Bean
    public HourglassProperties hourglassConfiguration(
            DirectConfig directConfig) {
        DirectConfig schedulerConfig = directConfig.getBranch(SCHEDULER_CONFIG_BRANCH);

        HourglassProperties.Builder builder = HourglassProperties.builder();

        schedulerConfig.findInt("max_threads").ifPresent(builder::setThreadPoolSize);

        schedulerConfig.findDuration("missing_tasks_threshold")
                .ifPresent(d -> builder.setMissedTaskThreshold(d.toMillis(), ChronoUnit.MILLIS));

        schedulerConfig.findInt("leader_voting_lower_bound")
                .ifPresent(builder::setLeaderVotingLowerBound);

        schedulerConfig.findDuration("fetch_interval")
                .ifPresent(d -> builder.setTaskFetchingInterval(d.toMillis(), ChronoUnit.MILLIS));

        schedulerConfig.findDuration("ping_interval")
                .ifPresent(d -> builder.setPingInterval(d.toMillis(), ChronoUnit.MILLIS));

        schedulerConfig.findDuration("expiration_time")
                .ifPresent(d -> builder.setMaxHeartbeatAge(d.toMillis(), ChronoUnit.MILLIS));

        schedulerConfig.findDuration("expiration_check_interval")
                .ifPresent(d -> builder.setJobsFreeingInterval(d.toMillis(), ChronoUnit.MILLIS));

        schedulerConfig.findDuration("instance_ping_interval")
                .ifPresent(d -> builder.setInstancePingInterval(d.toMillis(), ChronoUnit.MILLIS));

        schedulerConfig.findDuration("schedule_heartbeat_expiration_time")
                .ifPresent(d -> builder.setScheduleHeartbeatExpiration(d.toMillis(), ChronoUnit.MILLIS));

        schedulerConfig.findDuration("schedule_update_interval")
                .ifPresent(d -> builder.setScheduleUpdateInterval(d.toMillis(), ChronoUnit.MILLIS));

        return builder.build();
    }

    @Bean
    public SchedulerPingMonitoring schedulerPingMonitoring(List<SchedulerPingListener> schedulerPingListenerList) {
        return new SchedulerPingMonitoring(schedulerPingListenerList);
    }


    @Bean
    public MetricRegistry metricRegistry() {
        Labels solomonRegistryLabels = Labels.of("hourglass_scheduler", "HourglassApp");
        return SOLOMON_REGISTRY.subRegistry(solomonRegistryLabels);
    }

    @Bean
    public MonitoringWriter monitoringWriter(MetricRegistry metricRegistry,
                                             SchedulerPingMonitoring schedulerPingMonitoring) {
        return new MonitoringWriterImpl(metricRegistry, schedulerPingMonitoring);
    }

    @Bean
    public ThreadsHierarchy threadsHierarchy() {
        return new ThreadsHierarchy();
    }

    @Bean
    public TaskThreadPool taskThreadPool(ThreadsHierarchy threadsHierarchy, HourglassProperties hourglassProperties) {
        return new TaskThreadPoolImpl(hourglassProperties.getThreadPoolSize(), threadsHierarchy);
    }

    @Bean
    public RandomChooser<PrimaryId> randomChooser(
            @Qualifier(VERSION_BEAN_NAME) String version,
            @Qualifier(INSTANCES_REPOSITORY_BEAN_NAME) SchedulerInstancesRepository schedulerInstancesRepository,
            InstanceId instanceId
    ) {
        return new RendezvousRandomChooser<>(version, schedulerInstancesRepository, instanceId, new MD5Hash());
    }

    @Bean
    public SchedulerService schedulerService(@Qualifier(STORAGE_BEAN_NAME) Storage storage,
                                             InstanceId instanceId,
                                             RandomChooser<PrimaryId> randomChooser,
                                             ThreadsHierarchy threadsHierarchy, TaskThreadPool taskThreadPool,
                                             HourglassProperties hourglassProperties,
                                             SchedulerInstancePinger schedulerInstancePinger,
                                             TraceHelper traceHelper,
                                             MonitoringWriter monitoringWriter) {
        return new SchedulerTracingWrapper(storage,
                threadsHierarchy,
                taskThreadPool,
                instanceId,
                randomChooser,
                hourglassProperties,
                monitoringWriter,
                schedulerInstancePinger,
                traceHelper);
    }

    @Bean
    public JobScheduleInfoFactory taskScheduleGetter(ApplicationContext context) {
        return new JobScheduleInfoFactory(context);
    }

    @Bean
    public NextRunModifierFactory nextRunModifierFactory(List<NextRunModifier> modifiers) {
        return new NextRunModifierFactoryImpl(modifiers);
    }

    @Bean
    public TaskListProvider taskListProvider(ApplicationContext context) {
        return new TaskListProviderImpl(context);
    }

    @Bean
    public NextRunStrategiesFactory nextRunStrategiesFactory(List<NextRunCalcStrategy> nextRunCalcStrategies) {
        return new NextRunStrategiesFactoryImpl(nextRunCalcStrategies);
    }

    @Bean
    public ScheduleInfoProcessor scheduleInfoProcessor(NextRunStrategiesFactory nextRunStrategiesFactory,
                                                       NextRunModifierFactory nextRunModifierFactory) {
        return new ScheduleInfoProcessor(nextRunStrategiesFactory, nextRunModifierFactory);
    }

    @Bean
    public ScheduleInfoConverter scheduleInfoConverter() {
        return new ScheduleInfoConverterImpl();
    }

    @Bean
    public TaskFactory taskFactory(ApplicationContext context, TaskListProvider taskListProvider,
                                   InstanceId instanceId,
                                   DirectConfig config,
                                   JobInterceptorsList jobInterceptorsList, JobExecutingMonitor jobExecutingMonitor,
                                   ScheduleInfoConverter scheduleInfoConverter,
                                   ScheduleInfoProcessor scheduleInfoProcessor) {
        jobExecutingMonitor.setSchedulerName(instanceId.toString());
        jobExecutingMonitor.setInterval(config.getDuration("hourglass.execution_monitor_interval"));
        return new TaskFactoryImpl(context, taskListProvider, jobInterceptorsList, jobExecutingMonitor,
                scheduleInfoConverter, scheduleInfoProcessor);
    }

    @Bean
    public ScheduleRecordListProvider scheduleRecordListProvider(TaskListProvider taskListProvider,
                                                                 ScheduleInfoConverter scheduleInfoConverter,
                                                                 TaskParameterizer taskParameterizer,
                                                                 JobScheduleInfoFactory jobScheduleInfoFactory) {
        return new ScheduleRecordListProviderImpl(taskListProvider, scheduleInfoConverter, taskParameterizer,
                jobScheduleInfoFactory);
    }

    @Bean
    public SchedulerInstancePinger schedulerInstancePinger(
            InstanceId instanceId,
            InstanceMeta instanceMeta,
            @Qualifier(INSTANCES_REPOSITORY_BEAN_NAME) SchedulerInstancesRepository schedulerInstancesRepository,
            @Qualifier(VERSION_BEAN_NAME) String version,
            MonitoringWriter monitoringWriter
    ) {
        return new SchedulerInstancePingerImpl(instanceId, version, schedulerInstancesRepository, instanceMeta,
                monitoringWriter);
    }

    @Bean
    public ScheduleUpdateService scheduleUpdateService(
            MainScheduleVersionExtractor mainScheduleVersionExtractor,
            @Qualifier(INSTANCES_REPOSITORY_BEAN_NAME) SchedulerInstancesRepository schedulerInstancesRepository,
            @Qualifier(STORAGE_BEAN_NAME) Storage storage,
            ThreadsHierarchy threadsHierarchy,
            MonitoringWriter monitoringWriter,
            InstanceId instanceId,
            @Qualifier(VERSION_BEAN_NAME) String version,
            HourglassProperties hourglassProperties) {
        return new ScheduleUpdateServiceImpl.Builder()
                .withStorage(storage)
                .withMainVersionExtractor(mainScheduleVersionExtractor)
                .withSchedulerInstancesRepository(schedulerInstancesRepository)
                .withInstanceId(instanceId)
                .withCurrentVersion(version)
                .withThreadsHierarchy(threadsHierarchy)
                .withMonitoringWriter(monitoringWriter)
                .withFrequency(hourglassProperties.getScheduleUpdateInterval())
                .build();
    }
}
