package ru.yandex.direct.configuration;

import com.yandex.ydb.core.auth.AuthProvider;
import com.yandex.ydb.core.grpc.GrpcTransport;
import com.yandex.ydb.core.rpc.RpcTransport;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.rpc.grpc.GrpcTableRpc;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Lazy;

import ru.yandex.direct.common.configuration.TvmIntegrationConfiguration;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.ydb.YdbPath;
import ru.yandex.direct.ydb.client.YdbSessionProperties;
import ru.yandex.direct.ydb.tvm.YdbTvmAuthProvider;

import static ru.yandex.direct.common.configuration.TvmIntegrationConfiguration.TVM_INTEGRATION;


@Lazy
@Configuration
@Import(TvmIntegrationConfiguration.class)
public class YdbConfiguration {
    public static final String HOURGLASS_YDB_SESSION_PROPERTIES_BEAN = "HourglassYdbSessionProperties";
    public static final String HOURGLASS_YDB_PATH_BEAN = "HourglassYdbPath";
    public static final String HOURGLASS_YDB_RPC_TRANSPORT_BEAN = "HourglassRpcTransport";
    public static final String HOURGLASS_YDB_TABLE_CLIENT_BEAN = "HourglassTableClient";

    @Bean(HOURGLASS_YDB_PATH_BEAN)
    public YdbPath ydbPath(@Value("${ydb.cluster}") String cluster,
                           @Value("${ydb.hourglass.db_name}") String dbName) {
        return YdbPath.of(cluster, dbName);
    }

    @Bean
    public AuthProvider authProvider(@Qualifier(TVM_INTEGRATION) TvmIntegration tvmIntegration) {
        return new YdbTvmAuthProvider(tvmIntegration);
    }


    @Bean(HOURGLASS_YDB_RPC_TRANSPORT_BEAN)
    public RpcTransport grpcTransport(
            @Value("${ydb.endpoint}") String endpoint,
            @Qualifier(HOURGLASS_YDB_PATH_BEAN) YdbPath dbPath,
            AuthProvider authProvider
    ) {
        GrpcTransport.Builder transportBuilder = GrpcTransport.forEndpoint(endpoint, dbPath.getPath())
                .withAuthProvider(authProvider);
        return transportBuilder.build();
    }


    @Bean(HOURGLASS_YDB_SESSION_PROPERTIES_BEAN)
    public YdbSessionProperties ydbProperties(
            @Value("${ydb.hourglass.max_query_retries}") int maxQueryRetries,
            @Value("${ydb.hourglass.retry_not_found}") boolean retryNotFound
    ) {
        return YdbSessionProperties.builder()
                .withMaxQueryRetries(maxQueryRetries)
                .withRetryNotFound(retryNotFound)
                .build();
    }

    @Bean(HOURGLASS_YDB_TABLE_CLIENT_BEAN)
    public TableClient tableClient(@Qualifier(HOURGLASS_YDB_RPC_TRANSPORT_BEAN) RpcTransport rpcTransport,
                                   @Value("${ydb.hourglass.session_pool_min_size}") int sessionPoolMinSize,
                                   @Value("${ydb.hourglass.session_pool_max_size}") int sessionPoolMaxSize) {
        return TableClient.newClient(GrpcTableRpc.useTransport(rpcTransport))
                .sessionPoolSize(sessionPoolMinSize, sessionPoolMaxSize)
                .build();
    }
}
